package com.xdja.multichip.process.bluetooth;

import android.content.Context;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.blesafekey.BleCallback;
import com.xdja.blesafekey.BleErrID;
import com.xdja.blesafekey.BleStateDefine;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.param.ParamKeywords;
import com.xdja.multichip.process.SupperJniApiBinder;

import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import static com.xdja.multichip.process.bluetooth.JniApiBluetoothKey.RET_NOT_SUPPORT_METHOD;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/7/14 <br/>
 */

public class JniApiBluetoothKeyBinder extends SupperJniApiBinder {

    public static final int RET_EXCEPTION = -1;
    private String connBleDevAddr;
    private String connBleDevName;

    /**
     * proxy端的设置过来的SetCallBack
     */
    private ArrayList<Messenger> setCallBackList = new ArrayList<>();

    private JniApiBluetoothKeyBinder(Context context, String connBleDevAddr, String connBleDevName) {
        super(context);
        this.connBleDevAddr = connBleDevAddr;
        this.connBleDevName = connBleDevName;
    }

    /**
     * 创建实例。
     * 如果能够连接上传的address和name，就返回实例。
     * 否则不返回。
     *
     * @param context
     * @param connBleDevAddr
     * @param connBleDevName
     * @return
     */
    public static Pair<Integer, JniApiBluetoothKeyBinder> make(Context context, String connBleDevAddr, String connBleDevName) {
        JniApiBluetoothKeyBinder instance = new JniApiBluetoothKeyBinder(context, connBleDevAddr, connBleDevName);
        try {
            int state = instance.selfOpenDev(1000 * 30);
            if (state == BleStateDefine.BLE_STATE_SERVICE_DISCOVERED) {
                return Pair.create(0, instance);
            } else {
                return Pair.create(state, null);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Pair.create(RET_EXCEPTION, null);

    }

    JniApiBluetoothKey api = new JniApiBluetoothKey();

    @Override
    protected JNIAPI getApi() {
        return api;
    }

    /**
     * 用于打开设备
     */
    @Override
    public int selfOpenDev() throws RemoteException {
        return selfOpenDev(500);
    }

    private int selfOpenDev(int waitTime) {
        devInfo = null;
        final CountDownLatch latch = new CountDownLatch(1);
        api.getBlueApi().ConnectService(mContext, connBleDevAddr, connBleDevName);
        api.getBlueApi().SetCallBack(new BleCallback() {
            @Override
            public void OnBleStateChange(int newStatus) {
                if (newStatus == BleStateDefine.BLE_STATE_SERVICE_DISCOVERED) {
                    latch.countDown();
                }
                notifyCallBack(newStatus);
            }
        });
        try {
            //回调的等待时间先定为250ms
            latch.await(waitTime, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return api.getBlueApi().GetBleState();
    }

    /**
     * 得到芯片类型，参看：{@link JniApiParam#TYPE_ONBOARD}, {@link JniApiParam#TYPE_TF}, {@link JniApiParam#TYPE_BLUETOOTH}
     *
     * @return
     */
    @Override
    protected int getChipType() {
        return JniApiParam.TYPE_BLUETOOTH;
    }

    @Override
    public JniApiParam getJniApiParam() {
        JniApiParam param = super.getJniApiParam();
        if (param != null) {
            param.other = convertToKey(connBleDevAddr, connBleDevName);
        }
        return param;
    }

    /**
     * 内部方法:
     * 判断是否是因为未连接引起的错误
     *
     * @param errcode
     * @return
     */
    @Override
    protected boolean isDevOnline(int errcode) throws RemoteException {
        if (errcode == BleErrID.XKR_NO_KEY || errcode == BleErrID.XKR_NO_HANDLE) {
            selfOpenDev();
            if (api.getBlueApi().GetBleState() == BleStateDefine.BLE_STATE_CONNECTED) {
                return true;
            }
        }
        return false;
    }

    @Override
    public Bundle callMethod(Bundle bundle) throws RemoteException {
        if (bundle == null) {
            return null;
        }

        String method = bundle.getString(ParamKeywords.KEY_String_method);
        if ("GetBleState".equals(method)) {
            return methodGetBleState();
        } else if ("SetCallBack".equals(method)) {
            return methodSetCallBack(bundle);
        } else if ("UnlockPinUseUsn".equals(method)) {
            return super.callMethod(bundle);
        } else if ("GetBleBattery".equals(method)) {
            return methodGetBattery();
        }
        return super.callMethod(bundle);
    }

     
    private Bundle methodSetCallBack(Bundle bundle) {
        Bundle result = new Bundle();
        result.putInt("ret", 0);
        if (!bundle.containsKey("Messenger")) {
            result.putInt("ret", RET_EXCEPTION);
        } else {
            IBinder binder = bundle.getBinder("Messenger");
            if (binder == null) {
                result.putInt("ret", RET_EXCEPTION);
            } else {
                final Messenger messenger = new Messenger(binder);
                setCallBackList.add(messenger);
                try {
                    binder.linkToDeath(new DeathRecipient() {
                        @Override
                        public void binderDied() {
                            setCallBackList.remove(messenger);
                        }
                    }, 0);
                } catch (RemoteException e) {
                    e.printStackTrace();
                }
            }
        }
        return result;
    }

    
    private Bundle methodUnlockPinUseUsn() {
        Bundle result = new Bundle();
        result.putInt("ret", RET_NOT_SUPPORT_METHOD);
        return result;
    }

    
    private Bundle methodGetBleState() {
        Bundle result = new Bundle();
        int ret = api.getBlueApi().GetBleState();
        result.putInt("ret", ret);
        return result;
    }

    
    private Bundle methodGetBattery() {
        Bundle result = new Bundle();
        int ret = api.getBlueApi().GetBattery();
        result.putInt("ret", ret);
        return result;
    }


    /**
     * 转换成key
     *
     * @param devAddress
     * @param devName
     * @return
     */
    public static String convertToKey(String devAddress, String devName) {
        return devAddress + "#" + devName;
    }

    /**
     * 处理断开连接
     */
    public void handleDisconnect() {
        api.getBlueApi().DisConnectService();
    }

    private void notifyCallBack(int newStatus) {
        for (Messenger messenger : setCallBackList) {
            Message msg = new Message();
            msg.what = 1;
            msg.arg1 = newStatus;
            try {
                messenger.send(msg);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

}
