package com.xdja.multichip;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Binder;
import android.os.Bundle;
import android.os.Environment;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.elvishew.xlog.LogConfiguration;
import com.elvishew.xlog.LogLevel;
import com.elvishew.xlog.XLog;
import com.elvishew.xlog.printer.AndroidPrinter;
import com.elvishew.xlog.printer.file.FilePrinter;
import com.elvishew.xlog.printer.file.backup.FileSizeBackupStrategy;
import com.elvishew.xlog.printer.file.clean.FileLastModifiedCleanStrategy;
import com.elvishew.xlog.printer.file.naming.DateFileNameGenerator;
import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.datapersistence.DataPersistence;
import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.multichip.jniapi.JarMultiJniApiManager;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.utils.FuncUtils;
import com.xdja.multichip.utils.SharedPreferencesUtil;
import com.xdja.unlockcode.UnlockCode;

import java.io.File;
import java.security.InvalidParameterException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/6/21 <br/>
 * 此类主要用于提供统一PIN码相关的接口。
 */

public class MultiChipUnitePinBinder extends IMultiChipUnitePin.Stub {

    private static final String TAG = "MultiChipUnitePinBinder";

    private static final String ACTION_SAFEPIN_ADD = "com.xdja.safepin.action.SAFEPIN_ADD";
    private static final String ACTION_SAFEPIN_DEL = "com.xdja.safepin.action.SAFEPIN_DEL";
    private static final String ACTION_SAFEPIN_CHANGED = "com.xdja.safepin.action.SAFEPIN_CHANGED";

    private static final String KEY_INTERFACE_JNIAPI = "JNIAPI";
    private static final String KEY_INTERFACE_SKFAPI = "SkfApi";

    //    日志最长保存时间
    private final long MAX_TIME_LOG_DEL = 3 * 24 * 60 * 60 * 1000;
    //    日志文件最大size
    private final long MAX_LOG_FILE_SIZE = 1024 * 1024;
    private boolean log_file_create = false;
    private String xlogPath = "";


    private Map<String, Integer> cardPinCountMap = null;


    /**
     * -60002: 获取多芯片管理主进程Binder为空
     */
    private final static int MULTI_BINDER_ERROR = -60002;


    private static MultiChipUnitePinBinder instance;
    private Context context;

    private MultiChipUnitePinBinder(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }

        //add 2018年12月3日14:16:36 weizg xlog初始化
        try {
            initXlog();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void initXlog() {
        if (TextUtils.isEmpty(xlogPath)) {
            xlogPath = Environment.getExternalStorageDirectory().getPath() + "/xdja/unitpinlog/";
        }

        File logDirFile = new File(xlogPath);
        if (!logDirFile.exists()) {
            log_file_create = logDirFile.mkdirs();
        } else {
            log_file_create = true;
        }

        //如果日志目录创建创建成功
        if (log_file_create) {
            FilePrinter filePrinter = new FilePrinter.Builder(xlogPath)
                    .fileNameGenerator(new DateFileNameGenerator())
                    .backupStrategy(new FileSizeBackupStrategy(MAX_LOG_FILE_SIZE))
                    .cleanStrategy(new FileLastModifiedCleanStrategy(MAX_TIME_LOG_DEL))
                    .build();

            LogConfiguration logConfig = new LogConfiguration.Builder().
                    logLevel(LogLevel.ALL).
                    tag(TAG).
                    build();
            XLog.init(logConfig, new AndroidPrinter(), filePrinter);
        }
    }

    public static MultiChipUnitePinBinder getInstance(Context context) {
        if (context == null) {
            throw new InvalidParameterException("context is null");
        }
        if (instance == null) {
            synchronized (MultiChipUnitePinBinder.class) {
                if (instance == null) {
                    instance = new MultiChipUnitePinBinder(context);
                }
            }
        }
        return instance;
    }

    //add 2018年12月3日14:37:13 weizg
    //重新日志输出，如果xlog框架初始化成功则写入文件，否则还是写入logcat
    private void logMsg(String tag, String msg) {
        if (log_file_create) {
            //如果log文件目录创建成功
            XLog.d(msg);
        } else {
            Log.d(tag, msg);
        }
    }


    /**
     * 处理PIN码验证结果.
     *
     * @param cardId        卡号
     * @param role          角色
     * @param pin           正确的PIN码
     * @param verifyResult  验证的结果。比如：信大捷安的卡，验证正确了，结果为0，验证失败为1到9，-10或-16表示锁死
     * @param method        方法，比如：VerifyPIN,ChangePIN,ReloadPIN等
     * @param callingUid    调用method者的uid
     * @param callingPid    调用method者的pid
     * @param fromInterface 哪种类型的接口调用，比如：JNIAPI，SkfApi(国密接口)
     * @return
     */
    @Override
    public int handlePinResult(String cardId,
                               int role,
                               byte[] pin,
                               int verifyResult,
                               String method,
                               int callingUid,
                               int callingPid,
                               String fromInterface) {
        logMsg(TAG, "getPin,handlePinResult ret: " + verifyResult);
        if (isEmpty(cardId, method, fromInterface)) {
            logMsg(TAG, "getPin,handlePinResult data null ");
            return 0;
        }
        //判断调用者是否为自己
//        int uid = Binder.getCallingUid();
//        if (uid != Process.myUid()) {
//            return 0;
//        }

        // 3.1 验证通过
        cardId = cardId.trim();
        if (verifyResult == 0) {
            if (pin == null || pin.length == 0) {
                return 0;
            }
            savePinToFile(cardId, role, pin);
            savePinTryCount(cardId, role);
        } else {
            if (KEY_INTERFACE_JNIAPI.equalsIgnoreCase(fromInterface)) {
                if (verifyResult == JNIAPI.XKR_PASSWORD
                        || verifyResult == JNIAPI.XKR_KEY_LOCKED
                        || verifyResult == JNIAPI.XKR_KEYFILE_NOT_EXIST
                        || verifyResult == JNIAPI.XKR_KEY_NOT_EXIST) {
                    delPinFromFile(cardId, role);
                } else if (verifyResult > 0 || verifyResult < getPinTryMax(cardId, role)) {
                    delPinFromFile(cardId, role);
                }
            }
        }
        // TODO: 2017/6/22 增加验证失败的时候日志记录
        return 0;
    }

    /**
     * 保存PIN码最大重试次数
     *
     * @param cardId
     * @param role
     */
    private void savePinTryCount(String cardId, int role) {
        JarJniApiProxy proxy = getJarJniApiProxy(cardId);
        if (proxy == null) {
            return;
        }
        int ret = proxy.GetPinTryCount(role);
        if (ret > 0) {
            SharedPreferencesUtil.setPinTryCount(context, cardId, role, ret);
        }

        logMsg(TAG, "savePin,getTryCount = " + ret);
    }


    /**
     * 得到PIN码。
     *
     * @param cardId 卡号
     * @param role   角色
     * @return Bundle中有两个值:
     * bundle.getInt("ret") 得到的是返回值，如果为0，则表示获取PIN码成功，其他表示失败
     * bundle.getString("pin") 得到的是PIN码，只有得到ret为0时，此项才会有值。
     */
    @Override
    public Bundle getPin(String cardId, int role) throws RemoteException {
        int callingUid = Binder.getCallingUid();
        int callingPid = Binder.getCallingPid();
//        int ret = AccessControlManager.getInstance(context).checkByUidPid(callingUid, callingPid, "getPin");
        //add 为了政务微信能够调用到统一PIN码
        int ret = checkCallApp(context,callingUid,callingPid);
        logMsg(TAG, "getPin,checkByUidPid =" + ret);

        Bundle bundle = new Bundle();
        if (ret != 0) {
            bundle.putInt("ret", ret);
            return bundle;
        }
        synchronized (MultiChipUnitePinBinder.class) {
            try {
                //add by zhangxiaolong 2019-1-16 如果是虚拟卡，先直接返回；
                //如果还按之前的逻辑，走不通，因为这时候有可能是虚拟卡创建时getPin
                int cardType = SharedPreferencesUtil.getCardType(context, cardId);
                if (cardType == JniApiParam.TYPE_VHSM || judgeIsVhsmCardId(cardId)) {
                    Pair<String, byte[]> pinCrash = getCachePin(cardId, role);
                    if (pinCrash.second == null) {
                        //暂时兼容以前的jar包
                        bundle.putInt("ret", DataPersistence.ERROR);
                        logMsg(TAG, "getPin,getData failed");
                        return bundle;
                    } else {//有PIN码
                        bundle.putInt("ret", DataPersistence.RET_OK);
                        bundle.putString("pin", new String(pinCrash.second));
                        return bundle;
                    }
                }
                //2.没有锁死的情况下，判断当前PIN码是否重试测试是否为最大重试次数
                // 此步骤中即使锁死 最大重试次数与当前剩余次数也不相等，故省去1.获取卡内剩余重试次数
                Pair<Integer, Boolean> pair = judgePinEnable(context, cardId, role);
                logMsg(TAG, "getPin,judgePinEnable =" + pair.second);

                //获取本地缓存
                Pair<String, byte[]> pinCrash = getCachePin(cardId, role);

                //2.1当前PIN码不是最大重试次数,应判断本地是否有PIN码，有的话删除并发送广播
                if (pair.first != 0 || !Objects.equals(pair.second, Boolean.TRUE)) {
                    boolean notOveredCard = true;
                    if (pair.first != 0) {
                        int type = SharedPreferencesUtil.getCardType(context, cardId);
                        //add by zhangxiaolong 用于解决贴膜卡有时获取PIN码失败的现实情况
                        //如果是贴膜卡，并且获取贴膜卡重试次数失败了，则继续下面的步骤
                        if (type == JniApiParam.TYPE_COVERED) {
                            notOveredCard = false;
                        }
                    }
                    if (notOveredCard) {
                        bundle.putInt("ret", DataPersistence.ERROR);
                        // 3.1 有PIN码,删除PIN码，并发送广播
                        if (pinCrash.second != null) {
                            delPinFromFile(pinCrash.first);
                            safePinDelBroadcast(role);
                        }
                        return bundle;
                    }
                }
                //2.2当前PIN码是最大重试次数情况下，判断本地是否有PIN码，有的话返回PIN码，没有的话返回错误码
                //没有PIN码
                if (pinCrash.second == null) {
                    //暂时兼容以前的jar包
                    bundle.putInt("ret", DataPersistence.ERROR);
                    logMsg(TAG, "getPin,getData failed");
                    return bundle;
                } else {//有PIN码
                    bundle.putInt("ret", DataPersistence.RET_OK);
                    bundle.putString("pin", new String(pinCrash.second));
                    return bundle;
                }
            } catch (Exception e) {
                e.printStackTrace();
                bundle.putInt("ret", DataPersistence.ERROR);
            }
            return bundle;
        }
    }

    //add 2019年5月6日17:43:40 weizg
    //政务微信集成密钥SDK，当使用贴膜卡v1版本时需要从芯片管家获取PIN码；修改PIN码获取校验规则，
    //允许政务微信获取；其他类型芯片密钥SDK通过使用0x04角色来使用芯片
    private int checkCallApp(Context context, int callingUid, int callingPid){
        int ret = AccessControlManager.getInstance(context).
                checkByUidPid(callingUid, callingPid, "getPin");
        if(ret != 0){
            //如果so库未校验通过，则通过java代码验证
            ret = checkCallAppByPkgName(context, callingUid, callingPid);
        }

        return ret;
    }

    private int checkCallAppByPkgName(Context context, int callingUid, int callingPid){

        logMsg(TAG, "checkCallAppByPkgName ");
        if (context == null || callingUid < 0 || callingPid < 0) {
            return -1;
        }

        //政务微信包名
        String pkg_white = "com.tencent.weworklocal";

        //政务微信的包指纹
        String pkg_white_zwwx ="B2:E0:B6:4D:75:36:E4:AF:83:63:B4:02:2A:9F:74:72:D5:80:FA:0B";

        String pkgName = "";
        PackageManager pm = context.getPackageManager();
        String[] pkgArray = pm.getPackagesForUid(callingUid);

        if (pkgArray == null || pkgArray.length <= 0) {
            logMsg(TAG, "checkCallAppByPkgName ret -1");
            return -1;
        } else {

            int ret = -1;
            for (String pkg : pkgArray) {
                if(pkg_white.equals(pkg)){
                    ret = 0;
                    break;
                }
            }

            //如果包名是在白名单，则再检查包指纹
            if(ret == 0){
                String certFingerStr = FuncUtils.getCertFingerStr(context, pkg_white);
                //如果包指纹校验不通过
                if(!pkg_white_zwwx.equalsIgnoreCase(certFingerStr)){
                    ret = -2;
                }
            }

            logMsg(TAG, "checkCallAppByPkgName ret " + ret);

            return ret;
        }
    }

    /**
     * <p>删除存储的PIN
     * <p>内部流程：
     * <p>1 查看本地是否存有PIN码
     * <p>1.1 有PIN码，则删除，并发PIN码删除广播
     * <p>1.2 没有PIN码，什么都不做
     *
     * @param role   角色
     * @param cardId 卡号
     * @return first:是否成功，0表示成功。
     * second：卡ID
     */
    private int delPinFromFile(String cardId, int role) {
        int ret = 0;
        Pair<String, byte[]> pair = getCachePin(cardId, role);
        if (pair != null && !TextUtils.isEmpty(pair.first)) {
            if (pair.second != null) {
                ret = delPinFromFile(pair.first);
                safePinDelBroadcast(role);
            }
        }
        //for test
        logMsg(TAG, "delPinFromFile ret = " + ret);
        return ret;
    }

    /**
     * 保存PIN码
     * <p>流程如下：
     * <p>1.获取本地是否保存PIN码，
     * <p>1.1 如果没有保存PIN码，保存PIN码后，发送PIN码从无到有的广播
     * <p>1.2 如果保存的有PIN码，再次保存PIN码，确保PIN码为最新的
     * <p>1.3 如果保存的有PIN码，比较现在的PIN码跟存储的PIN码是否一致,如果不一致则发送PIN码改变广播
     *
     * @param role   角色
     * @param cardId 卡号
     * @return 0:成功；其他：失败
     */
    private int savePinToFile(String cardId, int role, byte[] pin) {
        int ret = 0;
        Pair<String, byte[]> pair = getCachePin(cardId, role);
        if (pair != null && !TextUtils.isEmpty(pair.first)) {
            ret = savePinToFile(pair.first, pin);
            if (pair.second == null) {
                safePinAddBroadcast(role);
            } else if (!Arrays.equals(pair.second, pin)) {
                safePinChangedBroadcast(role);
            }
        }

//      add 2018年11月29日10:46:12 weizg
        logMsg(TAG, "save pin ret: " + ret);

        return ret;
    }


    /**
     * 发送PIN码添加广播
     *
     * @param
     */
    private void safePinAddBroadcast(int role) {
        sendBroadcast(role, ACTION_SAFEPIN_ADD);
    }

    /**
     * 发送PIN码删除广播
     *
     * @param
     */
    private void safePinDelBroadcast(int role) {
        sendBroadcast(role, ACTION_SAFEPIN_DEL);
    }

    /**
     * 发送PIN码已修改广播
     *
     * @param
     */
    private void safePinChangedBroadcast(int role) {
        sendBroadcast(role, ACTION_SAFEPIN_CHANGED);
    }

    /**
     * 内部使用方法
     * <p> 发送广播
     *
     * @param role   角色
     * @param action action
     */
    private void sendBroadcast(int role, String action) {
        Intent intent = new Intent();
        intent.putExtra("role", role);
        intent.setAction(action);
        logMsg(TAG, "sendBroadcast: " + action + " role: " + role);
        context.sendBroadcast(intent);
    }

    /**
     * 内部使用方法
     * <p>获取存储的PIN码
     * <p>first为key，second为pin
     *
     * @param cardId 卡号
     * @param role   角色
     * @return first为key，second为pin
     */
    private Pair<String, byte[]> getCachePin(String cardId, int role) {
        //byte[] cardId = new byte[32];
        //int ret = SafeCardUtils.getCardId(cardId);
        //if (ret == 0) {
        String key = getPinKey(role, cardId.getBytes());
        if (key == null) {
            return Pair.create(null, null);
        }

        byte[] pin = getPinFromFile(key);
        return Pair.create(key, pin);
        //}
    }


    /**
     * 内部使用方法
     * <p>获取PIN码
     *
     * @param key key
     * @return 0:成功
     */
    private byte[] getPinFromFile(String key) {
        return DataPersistence.getInstance(context).getData(key);
    }

    /**
     * 内部使用方法
     * <p>保存PIN码
     *
     * @param key key
     * @param pin pin
     * @return 0:成功
     */
    private int savePinToFile(String key, byte[] pin) {
        return DataPersistence.getInstance(context).addData(key, pin);
    }

    /**
     * 内部使用方法<p>
     * 删除PIN码
     *
     * @param key
     * @return
     */
    private int delPinFromFile(String key) {
        return DataPersistence.getInstance(context).deleteData(key);
    }

    /**
     * 内部使用接口
     * <p>得到存储PIN时用的KEY
     *
     * @param role   role
     * @param cardId cardId
     * @return 当cardId为null或size为0时，返回null
     */
    private String getPinKey(int role, byte[] cardId) {
        if (cardId == null || cardId.length == 0) {
            return null;
        }
        String result = new String(cardId);
        result = result + "#" + role;
        return result;
    }

    private boolean isEmpty(Object... objects) {
        Object object = null;
        for (Object object1 : objects) {
            object = object1;
            if (object == null) {
                return true;
            } else if (object instanceof byte[] && ((byte[]) object).length == 0) {
                return true;
            } else if (object instanceof String && ((String) object).length() == 0) {
                return true;
            } else if (object instanceof List && ((List) object).size() == 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * -20003：参数错误
     */
    public final static int ERROR_PARAM = -20003;

    //add 2017年7月17日10:33:50 weizg 获取解锁/重置usn接口

    /**
     * 获取解锁/重置PIN码所需的USN
     *
     * @param cardId 解锁/重置的芯片卡号
     * @return Bundle中有两个值:
     * bundle.getInt("ret") 得到的是返回值，如果为0，则表示获取usn成功，其他表示失败
     * bundle.getString("usn") 得到的是usn，只有得到ret为0时，此项才会有值。
     */
    @Override
    public Bundle getUsn(String cardId) {
        Bundle retBundle = new Bundle();
        retBundle.putInt("ret", ERROR_PARAM);

        if (TextUtils.isEmpty(cardId)) {
            return retBundle;
        }

        synchronized (MultiChipUnitePinBinder.class) {
            try {
                int ret;
                byte[] usn = new byte[14];
                byte[] uid = new byte[10];

                UnlockCode unlockCode = new UnlockCode();
                logMsg(TAG, "card id : " + cardId.toLowerCase());

                ret = unlockCode.CID2UID(cardId.toLowerCase().getBytes(), uid);
                if (ret != 0) {
                    logMsg(TAG, "CID2UID failed");
                    retBundle.putInt("ret", ret);
                } else {
                    logMsg(TAG, "uid: " + new String(uid));
                    ret = unlockCode.UID2USN(uid, usn);
                    logMsg(TAG, "usn: " + new String(usn));

                    retBundle.putString("usn", new String(usn));
                    retBundle.putInt("ret", ret);
                }

            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return retBundle;
    }

    /**
     * 获取做大PIN码重试次数
     *
     * @param cardId
     * @param role
     * @return
     */
    private int getPinTryMax(String cardId, int role) {

        //1.首先从本地获取最大重试次数
        int count = SharedPreferencesUtil.getPinTryCount(context, cardId, role);
        if (count > 0) {
            return count;
        }
        //2.从本地获取失败，从配置文件中获取
        JarJniApiProxy proxy = getJarJniApiProxy(cardId);
        if (proxy == null) {
            return 0;
        }
        XDJA_DEVINFO devInfo = new XDJA_DEVINFO();
        int ret = proxy.GetDevInfo(devInfo);
        if (ret != 0) {
            return 0;
        }

        int cardtype = devInfo.cardtype;

        return getPinTryCountByIdAndRole(role, cardtype);
    }

    /**
     * 根据卡号获取对应代理对象
     *
     * @param cardId
     * @return
     */
    private JarJniApiProxy getJarJniApiProxy(String cardId) {
        Pair<Integer, JarJniApiProxy> makrPair = JarMultiJniApiManager.getInstance().make(this.context, cardId);
        if (makrPair.first != 0 || makrPair.second == null) {
            return null;
        }
        JarJniApiProxy proxy = makrPair.second;
        return proxy;
    }

    /**
     * 判断PIN码是否重试测试是否为最大重试次数
     * <p>
     * PIN码剩余次数 和 角色，作为是否返回PIN码的条件<p>
     * 当返回为false时，不返回PIN码<p>
     * 当返回为true时，接着流程判断.<p>
     * <p>
     * 当PIN码剩余次数等于10，返回true<p>
     * 当PIN码剩余次数等于5，同时角色为0x01时，返回true<p>
     * 其他返回false<p>
     *
     * @param context
     * @param cardId  卡ID
     * @param role    角色
     * @return Pair.first错误码，为0是second有效
     * Pair.second为当前PIN码是否重试测试是否为最大重试次数
     */
    private Pair<Integer, Boolean> judgePinEnable(Context context, String cardId, int role) {
        Pair<Integer, JarJniApiProxy> makePair = JarMultiJniApiManager.getInstance().make(context, cardId);

        if (makePair == null || makePair.first != 0) {
            logMsg(TAG, "getPin,pinTryCount ret = " + (makePair == null ? -6666 : makePair.first));
            return Pair.create(-1, false);
        }

        int pinTryCount = makePair.second.GetPinTryCount(role);

        //add 2018年7月25日09:49:20 weizg
        //细化GetPinCount接口返回：区分操作失败和PIN码次数获取成功
        if (pinTryCount < 0 && pinTryCount != -10 && pinTryCount != -16) {

            logMsg(TAG, "getPin,pinTryCount ret =" + pinTryCount);

            //如果操作失败，则直接返回
            return Pair.create(-1, false);
        }

        int maxCount = getPinTryMax(cardId, role);

        logMsg(TAG, "getPin,maxCount = " + maxCount);
        logMsg(TAG, "getPin,pinTryCount = " + pinTryCount);

        if (pinTryCount == maxCount) {
            return Pair.create(0, true);
        }
        return Pair.create(0, false);
    }

    /**
     * 根据芯片类型和角色获取最大重试次数
     *
     * @param role
     * @param cardType
     * @return
     */
    private int getPinTryCountByIdAndRole(int role, int cardType) {
        if (cardPinCountMap == null) {
            initCardPinCountMap();
        }
        if (cardPinCountMap.containsKey("" + cardType + role)) {
            return cardPinCountMap.get("" + cardType + role);
        }
        return 0;
    }

    /**
     * 初始化芯片类型最大重试次数map
     * 芯片，TF卡，蓝牙key:
     * 0x11  10次
     * 0x01  5次
     * 贴膜卡 ：
     * 0x01  6次
     */
    private void initCardPinCountMap() {
/*        //板载芯片类型
        int CT_TF_XDJA_CHIP = 0x0213;
        int CT_XDJA_SPI = 0x0217;
        //TF卡
        int CT_TF = 0x0200;
        //贴膜卡
        int CT_TMC = 0x0600;
        //蓝牙key
        int CT_BLE = 0x0700;*/

        int role_0x01 = 0x01;
        int role_0x11 = 0x11;

        if (cardPinCountMap == null) {
            cardPinCountMap = new HashMap<String, Integer>();
            cardPinCountMap.put("" + JNIAPI.CT_XDJA_SPI + role_0x01, 5);
            cardPinCountMap.put("" + JNIAPI.CT_XDJA_SPI + role_0x11, 10);

            cardPinCountMap.put("" + JNIAPI.CT_TF_XDJA_CHIP + role_0x01, 5);
            cardPinCountMap.put("" + JNIAPI.CT_TF_XDJA_CHIP + role_0x11, 10);

            cardPinCountMap.put("" + JNIAPI.CT_TF + role_0x01, 5);
            cardPinCountMap.put("" + JNIAPI.CT_TF + role_0x11, 10);

            cardPinCountMap.put("" + JNIAPI.CT_TF_XDJA + role_0x01, 5);
            cardPinCountMap.put("" + JNIAPI.CT_TF_XDJA + role_0x11, 10);

            cardPinCountMap.put("" + JNIAPI.CT_TMC + role_0x01, 6);
            //add 2018年11月30日09:22:39 weizg
            cardPinCountMap.put("" + JNIAPI.CT_TMC + role_0x11, 6);

            cardPinCountMap.put("" + JNIAPI.CT_BLE + role_0x01, 5);
            cardPinCountMap.put("" + JNIAPI.CT_BLE + role_0x11, 10);

        }
    }

    /**
     * 判断是是不是单机版虚拟卡卡号
     * @param cardId
     * @return
     */
    private boolean judgeIsVhsmCardId(String cardId){
        byte[] data = "xdjavhsm".getBytes();
        char[] arr = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
        StringBuffer sb = new StringBuffer(data.length * 2);
        for (int i = 0; i < data.length; ++i) {
            sb.append(arr[((data[i] & 0xF0) >> 4)]);
            sb.append(arr[(data[i] & 0xF)]);
        }
        String id = sb.toString();
        if (cardId.startsWith(id)){
            return true;
        }
        return false;
    }
}
