package com.xdja.multichip.utils;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.text.TextUtils;

import java.io.ByteArrayInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

/**
 * <br>author: wzg@xdja.com <br/>
 * date:   2018/7/24 11:10 <br/>
 */
public class FuncUtils {

    public static final String PKG_SAFE_KEYSERVICE = "com.xdja.safekeyservice";
    public static final String PKG_SAFE_CLIENT = "com.xdja.safeclient";

    public static boolean isApkExist(Context context, String pkgName){

        if(TextUtils.isEmpty(pkgName) || context == null){
            return false;
        }

        PackageManager pm = context.getPackageManager();
        try {
            PackageInfo packageInfo = pm.getPackageInfo(pkgName, PackageManager.GET_META_DATA);

            return packageInfo != null;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }

        return false;

    }

    /**
     * 根据pkgName 得到证书指纹
     *
     * @param context context
     * @param pkgName pkgName
     * @return result
     */
    public static String getCertFingerStr(Context context, String pkgName) {
        try {
            X509Certificate cert = getSignCer(context, pkgName);
            return format(parseCerFingerprint(cert));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取指定包名程序的签名信息
     *
     * @param context
     * @param packName
     */
    private static X509Certificate getSignCer(Context context, String packName) {
        try {
            // 获取前面信息
            PackageInfo packageInfo = context.getPackageManager().getPackageInfo(packName, PackageManager.GET_SIGNATURES);
            Signature[] signs = packageInfo.signatures;
            Signature signature = signs[0];

            // 获取证书Cer
            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");

            return (X509Certificate) certFactory.generateCertificate(new ByteArrayInputStream(signature.toByteArray()));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * X509Certificate 获取证书的指纹信息SHA1字符串
     *
     * @param cert
     * @return
     */
    private static String parseCerFingerprint(X509Certificate cert) {
        try {
            byte[] arrayOfByte = convert1(cert.getEncoded());
            if(arrayOfByte == null){
                return null;
            }
            return convert2(arrayOfByte);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 对字节数组进行SHA1加密
     *
     * @param data
     * @return
     */
    private static byte[] convert1(byte[] data) {
        try {
            MessageDigest localMessageDigest = MessageDigest.getInstance("SHA1");
            return localMessageDigest.digest(data);
        } catch (NoSuchAlgorithmException localNoSuchAlgorithmException) {
            localNoSuchAlgorithmException.printStackTrace();
        }
        return new byte[0];
    }

    /**
     * 将字节数组转换格式化为证书指纹SHA1格式
     *
     * @author 单红宇
     */
    private static String convert2(byte[] data) {
        char[] arr = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
        StringBuilder sb = new StringBuilder(data.length * 2);
        for (int i = 0; i < data.length; ++i) {
            sb.append(arr[((data[i] & 0xF0) >> 4)]);
            sb.append(arr[(data[i] & 0xF)]);
        }
        return sb.toString();
    }

    /**
     * 对证书指纹SHA1进行格式化（使用“:”分割）
     *
     * @param str
     * @return
     */
    private static String format(String str) {
        StringBuilder localStringBuffer = new StringBuilder();
        for (int i = 0; i < str.length(); ++i) {
            localStringBuffer.append(str.charAt(i));
            if ((i <= 0) || (i % 2 != 1) || (i >= str.length() - 1)) {
                continue;
            }
            localStringBuffer.append(":");
        }
        return localStringBuffer.toString();
    }

    /**
     * 判断字符串是否只含有数字和字母
     *
     * @param str
     * @return
     */
    public static boolean isLetterDigit(String str) {
        String regex = "^[a-z0-9A-Z]+$";
        return str.matches(regex);
    }
}
