package com.xdja.uniteauth;

import android.content.ContentResolver;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.AssetFileDescriptor;
import android.net.Uri;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.Gson;
import com.xdja.crypto.XDJACrypto;
import com.xdja.uniteauth.bean.ClientToken;
import com.xdja.uniteauth.data.ApiGen;
import com.xdja.uniteauth.data.net.model.CheckPkgRequest;
import com.xdja.uniteauth.data.net.model.ErrInfoBean;
import com.xdja.uniteauth.data.net.model.UASResponseBean;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.utils.AssetsUtil;
import com.xdja.uniteauth.utils.GetValueFromProperties;
import com.xdja.uniteauth.utils.PkgInfoUtil;
import com.xdja.uniteauth.utils.VerifyTokenUtil;
import com.xdja.xdjacrypto.XCT_CERT_INFO;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import retrofit.Callback;
import retrofit.RetrofitError;
import retrofit.client.Response;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2018/8/2 <br/>
 * 处理得到认证码的线程
 */
public class VerifyPkgThread implements Runnable {
    private static final String TAG = "VerifyPkgThreadTag";

    private Context context;
    private int callingUid;
    private int callingPid;
    private String pkgName;
    private Bundle bundle;
    private IThreadCallback callback;

    /**
     * 验证成功的pkg和证书指纹
     * key:为包名+uid；
     * value:为证书指纹；
     */
    private static HashMap<String, String> verifySuccessMap = new HashMap<>();

    /**
     * 验证成功的pkg和appId；
     * key：为包名+uid；
     * value：为appId
     */
    private static HashMap<String, String> verifySuccessAppIdMap = new HashMap<>();

    /**
     * 验签失败的 uid#pid 和 对应的错误码
     * key:为 uid#pid
     * value: 为错误码
     */
    private static HashMap<String, Integer> verifyFailMap = new HashMap<>();

    public VerifyPkgThread(Context context,
                           int callingUid,
                           int callingPid,
                           String pkgName,
                           Bundle bundle,
                           IThreadCallback callback) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        this.callingUid = callingUid;
        this.callingPid = callingPid;
        this.pkgName = pkgName;
        this.bundle = bundle;
        this.callback = callback;
    }

    @Override
    public void run() {
        //如果配置文件中配置是不验证签名，直接通过
        if (!isVerifyByConfig()) {
            String appId = verifySuccessAppIdMap.get(getSuccessMapKey());
            if (TextUtils.isEmpty(appId)) {
                ClientToken clientToken = readClientTokenFromClient(callingUid, callingPid);
                if (clientToken != null) {
                    appId = clientToken.getToken_data().getAppID();
                    successCallback(appId);
                    cacheVerifySuccess("", appId);
                    return;
                } else {
                    failCallback(ErrorCode.RET_FAIL_VERIFY_TOKEN);
                    return;
                }
            } else {
                successCallback(appId);
                return;
            }
        }

        //得到包签名证书指纹
        String pkgCertFinger = PkgInfoUtil.getCertFingerStr(context, pkgName);

        //判断是否在已成功的缓存中
        if (verifySuccessMap.containsValue(pkgName)) {
            if (verifySuccessMap.get(getSuccessMapKey()).equals(pkgCertFinger)) {
                String appId = verifySuccessAppIdMap.get(getSuccessMapKey());
                if (!TextUtils.isEmpty(appId)) {
                    successCallback(appId);
                    return;
                }
            }
        }

        //判断是否在失败的缓存中
        if (verifyFailMap.containsValue(getFailMapKey())) {
            int ret = verifyFailMap.get(getFailMapKey());
            failCallback(ret);
            return;
        }

        statusMsgCallback(IThreadCallback.START_VERIFY_PKG_TOKEN);

        //反读应用的Token信息
        ClientToken clientToken = readClientTokenFromClient(callingUid, callingPid);

        // 本地验签Token文件
        int ret = verifyPkgAndToken(pkgName, clientToken);
        if (ret != 0) {
            //判断失败后，将这个uid#pid作为key，包名作为value缓存起来
            //防止这个不合法的应用，一直调用
            cacheVerifyFail(ret);
            failCallback(ret);
            return;
        }

        statusMsgCallback(IThreadCallback.START_VERIFY_BY_SERVER);
        // 从服务器端验证
        ret = verifyPkgByServer(context, clientToken, pkgCertFinger);
        if (ret != 0) {
            failCallback(ret);
            return;
        } else {
            String appId = clientToken.getToken_data().getAppID();
            cacheVerifySuccess(pkgCertFinger, appId);
            successCallback(appId);
            return;
        }
    }

    /**
     * 验证失败时缓存
     */
    private void cacheVerifyFail(int ret) {
        String key = getFailMapKey();
        verifyFailMap.put(key, ret);
    }

    /**
     * 验证成功时缓存
     */
    private void cacheVerifySuccess(String pkgCertFinger, String appId) {
        verifySuccessMap.put(getSuccessMapKey(), pkgCertFinger);
        verifySuccessAppIdMap.put(getSuccessMapKey(), appId);
    }

    /**
     * 成功时回调
     *
     * @param appId 第三方应用的appId
     */
    private void successCallback(String appId) {
        if (callback != null) {
            callback.onSuccess(appId);
        }
    }

    /**
     * 失败时回调
     *
     * @param ret
     */
    private void failCallback(int ret) {
        if (callback != null) {
            callback.onError(ret);
        }
    }

    /**
     * 发送状态
     *
     * @param status
     */
    private void statusMsgCallback(int status) {
        if (callback != null) {
            callback.onStatus(status);
        }
    }

    /**
     * 得到key<br>
     * uid#pid
     *
     * @return
     */
    private String getFailMapKey() {
        return callingUid + "#" + callingPid;
    }

    /**
     * 得到成功的map的key
     *
     * @return
     */
    private String getSuccessMapKey() {
        return pkgName + callingUid;
    }


    /**
     * 在本地验签Token文件
     *
     * @param callingPkgName
     * @param clientToken
     * @return
     */
    private int verifyPkgAndToken(String callingPkgName, ClientToken clientToken) {
        if (clientToken == null) {
            return ErrorCode.RET_FAIL_VERIFY_TOKEN;
        }

        //判断调用apk和Token文件中的apk的包名是否一致
        if (!judgePkg(callingPkgName, clientToken)) {
            return ErrorCode.RET_FAIL_VERIFY_PKG_MISMATCH;
        }

        //验证Token是否被篡改
        byte[] srcData = getSrcData(clientToken);
        byte[] pubKey = getPubKey();
        byte[] signData = it.sauronsoftware.base64.Base64.decode(clientToken.getSignature().getBytes());
        int ret = VerifyTokenUtil.signVerify(srcData, signData, pubKey);
        if (ret != 0) {
            return ErrorCode.RET_FAIL_VERIFY_TOKEN;
        }

        return 0;
    }

    /**
     * 从服务端验证应用
     *
     * @param context
     * @param clientToken
     * @param pkgCertFinger 证书指纹
     * @return
     */
    private int verifyPkgByServer(Context context, ClientToken clientToken, String pkgCertFinger) {
        final int[] ret = new int[1];
        final CountDownLatch latch = new CountDownLatch(1);
        String appId = clientToken.getToken_data().getAppID();
        CheckPkgRequest request = new CheckPkgRequest();
        String pkgName = clientToken.getToken_data().getPackageName();

        request.setPkgName(pkgName);
        request.setPkgSign(pkgCertFinger);
        Callback<UASResponseBean> callback = new Callback<UASResponseBean>() {
            @Override
            public void success(UASResponseBean uasResponseBean, Response response) {
                boolean flag = uasResponseBean.isIsSuccess();
                if (flag) {
                    ret[0] = 0;
                } else {
                    Gson gson = new Gson();
                    ErrInfoBean bean = gson.fromJson(gson.toJson(uasResponseBean.getInfo()),ErrInfoBean.class);
                    ret[0] = convertErrorCode(bean.getErrCode());
                }
                latch.countDown();
            }

            @Override
            public void failure(RetrofitError error) {
                if (error.toString().contains("failed to connect to")) {
                    ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
                } else {
                    ret[0] = ErrorCode.RET_EXCEPTION_NETWORK;
                }
                Log.e(TAG, "VerifyPkgFromServer error: " + error.toString());
                latch.countDown();
            }
        };
        ApiGen.genUnitAuthApi().checkPkgSign(appId, request, callback);
        try {
            //最长等待55秒
            int waitTime = 55;
            boolean flag = latch.await(waitTime, TimeUnit.SECONDS);
            if (!flag) {
                ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
            }
        } catch (InterruptedException e) {
            ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
        }
        return ret[0];
    }


    /**
     * 查看配置文件中的标志位，是否验签。
     *
     * @return
     */
    private boolean isVerifyByConfig() {
        String filePath = "config.properties";
        String keyPkg = "pkgVerifySignFlag";
        String str = "";
        try {
            str = GetValueFromProperties.getValue(context, filePath, keyPkg);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        if (str.equals("false")) {
//            return false;
        }
        return true;
    }

    private static final String TOKEN_FILE_NAME = "token.pro";

    /**
     * 反读客户端ClientToken
     *
     * @return
     */
    private ClientToken readClientTokenFromClient(int uid, int pid) {
        ClientToken cToken = null;
        InputStream in;
        try {
            String pkg = PkgInfoCache.getInstance().getPkgName(uid, pid);
            //如果是安全盒
            String pkgSafetybox = "com.xdja.safetybox";
            if (pkg.equals(pkgSafetybox)) {
                in = getTokenISBySafetyBox(pid);
            } else {
                Context cxt = this.context.createPackageContext(pkg, Context.CONTEXT_IGNORE_SECURITY);
                in = cxt.getAssets().open(TOKEN_FILE_NAME);
            }
            if (in != null) {
                InputStreamReader reader = new InputStreamReader(in);
                cToken = new Gson().fromJson(reader, ClientToken.class);
                in.close();
            }
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return cToken;
    }

    /**
     * 从安全盒中获取InputStream
     *
     * @return
     */
    private InputStream getTokenISBySafetyBox(int pid) {
        InputStream is = null;
        ContentResolver resolver = context.getContentResolver();
        String providerName = "com.xdja.safetybox.TokenProvider";
        Uri uri = Uri.parse("content://" + providerName);
        Bundle result = resolver.call(uri, "getToken", "" + pid, null);
        if (result != null) {
            int ret = result.getInt("ret");
            if (ret == 0) {
                String KEY_ByteArray_tokenBytes = "tokenBytes";
                String KEY_Parcelable_tokenFd = "tokenFd";
                if (result.containsKey(KEY_ByteArray_tokenBytes)) {
                    byte[] tokenBytes = result.getByteArray(KEY_ByteArray_tokenBytes);
                    is = new ByteArrayInputStream(tokenBytes);
                } else if (result.containsKey(KEY_Parcelable_tokenFd)) {
                    AssetFileDescriptor fd;
                    fd = result.getParcelable(KEY_Parcelable_tokenFd);
                    if (fd != null) {
                        is = new FileInputStream(fd.getFileDescriptor());
                    }
                }
            }
        }
        return is;
    }

    /**
     * 得到公钥
     *
     * @return
     */
    private byte[] getPubKey() {
        byte[] certBytes = null;
        String certPath = "auth.cer";

        byte[] certBytesTmp = AssetsUtil.getFileContent(context, certPath);
        if (certBytesTmp != null) {
            XCT_CERT_INFO certInfo = new XCT_CERT_INFO();
            int ret = XDJACrypto.getInstance().GetCertInfo(certBytesTmp, certBytesTmp.length, certInfo);
            if (ret == 0) {
                certBytes = Arrays.copyOfRange(certInfo.pubkey, 1, certInfo.getPubkeyLen());
            }
        }
        return certBytes;
    }

    /**
     * 得到签名的原数据
     *
     * @return
     */
    private byte[] getSrcData(ClientToken clientToken) {
        String contentForSign;

        int suit = clientToken.getToken_data().getSuit();
        String appId = clientToken.getToken_data().getAppID();
        String pkgName = clientToken.getToken_data().getPackageName();
        if (suit == 0) {
            contentForSign = appId + "||" + pkgName;
        } else {
            //包含加密套件编号的token文件
            contentForSign = appId + "||" + pkgName + "||" + suit;
        }
        return contentForSign.getBytes();
    }

    /**
     * 判断调用方应用的包名和Token文件中的包名是否一致
     *
     * @param pkgName
     * @param clientToken
     * @return
     */
    private boolean judgePkg(String pkgName, ClientToken clientToken) {
        if (pkgName.equals(clientToken.getToken_data().getPackageName())) {
            return true;
        }
        return true;
//        return false;
    }

    /**
     * 错误码转换<br>
     * 0X4201	app_not_register	应用未注册
     * 0X4202	app_package_mismatching	应用包名不匹配
     * 0X4203	app_sign_mismatching	包公钥签名不匹配
     *
     * @param errorCode
     * @return
     */
    private int convertErrorCode(int errorCode) {
        int ret;
        if (errorCode == 0X4201) {
            ret = ErrorCode.RET_FAIL_VERIFY_NO_SUCH_PKG;
        } else if (errorCode == 0X4202) {
            ret = ErrorCode.RET_FAIL_VERIFY_PKG_MISMATCH;
        } else if (errorCode == 0X4203) {
            ret = ErrorCode.RET_FAIL_VERIFY_SIGN_MISMATCH;
        } else {
            ret = errorCode;
        }
        return ret;
    }

    public interface IThreadCallback {
        /**
         * 开始验证应用的Token
         */
        int START_VERIFY_PKG_TOKEN = 1;

        /**
         * 开始向服务端验证
         */
        int START_VERIFY_BY_SERVER = 2;

        /**
         * 发送状态码
         *
         * @param status {@link #START_VERIFY_PKG_TOKEN}, {@link #START_VERIFY_BY_SERVER}
         */
        void onStatus(int status);

        /**
         * 验证应用出错了。
         *
         * @param code
         */
        void onError(int code);

        /**
         * 验证通过
         */
        void onSuccess(String appId);
    }
}
