package com.xdja.uniteauth.net;

import android.content.Context;
import android.util.Log;

import com.squareup.okhttp.OkHttpClient;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.utils.PinUtil;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManagerFactory;

/**
 * <p>Summary:融入单项认证的OkHttpClient</p>
 */
public class OkHttpsClient_UAC extends OkHttpClient {
    /**
     * 默认的PIN码
     */
    private static final String DEFAULT_PASSWORD = PinUtil.getDefaultPin();

    private static final int TIME_OUT_UNIT = 30 * 1000;

    private final String tag = "uac_okhttps";

    private String password;

    /**
     * 是否验证服务器主机地址
     */
    private boolean isVerifyHostName = false;


    /**
     * {@link OkHttpsClient_UAC}
     */
    public OkHttpsClient_UAC(Context context) {
        this(DEFAULT_PASSWORD, context);
    }

    /**
     * {@link OkHttpsClient_UAC}
     *
     * @param password 安全卡PIN码
     */
    public OkHttpsClient_UAC(String password,Context context) {
        this.password = password;
        this.setConnectTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS);
        this.setReadTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS);
        this.setWriteTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS);
        build(context);
    }

    private OkHttpsClient_UAC build(Context context) {
        SSLContext sslContext = getSSLContext(
                readKeyStore(R.raw.truststore1, context));
        if (sslContext == null) {
            return null;
        }
        this.setSslSocketFactory(sslContext.getSocketFactory());
        if (!isVerifyHostName) {
            this.setHostnameVerifier(new HostnameVerifier() {
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            });
        }
        return this;
    }




    /**
     * 读取本地证书
     *
     * @param res 证书资源ID
     * @return 获取到的证书
     */
    private KeyStore readKeyStore(int res, Context context) {
        InputStream inputStream = null;
        KeyStore keyStore = null;
        try {
            keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            inputStream = context.getResources().openRawResource(res);
            keyStore.load(inputStream, password.toCharArray());
        } catch (KeyStoreException | CertificateException
                | NoSuchAlgorithmException | IOException kse) {
            Log.d(tag, kse.getMessage());
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException ie) {
                    Log.d(tag, ie.getMessage());
                }
            }
            return keyStore;
        }
    }

    /**
     * 获取SSL连接上下文
     *
     * @param keyStore
     * @return
     */
    private SSLContext getSSLContext(KeyStore keyStore) {
        if (keyStore == null) {
            return null;
        }
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("TLS");

            TrustManagerFactory trustManagerFactory =
                    TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(keyStore);

            KeyManagerFactory keyManagerFactory =
                    KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keyStore, password.toCharArray());

            sslContext.init(keyManagerFactory.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());
        } catch (NoSuchAlgorithmException | KeyStoreException
                | UnrecoverableKeyException | KeyManagementException ne) {
            Log.d(tag,ne.getMessage());
        } finally {
            return sslContext;
        }
    }
}
