package com.xdja.ckms.common.chip;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.ckms.chip.CKMSChip;
import com.xdja.multichip.jniapi.JarMultiJniApiManager;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.safekeyjar.XDJASafeTF;
import com.xdja.safekeyjar.util.Arithmetic;
import com.xdja.safekeyjar.util.StringResult;
import com.xdja.unitepin.UnitePinManager;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/6/20 <br/>
 */

public class ChipOperate {

    public static Pair<Integer, List<AppChipParam>> getAll(Context context) {

        Log.w("GetAll","ChipOperate getMultiJniApi start: " );
        Pair<Integer, List<JniApiParam>> listPair = JarMultiJniApiManager.getInstance().getAll(context);
        if (listPair.first != 0) {
            return Pair.create(listPair.first, null);
        }
        List<AppChipParam> jniApiParamList = new ArrayList<>();
        for (JniApiParam jniApiParam : listPair.second) {
            jniApiParamList.add(convert(jniApiParam));
        }
        return Pair.create(listPair.first, jniApiParamList);
    }

    public static Pair<Integer, List<AppChipParam>> getChip(Context context, int chipType) {
        Pair<Integer, List<JniApiParam>> listPair = JarMultiJniApiManager.getInstance().getAll(context);
        if (listPair.first != 0) {
            return Pair.create(listPair.first, null);
        }
        List<AppChipParam> jniApiParamList = new ArrayList<>();
        for (JniApiParam jniApiParam : listPair.second) {
            if (jniApiParam.chipType == chipType) {
                jniApiParamList.add(convert(jniApiParam));
            }
        }
        return Pair.create(listPair.first, jniApiParamList);
    }

    /**
     * 修改PIN码
     *
     * @param context context
     * @param param   参数。如果param为null，则使用XDJASafeTF、UnitePinManager的方式；
     *                如果为AppParamChip，则使用MultiChipJar的方式
     * @param oldPin  oldPin
     * @param newPin  newPin
     * @return
     */
    public static int changePin(Context context, Object param, String oldPin, String newPin, int role) {
        if (param == null) {
            return useUnitePinManagerChangePin(context, oldPin, newPin, role);
        } else if (param instanceof AppChipParam) {
            return useMultiChipJarChangePin(context, (AppChipParam) param, oldPin, newPin, role);
        } else {
            return useUnitePinManagerChangePin(context, oldPin, newPin, role);
        }
    }

    /**
     * 验证PIN码
     *
     * @param context context
     * @param param   参数。如果param为null，则使用XDJASafeTF、UnitePinManager的方式；
     *                如果为AppParamChip，则使用MultiChipJar的方式
     * @param pin     pin
     * @return
     */
    public static int verifyPin(Context context, Object param, String pin, int role) {
        if (param == null) {
            return useUnitePinManagerVerifyPin(context, pin, role);
        } else if (param instanceof AppChipParam) {
            return useMultiChipJarVerifyPin(context, (AppChipParam) param, pin, role);
        } else {
            return useUnitePinManagerVerifyPin(context, pin, role);
        }
    }

    /**
     * 得到PIN码重试次数
     *
     * @param context context
     * @param param   param
     * @param role    role
     * @return
     */
    public static int getPinTryCount(Context context, Object param, int role) {
        if (param == null) {
            return useXDJASafeTFGetPinTryCount(context, role);
        } else if (param instanceof AppChipParam) {
            return useMultiChipJarGetPinTryCount(context, (AppChipParam) param, role);
        } else {
            return useXDJASafeTFGetPinTryCount(context, role);
        }
    }

    /**
     * 解锁PIN码
     *
     * @param context
     * @param param
     * @param role
     * @param usn
     * @param usnlockCode
     * @return
     */
    public static int unLockPin(Context context, Object param, int role, byte[] usn, byte[] usnlockCode) {
        if (param == null) {
            return useUnitePinManagerUnlockPin(context, role, usn, usnlockCode);
        } else if (param instanceof AppChipParam) {
            return useMultiChipJarUnlockPin(context, (AppChipParam) param, role, usn, usnlockCode);
        } else {
            return useUnitePinManagerUnlockPin(context, role, usn, usnlockCode);
        }
    }

    /**
     * 得到cos的版本信息
     *
     * @param context context
     * @param param   param
     * @return
     */
    public static String getCosInfo(Context context, AppChipParam param) {
        if (param == null) {
            XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
//            int ret = XDJASafeTF.getInstance().GetDevInfo(devinfo);
            int ret = CKMSChip.getInstance().GetDevInfo(devinfo);
            if (ret == 0) {
                return new String(devinfo.cosver);
            } else {
                return "";
            }
        }

        JniApiParam jniApiParam = convertToJniApiParam(param);

        return UseMultiJniApi.getCosInfo(context, jniApiParam);
    }

    /**
     * 得到业务上用的sn。
     * 默认使用6号容器的签名证书
     *
     * @param context
     * @param param
     * @return
     */
    public static String getSn(Context context, Object param) {
        String sn = "";
        if (param == null) {
            return useXDJASafeTFGetSn(context);
        } else if (param instanceof AppChipParam) {
            return useMultiChipJarGetSn(context, (AppChipParam) param);
        }
        return sn;
    }

    /**
     * 获取卡号
     *
     * @param context
     * @param param
     * @return
     */
    public static String getCardId(Context context, Object param) {
        if (param == null) {
            return useXDJASafeTFGetCardId(context);
        } else if (param instanceof AppChipParam) {
            return ((AppChipParam) param).uniqueID;
        } else {
            return useXDJASafeTFGetCardId(context);
        }
    }

    //add 2017年6月29日15:43:32 weizg

    /**
     * 获取intent中芯片信息bean
     *
     * @param intent intent
     * @return 获取出错或者intent中没有该参数，则返回null；否则返回AppParamChip对象
     */
    public static AppChipParam getAppParamChip(Intent intent) {
        if (intent == null) {
            return null;
        }
        Bundle bundle = intent.getExtras();
        if (bundle == null || !bundle.containsKey(param_AppParamChip)) {
            return null;
        }
        bundle.setClassLoader(AppChipParam.class.getClassLoader());
        return bundle.getParcelable(param_AppParamChip);
    }

    public static AppChipParam getBleAppChipParam(String cardId) {
        AppChipParam appChipParam = new AppChipParam();
        appChipParam.chipType = AppChipParam.TYPE_BLUETOOTH;
        appChipParam.manufacturer = AppChipParam.MANUFACTURER_XDJA;
        appChipParam.uniqueID = cardId;
        return appChipParam;
    }

    public static Pair<String, String> getCardIdAndSn(Context context, AppChipParam AppParamChip) {
        String cardId = "";
        if (AppParamChip == null) {
            //modify 2017年7月28日10:16:16 weizg 如果兼容AppParamChip为null的情况，当既有芯片和tf卡时，
            //可能会出现获取错误芯片信息情况
//            cardId = ChipOperate.getCardId(context.getApplicationContext(),null);
            cardId = "";
        } else {
            cardId = AppParamChip.uniqueID;
        }
        String sn = ChipOperate.getSn(context.getApplicationContext(), AppParamChip);
        if (TextUtils.isEmpty(cardId) || TextUtils.isEmpty(sn)) {
            return null;
        }

        return Pair.create(cardId, sn);
    }

    public static final String param_AppParamChip = "AppChipParam";

    //↓--------------------------------------------------------------------------------↓
    //↓                           私有方法
    //↓--------------------------------------------------------------------------------↓

    /**
     * 使用UnitePinManager修改PIN码
     *
     * @param context context
     * @param oldPin  oldPin
     * @param newPin  newPin
     * @return
     */
    private static int useUnitePinManagerChangePin(Context context, String oldPin, String newPin, int role) {
        return CKMSChip.getInstance().ChangePIN(role, oldPin.getBytes(), oldPin.length(), newPin.getBytes(), newPin.length());
//        return UnitePinManager.getInstance().uniteChangePin(role, oldPin, newPin);
    }

    /**
     * 使用MultiChipJar修改PIN码
     *
     * @param context      context
     * @param appParamChip appParamChip
     * @param oldPin       oldPin
     * @param newPin       newPin
     * @return
     */
    private static int useMultiChipJarChangePin(Context context, AppChipParam appParamChip, String oldPin, String newPin, int role) {
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        return UseMultiJniApi.ChangePin(context, jniApiParam, oldPin, newPin, role);
    }

    /**
     * 使用UnitePinManager修改PIN码
     *
     * @param context context
     * @param pin     pin
     * @return
     */
    private static int useUnitePinManagerVerifyPin(Context context, String pin, int role) {
        return CKMSChip.getInstance().VerifyPIN(role, pin.getBytes(), pin.length());
//        return UnitePinManager.getInstance().uniteVerifyPin(role, pin);
    }

    /**
     * 使用MultiChipJar验证PIN码
     *
     * @param context      context
     * @param appParamChip appParamChip
     * @param pin          pin
     * @return
     */
    private static int useMultiChipJarVerifyPin(Context context, AppChipParam appParamChip, String pin, int role) {
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        return UseMultiJniApi.VerifyPin(context, jniApiParam, pin, role);
    }

    /**
     * 使用XDJASafeTF得到PIN码重试次数
     *
     * @param context context
     * @param role    role
     * @return
     */
    private static int useXDJASafeTFGetPinTryCount(Context context, int role) {
//        int ret = XDJASafeTF.getInstance().GetPinTryCount(role);
//        if (ret == XDJASafeTF.INIT_NOT) {
//            //未初始化，则初始化一次
//            XDJASafeTF.getInstance().init(context.getApplicationContext());
//        }
//
//        return XDJASafeTF.getInstance().GetPinTryCount(role);
        return CKMSChip.getInstance().GetPinTryCount(role);
    }

    /**
     * 使用MultiChipJar获取PIN码重试次数
     *
     * @param context
     * @param appParamChip
     * @param role
     * @return
     */
    private static int useMultiChipJarGetPinTryCount(Context context, AppChipParam appParamChip, int role) {
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        return UseMultiJniApi.GetPinTryCount(context, jniApiParam, role);
    }

    /**
     * 使用UnitePinManager解锁PIN
     *
     * @param context     context
     * @param role        role
     * @param usn         usn
     * @param usnlockCode usnlockCode
     * @return
     */
    private static int useUnitePinManagerUnlockPin(Context context, int role, byte[] usn, byte[] usnlockCode) {
        return UnitePinManager.getInstance().uniteUnlockPin(role, usn, usnlockCode);
    }

    /**
     * 使用多MultiChipJar解锁PIN
     *
     * @param context      context
     * @param appParamChip AppChipParam
     * @param role         role
     * @param usn          usn
     * @param usnlockCode  usnlockCode
     * @return
     */
    private static int useMultiChipJarUnlockPin(Context context, AppChipParam appParamChip, int role, byte[] usn, byte[] usnlockCode) {
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        return UseMultiJniApi.UnlockPin(context, jniApiParam, role, usn, usnlockCode);
    }

    /**
     * 使用XDJASafeTF接口获取sn
     *
     * @param context
     * @return
     */
    private static String useXDJASafeTFGetSn(Context context) {
        int containNo = 6;
        int certType = 2;
        String chipCerSN = "";
//        StringResult result = XDJASafeTF.getInstance().getSafeCardSn(containNo, certType);
        StringResult result = CKMSChip.getInstance().getSafeCardSn(containNo, certType);
        if (result.getErrorCode() == 0) {
            // 首位不加零，小写
            chipCerSN = result.getResult().toLowerCase();
        }
        return chipCerSN;
    }

    /**
     * 使用MultiChipJar获取sn
     *
     * @param context
     * @param appParamChip
     * @return
     */
    private static String useMultiChipJarGetSn(Context context, AppChipParam appParamChip) {
        String chipCerSN = "";
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        Pair<Integer, byte[]> bytePair = UseMultiJniApi.GetCert(context, jniApiParam);
        if (bytePair.first != 0) {
            return chipCerSN;
        }

        byte[] certBuf = bytePair.second;
        int certLen = certBuf.length;
        X509Certificate x509Cert = Arithmetic.getX509Certificate(certBuf, certLen);
        if (x509Cert != null) {
            byte[] certSn = x509Cert.getSerialNumber().toByteArray();
            System.arraycopy(certSn, 0, certBuf, 0, certSn.length);
            certLen = certSn.length;
        } else {
            return "";
        }

        for (int i = 0; i < certLen; i++) {
            chipCerSN += String.format("%02x", certBuf[i]);
        }

        return chipCerSN.toLowerCase();
    }

    /**
     * 使用XDJASafeTF获取卡号
     *
     * @param context
     * @return
     */
    private static String useXDJASafeTFGetCardId(Context context) {
        String chipId = "";
        XDJA_DEVINFO info = new XDJA_DEVINFO();
//        XDJASafeTF.getInstance().init(context);
        int ret = XDJASafeTF.getInstance().GetDevInfo(info);
        if (ret == 0) {
            chipId = new String(info.cardid, 0, 32);
        }
        return chipId;
    }

    /**
     * 判断芯片是否锁死
     *
     * @param context      上下文环境
     * @param AppParamChip 芯片信息
     * @param role         角色
     * @return true 锁死；否则表示未锁死或者获取失败
     */
    public static boolean judgeSafeKeyIsLocked(Context context, AppChipParam AppParamChip, int role) {
        boolean result = false;
        int ret;
        ret = getPinTryCount(context, AppParamChip, role);
        if (ret == JNIAPI.XKR_PASSWORD || ret == JNIAPI.XKR_KEY_LOCKED) {
            result = true;
        }
        return result;
    }

    /**
     * 使用MultiChipJar获取六号容器签名证书
     *
     * @param context      上下文环境
     * @param appParamChip 芯片信息
     * @return 证书pair
     */
    public static Pair<Integer, byte[]> useMultiChipJarGetCert(Context context, AppChipParam appParamChip) {
        JniApiParam jniApiParam = convertToJniApiParam(appParamChip);
        return UseMultiJniApi.GetCert(context, jniApiParam);
    }

    private static AppChipParam convert(JniApiParam jniApiParam) {
        if (jniApiParam == null) {
            return null;
        }
        AppChipParam result = new AppChipParam();
        result.chipType = jniApiParam.chipType;
        result.manufacturer = AppChipParam.MANUFACTURER_XDJA;
        result.uniqueID = jniApiParam.cardId;
        return result;
    }

    private static JniApiParam convertToJniApiParam(AppChipParam appParamChip) {
        if (appParamChip == null) {
            return null;
        }
        JniApiParam jniApiParam = new JniApiParam();
        jniApiParam.cardId = appParamChip.uniqueID;
        jniApiParam.chipType = appParamChip.chipType;
        return jniApiParam;
    }
}
