package com.xdja.ckms.common.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.text.Editable;
import android.text.Layout;
import android.text.StaticLayout;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.ViewGroup;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.PopupWindow;
import android.widget.TextView;

import com.xdja.ckms.common.R;


/**
 *
 */
public class XEditText extends EditText {

    private boolean hasFallen = false;
    private ErrorPopup mPopup;
    private Drawables mDrawables;
    private CharSequence mError;

    private boolean mShowErrorAfterAttach;
    Drawable mErrorIcon;
    Drawable mErrorBackgroundAbove;
    Drawable mErrorBackground;
    int mErrorTextColor;

    float errorTextSize;
    /**
     * 输入内容改变监听
     */
    XTextWatcher xTextWatcher;
    /**
     * 校验规则
     */
    XValidator xValidator;

    public void setmErrorIcon(Drawable mErrorIcon) {
        this.mErrorIcon = mErrorIcon;
    }

    public void setmErrorBackground(Drawable mErrorBackground) {
        this.mErrorBackground = mErrorBackground;
    }

    public void setmErrorBackgroundAbove(Drawable mErrorBackgroundAbove) {
        this.mErrorBackgroundAbove = mErrorBackgroundAbove;
    }

    public void setErrorTextSize(float errorTextSize) {
        this.errorTextSize = errorTextSize;
    }

    public void setmErrorTextColor(int mErrorTextColor) {
        this.mErrorTextColor = mErrorTextColor;
    }

    public XEditText(Context context) {
        super(context);
        init(context, null);
    }

    public XEditText(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    public XEditText(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.XEditText);

        if (a.hasValue(R.styleable.XEditText_xErrorDefaultIcon)) {
            mErrorIcon = a.getDrawable(R.styleable.XEditText_xErrorDefaultIcon);
        }
        if (mErrorIcon == null) {
        }
        if (a.hasValue(R.styleable.XEditText_xErrorDefaultBackground)) {
            mErrorBackground = a.getDrawable(R.styleable.XEditText_xErrorDefaultBackground);
        }
        if (mErrorBackground == null) {
        }
        if (a.hasValue(R.styleable.XEditText_xErrorDefaultBackgroundAbove)) {
            mErrorBackgroundAbove = a.getDrawable(R.styleable.XEditText_xErrorDefaultBackgroundAbove);
        }
        if (mErrorBackgroundAbove == null) {
        }
        if (a.hasValue(R.styleable.XEditText_xErrorTextColor)) {
            mErrorTextColor = a.getColor(R.styleable.XEditText_xErrorTextColor, Color.rgb(50, 50, 50));
        }
        this.addTextChangedListener(new TextWatcherImp());
    }

    /**
     * 设置输入内容改变监听
     *
     * @param xTextWatcher {@link #xTextWatcher}
     */
    public void setxTextWatcher(XTextWatcher xTextWatcher) {
        this.xTextWatcher = xTextWatcher;
    }

    /**
     * @param xValidator {@link #xValidator}
     */
    public void setxValidator(XValidator xValidator) {
        this.xValidator = xValidator;
    }

    /**
     * 输入内容改变监听接口
     */
    public interface XTextWatcher {
        void beforeTextChanged(CharSequence s, int start, int count, int after);

        void onTextChanged(CharSequence s, int start, int before, int count);

        void afterTextChanged(Editable s);
    }

    /**
     * 校验输入内容是否合法接口定义
     */
    public interface XValidator {
        /**
         * 校验规则
         *
         * @param charSequence
         * @return 是否合法
         */
        public String validat(CharSequence charSequence);

        /**
         * 如果不合法，显示的提示信息
         *
         * @return
         */
//        public String getErrMsg();
    }

    class TextWatcherImp implements TextWatcher {
        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            if (xTextWatcher != null) {
                xTextWatcher.beforeTextChanged(s, start, count, after);
            }
        }

        @Override
        public void onTextChanged(CharSequence s, int start, int before, int count) {
            if (xTextWatcher != null) {
                xTextWatcher.onTextChanged(s, start, before, count);
            }
            hideError();
        }

        @Override
        public void afterTextChanged(Editable s) {
            if (xTextWatcher != null) {
                xTextWatcher.afterTextChanged(s);
            }
        }
    }

    @Override
    public void setError(CharSequence error) {
        if (error == null) {
            setError(null, null);
        } else {
            Drawable dr = mErrorIcon;

            if (dr != null) {
                dr.setBounds(0, 0, dr.getIntrinsicWidth(), dr.getIntrinsicHeight());
            }
            setError(error, dr);
        }
    }

    private void setTheError(CharSequence error) throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
        error = TextUtils.stringOrSpannedString(error);
        mError = error;
        return;

    }

    @Override
    public void setError(CharSequence error, Drawable icon) {
        try {
            setTheError(error);
        } catch (Exception ex) {
            ex.printStackTrace();
            hasFallen = true;
            super.setError(error, icon);
            return;
        }
        final Drawables dr = mDrawables;
        if (dr != null) {
            setCompoundDrawables(dr.mDrawableLeft, dr.mDrawableTop,
                    icon, dr.mDrawableBottom);
        } else {
            setCompoundDrawables(null, null, icon, null);
        }

        if (error == null) {
            if (mPopup != null) {
                if (mPopup.isShowing()) {
                    mPopup.dismiss();
                }

                mPopup = null;
            }
        } else {
            showError();
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        if (!hasFallen && mShowErrorAfterAttach) {
            showError();
            mShowErrorAfterAttach = false;
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (!hasFallen && mError != null) {
            hideError();
        }
    }

    private void hideError() {
        if (mPopup != null) {
            if (mPopup.isShowing()) {
                setCompoundDrawables(null, null, null, null);
                mPopup.dismiss();
            }
        }
        mShowErrorAfterAttach = false;
    }

    private void showError() {
        if (getWindowToken() == null) {
            mShowErrorAfterAttach = true;
            return;
        }

        if (mPopup == null) {
            final float scale = getResources().getDisplayMetrics().density;
            final TextView err = new TextView(getContext());
            err.setTextColor(mErrorTextColor);
            err.setTextSize(errorTextSize);
            err.setGravity(Gravity.CENTER);
            err.setPadding((int) (16 * scale + 0.5f), 0
                    , (int) (16 * scale + 0.5f)
                    , 0);
            LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            layoutParams.setMargins(0, 0, (int) (20 * scale + 0.5f), 0);
            err.setLayoutParams(layoutParams);

            mPopup = new ErrorPopup(err, (int) (130 * scale + 0.5f),
                    (int) (30 * scale + 0.5f));
            mPopup.setFocusable(false);
            mPopup.setInputMethodMode(PopupWindow.INPUT_METHOD_NEEDED);
        }

        TextView tv = (TextView) mPopup.getContentView();
        tv.setTextSize(12);
        chooseSize(mPopup, mError, tv);
        tv.setText(mError);


        mPopup.showAsDropDown(this, getErrorX(), getErrorY());
        mPopup.fixDirection(mPopup.isAboveAnchor());
    }

    @Override
    protected void onFocusChanged(boolean focused, int direction, Rect previouslyFocusedRect) {
        super.onFocusChanged(focused, direction, previouslyFocusedRect);
        if (focused) {
//            hideError();
        } else {
            if (xValidator != null) {
                String validStr = xValidator.validat(getText());
                if (!TextUtils.isEmpty(validStr)) {
                    setError(validStr);
                }
            }
        }
    }

    /**
     * 校验输入
     */
    public boolean valid() {
        if (xValidator != null) {
            String validStr = xValidator.validat(getText());
            if (!TextUtils.isEmpty(validStr)) {
                setError(validStr);
                return false;
            }
        }
        return true;
    }

    private int getErrorX() {
        final float scale = getResources().getDisplayMetrics().density;

        final Drawables dr = mDrawables;
        return getWidth()
                - mPopup.getWidth()
                - getPaddingRight()
                - (dr != null ? dr.mDrawableSizeRight : 0) / 2 + (int) (10 * scale + 0.5f)
                + (int) (26 * scale + 0.5f);
    }


    private int getErrorY() {
        int vspace = getBottom() - getTop() -
                getCompoundPaddingBottom() - getCompoundPaddingTop();

        final Drawables dr = mDrawables;

        final float scale = getResources().getDisplayMetrics().density;
        int icontop = getCompoundPaddingTop()
                + (vspace - (dr != null ? dr.mDrawableHeightRight : 0)) / 2;

        int res =  icontop + (dr != null ? dr.mDrawableHeightRight : 0)
                - getHeight() - 2 + (int) (3 * scale + 0.5f);
        return res;
    }

    private void chooseSize(PopupWindow pop, CharSequence text, TextView tv) {
        tv.setPadding(tv.getPaddingLeft(), tv.getPaddingTop(), tv.getCompoundPaddingRight(), tv.getPaddingBottom());
        int wid = tv.getPaddingLeft() + tv.getPaddingRight();
        int ht = tv.getPaddingTop() + tv.getPaddingBottom();
        //add by xwy
        final float scale = getResources().getDisplayMetrics().density;

        int cap = getWidth() - wid;
        if (cap < 0) {
            cap = 200;
        }

        Layout l = new StaticLayout(text, tv.getPaint(), cap + 80,
                Layout.Alignment.ALIGN_CENTER, 1, 0, true);
        float max = 0;
        for (int i = 0; i < l.getLineCount(); i++) {
            max = Math.max(max, l.getLineWidth(i));
        }

        pop.setWidth(wid + (int) Math.ceil(max));
        //pop.setHeight(ht + l.getHeight() + 20);
        pop.setHeight( (int) (30 * scale + 0.5f));
    }

    class Drawables {
        final Rect mCompoundRect = new Rect();
        Drawable mDrawableTop, mDrawableBottom, mDrawableLeft, mDrawableRight;
        int mDrawableSizeTop, mDrawableSizeBottom, mDrawableSizeLeft, mDrawableSizeRight;
        int mDrawableWidthTop, mDrawableWidthBottom, mDrawableHeightLeft, mDrawableHeightRight;
        int mDrawablePadding;
    }

    private class ErrorPopup extends PopupWindow {
        private boolean mAbove = false;
        private TextView mView;

        ErrorPopup(TextView v, int width, int height) {
            super(v, width, height);
            mView = v;
        }

        void fixDirection(boolean above) {
            mAbove = above;

            if (above) {
                if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN) {
                    mView.setBackground(mErrorBackgroundAbove);
                } else {
                    mView.setBackgroundDrawable(mErrorBackgroundAbove);
                }
            } else {
                if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN) {
                    mView.setBackground(mErrorBackground);
                } else {
                    mView.setBackgroundDrawable(mErrorBackground);
                }
            }
        }

        @Override
        public void update(int x, int y, int w, int h, boolean force) {
            super.update(x, y, w, h, force);

            boolean above = isAboveAnchor();
            if (above != mAbove) {
                fixDirection(above);
            }
        }
    }

    @Override
    protected boolean setFrame(int l, int t, int r, int b) {
        boolean result = super.setFrame(l, t, r, b);

        if (!hasFallen && mPopup != null) {
                TextView tv = (TextView) mPopup.getContentView();
                chooseSize(mPopup, mError, tv);
                mPopup.update(this, getErrorX(), getErrorY(),
                        mPopup.getWidth(), mPopup.getHeight());
        }


        return result;
    }

    @Override
    public void setCompoundDrawables(Drawable left, Drawable top, Drawable right, Drawable bottom) {
        super.setCompoundDrawables(left, top, right, bottom);
        if (hasFallen) {
            return;
        }
        Drawables dr = mDrawables;

        final boolean drawables = left != null || top != null || right != null || bottom != null;

        if (!drawables) {
            // Clearing drawables...  can we free the data structure?
            if (dr != null) {
                if (dr.mDrawablePadding == 0) {
                    mDrawables = null;
                } else {
                    // We need to retain the last set padding, so just clear
                    // out all of the fields in the existing structure.
                    if (dr.mDrawableLeft != null) {
                        dr.mDrawableLeft.setCallback(null);
                    }
                    dr.mDrawableLeft = null;
                    if (dr.mDrawableTop != null) {
                        dr.mDrawableTop.setCallback(null);
                    }
                    dr.mDrawableTop = null;
                    if (dr.mDrawableRight != null) {
                        dr.mDrawableRight.setCallback(null);
                    }
                    dr.mDrawableRight = null;
                    if (dr.mDrawableBottom != null) {
                        dr.mDrawableBottom.setCallback(null);
                    }
                    dr.mDrawableBottom = null;
                    dr.mDrawableSizeLeft = dr.mDrawableHeightLeft = 0;
                    dr.mDrawableSizeRight = dr.mDrawableHeightRight = 0;
                    dr.mDrawableSizeTop = dr.mDrawableWidthTop = 0;
                    dr.mDrawableSizeBottom = dr.mDrawableWidthBottom = 0;
                }
            }
        } else {
            if (dr == null) {
                mDrawables = dr = new Drawables();
            }

            if (dr.mDrawableLeft != left && dr.mDrawableLeft != null) {
                dr.mDrawableLeft.setCallback(null);
            }
            dr.mDrawableLeft = left;

            if (dr.mDrawableTop != top && dr.mDrawableTop != null) {
                dr.mDrawableTop.setCallback(null);
            }
            dr.mDrawableTop = top;

            if (dr.mDrawableRight != right && dr.mDrawableRight != null) {
                dr.mDrawableRight.setCallback(null);
            }
            dr.mDrawableRight = right;

            if (dr.mDrawableBottom != bottom && dr.mDrawableBottom != null) {
                dr.mDrawableBottom.setCallback(null);
            }
            dr.mDrawableBottom = bottom;

            final Rect compoundRect = dr.mCompoundRect;
            int[] state;

            state = getDrawableState();

            if (left != null) {
                left.setState(state);
                left.copyBounds(compoundRect);
                left.setCallback(this);
                dr.mDrawableSizeLeft = compoundRect.width();
                dr.mDrawableHeightLeft = compoundRect.height();
            } else {
                dr.mDrawableSizeLeft = dr.mDrawableHeightLeft = 0;
            }

            if (right != null) {
                right.setState(state);
                right.copyBounds(compoundRect);
                right.setCallback(this);
                dr.mDrawableSizeRight = compoundRect.width();
                dr.mDrawableHeightRight = compoundRect.height();
            } else {
                dr.mDrawableSizeRight = dr.mDrawableHeightRight = 0;
            }

            if (top != null) {
                top.setState(state);
                top.copyBounds(compoundRect);
                top.setCallback(this);
                dr.mDrawableSizeTop = compoundRect.height();
                dr.mDrawableWidthTop = compoundRect.width();
            } else {
                dr.mDrawableSizeTop = dr.mDrawableWidthTop = 0;
            }

            if (bottom != null) {
                bottom.setState(state);
                bottom.copyBounds(compoundRect);
                bottom.setCallback(this);
                dr.mDrawableSizeBottom = compoundRect.height();
                dr.mDrawableWidthBottom = compoundRect.width();
            } else {
                dr.mDrawableSizeBottom = dr.mDrawableWidthBottom = 0;
            }
        }
    }
}
