package com.xdja.permission;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;

import com.xdja.ckms.common.R;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by xwy on 2017/10/13.
 */

public class PermissionUtil {

    /**
     * 返回对应权限是否已经开启
     * @param context
     * @param permissions
     * @return
     */
    public static List<OpenPowerBean> getPermissionStateList(@NonNull Context context, @NonNull List<String> permissions){
        List<OpenPowerBean> openPowerBeanList = new ArrayList<>();
        if (permissions == null || permissions.size() == 0){
            return openPowerBeanList;
        }
        for (String p:permissions) {
            OpenPowerBean openPowerBean = new OpenPowerBean();
            if (checkPermission(context, p)) {
                //此时在系统检查没有此权限
                openPowerBean.setPowerName(permissonTransverter(context,p));
                openPowerBean.setResId(R.drawable.ic_no);
            }
            else {
                openPowerBean.setPowerName(permissonTransverter(context,p));
                openPowerBean.setResId(R.drawable.ic_yes);
            }
            openPowerBeanList.add(openPowerBean);
        }
        return openPowerBeanList;
    }
    private static String permissonTransverter(Context context,String permisson){
        String permissonName = context.getResources().getString(R.string.permission);
        if(permisson != null){
            switch (permisson){
                case Manifest.permission.READ_SMS:
                    permissonName =  context.getResources().getString(R.string.read_sms);
                    break;
                case Manifest.permission.SEND_SMS:
                    permissonName =  context.getResources().getString(R.string.send_sms);
                    break;
                case Manifest.permission.READ_CONTACTS:
                    permissonName =  context.getResources().getString(R.string.read_contacts);
                    break;
                case Manifest.permission.RECEIVE_SMS:
                    permissonName =  context.getResources().getString(R.string.receive_sms);
                    break;
                case Manifest.permission.WRITE_CONTACTS:
                    permissonName =  context.getResources().getString(R.string.modify_contacts);
                    break;
                case Manifest.permission.READ_PHONE_STATE:
                    permissonName =  context.getResources().getString(R.string.read_phone_state);
                    break;
                default:
                    break;
            }
        }

        return permissonName;
    }

    /**
     * 返回对应权限是否已经开启
     * @param context
     * @param permissions
     * @return
     */
    public static Map<String, Boolean> getPermissionState(@NonNull Context context, @NonNull List<String> permissions){
        if (permissions == null || permissions.size() == 0){
            return new HashMap<>();
        }
        Map<String,Boolean> permissionsState = new HashMap<>();
        for (String p:permissions) {
            if (checkPermission(context, p)) {
                //此时在系统检查没有此权限
                permissionsState.put(p,false);
            }
            else {
                permissionsState.put(p,true);
            }
        }
        return permissionsState;
    }


    /**
     * 获取请求权限中需要授权的权限
     */
    public static List<String> getDeniedPermissions(@NonNull Context context, @NonNull List<String> permissions) {
        List<String> deniedPermissions = new ArrayList<>();
        for (String permission : permissions) {
            if (checkPermission(context, permission)) {
                //此时在系统检查没有此权限
                deniedPermissions.add(permission);
            }
        }
        return deniedPermissions;
    }

    public static boolean checkPermission(Context context, String permission) {
        return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_DENIED;
    }

    public static boolean isPermissionGranted(Context context, String permission) {
        return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED;
    }

    /**
     * 请求所有权限
     *
     * @param context
     * @param permissionBean
     * @param permissionListener
     */
    public static void requestPermissions(@NonNull Context context, @NonNull PermissionBean permissionBean,
                                          @NonNull OnPermissionListener permissionListener) {

        //1.版本小于23，直接返回成功
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            permissionListener.onPermissionGranted();
            return;
        }
        //2将PermissionBean中的权限列表过滤掉已经授权的
        permissionBean.setPermissions(getDeniedPermissions(context, permissionBean.getPermissions()));
        //3.若全部授权则返回
        if (permissionBean.getPermissions() == null || permissionBean.getPermissions().size() == 0) {
            permissionListener.onPermissionGranted();
            return;
        }

        //3.判断是否是Activity
        if (context instanceof Activity) {
            requestPermission(context, permissionBean.getPermissions().get(0), permissionBean.getRequestCode());
        } else {
            throw new IllegalArgumentException("Context must be an Activity");
        }
    }

    private static void requestPermission(Context context, String permission, int requestCode) {
        if (permission == null) {
            return;
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            String[] p = {permission};
            ((Activity) context).requestPermissions(p, requestCode);
        }
    }

    public static void onRequestPermissionsResult(@NonNull Context context, int requestCode, @NonNull OnPermissionListener permissionListener,
                                                  @NonNull PermissionBean permissionBean) {
        if (requestCode == permissionBean.getRequestCode()) {

            if (permissionBean.getPermissions() == null || permissionBean.getPermissions().size() == 0) {
                permissionListener.onPermissionGranted();
                return;
            }

            if (checkPermission(context, permissionBean.getPermissions().get(0))) {
                boolean alwaysDenied = hasAlwaysDeniedPermission(context, permissionBean.getPermissions().get(0));
                if (alwaysDenied) {
                    //请求被永远拒绝
                    permissionListener.onPermissionDeniedAlways(permissionBean.getPermissions().get(0));
                    return;
                }
                //请求被拒绝
                permissionListener.onPermissionDenied(permissionBean.getPermissions().get(0));
            } else {
                //请求通过
                permissionBean.getPermissions().remove(permissionBean.getPermissions().get(0));
                if (permissionBean.getPermissions() == null || permissionBean.getPermissions().size() == 0) {
                    permissionListener.onPermissionGranted();
                    return;
                } else {
                    requestPermission(context, permissionBean.getPermissions().get(0), permissionBean.getRequestCode());
                }
            }
        }
    }

    /**
     * 是否彻底拒绝了某项权限
     */
    public static boolean hasAlwaysDeniedPermission(@NonNull Context context, @NonNull String... deniedPermissions) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) return false;
        boolean rationale;
        for (String permission : deniedPermissions) {
            rationale = ActivityCompat.shouldShowRequestPermissionRationale((Activity) context, permission);
            if (!rationale) return true;
        }
        return false;
    }


    public interface OnPermissionListener {
        /**
         * 请求通过
         */
        void onPermissionGranted();

        /**
         * 请求被拒绝
         *
         * @param deniedPermissions
         */
        void onPermissionDenied(String deniedPermissions);

        /**
         * 请求被永远拒绝
         *
         * @param deniedPermissions
         */
        void onPermissionDeniedAlways(String deniedPermissions);
    }

}
