package com.xdja.multichip.jar;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.os.Binder;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Pair;

import com.xdja.jar.forcallprovider.ForCallProvider;
import com.xdja.multichip.IMultiChip;
import com.xdja.multichip.ISkfChip;
import com.xdja.multichip.param.ChipParam;
import com.xdja.multichip.param.SuperChip;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by xwy on 2017/6/6.
 */

public class MultiChipJar {

    private static final String KEY_STRING_BINDER = "Binder";
    private static final String KEY_PARCELABLE_CHIPPARAM = "ChipParam";

    private final static int OK = 0;
    /**
     * -60001: 参数错误
     */
    public final static int PARAM_ERROR = -60001;
    /**
     * -60002: 获取多芯片管理主进程Binder为空
     */
    private final static int MULTI_BINDER_ERROR = -60002;
    /**
     * -60003: 国密接口中参数为空
     */
    public final static int SKF_PARAM_ERROR = -60003;
    /**
     * -60004: 根据信息获取对应的Binder不存在
     */
    public final static int BINDER_NULL_ERROR = -60004;


    /**
     * 获取设备上所有的芯片信息<br/>
     *
     * @param context
     * @return 其中：Pair.first 是错误码  {@link #PARAM_ERROR }{@link #BINDER_NULL_ERROR}
     * Pair.second 是具体的芯片信息列表<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public static Pair<Integer, List<ChipParam>> getAllChip(Context context) {
        if (context == null) {
            return Pair.create(PARAM_ERROR, null);
        }

        Pair<Integer, List<Bundle>> bundles = getChipInfoBundle(context);
        List<ChipParam> chipParamList = new ArrayList<ChipParam>();

        if (bundles.first != 0) {
            return Pair.create(bundles.first, null);
        }

        for (Bundle bundle : bundles.second) {
            bundle.setClassLoader(ChipParam.class.getClassLoader());
            ChipParam chipParam = bundle.getParcelable(KEY_PARCELABLE_CHIPPARAM);
            chipParamList.add(chipParam);
        }

        return Pair.create(OK, chipParamList);
    }

    /**
     * 获取设备上指定类型的芯片信息<br/>
     *
     * @param context
     * @param chipType 芯片类型
     * @return 其中：Pair.first 是错误码
     * Pair.second 是具体的芯片信息列表<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public static Pair<Integer, List<ChipParam>> getChip(Context context, int chipType) {
        if (context == null || (chipType != ChipParam.TYPE_ONBOARD && chipType != ChipParam.TYPE_TF
                && chipType != ChipParam.TYPE_BLUETOOTH)) {
            return Pair.create(PARAM_ERROR, null);
        }

        Pair<Integer, List<ChipParam>> pair = getAllChip(context);
        if (pair.first != OK) {
            return pair;
        }
        List<ChipParam> chipParamList = pair.second;
        List<ChipParam> chipParamListInType = new ArrayList<>();

        for (ChipParam chipParam : chipParamList) {
            if (chipType == chipParam.chipType) {
                chipParamListInType.add(chipParam);
            }
        }
        return Pair.create(OK, chipParamListInType);
    }

    /**
     * 根据芯片信息获取对应的卡操作Binder<br/>
     *
     * @param context
     * @param chipParam
     * @return 其中：Pair.first 是错误码
     * Pair.second 是对应的操作服务Binder列表<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    private static Pair<Integer, ISkfChip> getSfkChipBinder(Context context, ChipParam chipParam) {
        if (context == null || chipParam == null) {
            return Pair.create(PARAM_ERROR, null);
        }

        Pair<Integer, List<Bundle>> bundles = getChipInfoBundle(context);
        if (bundles.first != 0) {
            return Pair.create(bundles.first, null);
        }

        for (Bundle bundle : bundles.second) {
            bundle.setClassLoader(ChipParam.class.getClassLoader());
            ChipParam chipParamTmp = bundle.getParcelable(KEY_PARCELABLE_CHIPPARAM);
            //此处用芯片唯一标示和公司信息做芯片标识
            if (chipParamTmp != null && chipParamTmp.uniqueID.equals(chipParam.uniqueID)
                    && chipParamTmp.manufacturer.equals(chipParam.manufacturer)) {
                IBinder binder = bundle.getBinder(KEY_STRING_BINDER);
                ISkfChip skfChip = null;
                if (binder != null) {
                    skfChip = ISkfChip.Stub.asInterface(binder);
                    return Pair.create(OK, skfChip);
                }
            }
        }
        return Pair.create(BINDER_NULL_ERROR, null);

    }

    /**
     * 根据芯片信息获取相应卡操作类
     *
     * @param context
     * @param chipParam
     * @return 其中：Pair.first 是错误码
     * Pair.second 是相应卡操作类<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public static Pair<Integer, SuperChip> getSuperChip(Context context, ChipParam chipParam) {
        if (context == null || chipParam == null) {
            return Pair.create(PARAM_ERROR, null);
        }
        Pair<Integer, ISkfChip> skfBinderPair = getSfkChipBinder(context, chipParam);
        ISkfChip skfChipBinder = skfBinderPair.second;
        return Pair.create(skfBinderPair.first, new SuperChip.Builder().build(skfChipBinder));
    }

    /**
     * 获取芯片管家中提供的所有的存有芯片相关信息的Bundle
     *
     * @return
     */
    private static Pair<Integer, List<Bundle>> getChipInfoBundle(Context context) {
        IMultiChip multiChip = null;
        List<Bundle> bundles = null;
        try {
            multiChip = getBinder(context);
            if (multiChip == null) {
                return Pair.create(MULTI_BINDER_ERROR, null);
            }
            //此处即使没有任何芯片，getChipInfo()返回的是一个大小为0的list
            bundles = multiChip.getChipInfo();
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return Pair.create(OK, bundles);
    }

    /**
     * 注册监听TF卡变化的广播
     *
     * @param context
     * @param changeListener
     * @return
     */
    public static int registerKeyReceiver(Context context, MultiChipListener.KeyChangeListener changeListener) {
        if (context == null || changeListener == null) {
            return PARAM_ERROR;
        }
        KeyBroadcastReceiver broadcastReceiver = KeyBroadcastReceiver.getInstance(context);
        broadcastReceiver.registerKeyReceiver(context, changeListener);
        return OK;
    }

    /**
     * 取消注册监听TF卡变化的广播
     *
     * @param context
     * @param changeListener
     * @return
     */
    public static int unregisterKeyReceiver(Context context, MultiChipListener.KeyChangeListener changeListener) {
        if (context == null || changeListener == null) {
            return PARAM_ERROR;
        }
        KeyBroadcastReceiver broadcastReceiver = KeyBroadcastReceiver.getInstance(context);
        broadcastReceiver.unregisterKeyReceiver(context, changeListener);
        return OK;
    }


    private MultiChipJar() {
    }

    private static IMultiChip multiChip = null;

    /**
     * 获取卡管理主进程Binder
     * @param context
     * @return
     */
    private static IMultiChip getBinder(Context context) {
        ContentResolver contentResolver = context.getContentResolver();
        Uri uri = getUri(context);
        if (uri == null){
            return null;
        }
        Bundle bundle = contentResolver.call(uri, "GetBinder", "MultiChipProxyBinder", null);
        if (bundle == null) {
            return null;
        }
        IBinder binder = bundle.getBinder("Binder");
        if (binder != null) {
            multiChip = IMultiChip.Stub.asInterface(binder);
            try {
                multiChip.asBinder().linkToDeath(new IBinder.DeathRecipient() {
                    @Override
                    public void binderDied() {
                        multiChip = null;
                    }
                }, Binder.FLAG_ONEWAY);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }
        return multiChip;
    }

    /**
     * 得到Uri
     *
     * @param context
     * @return
     */
    private static Uri getUri(Context context) {
        return ForCallProvider.getGetProxyProviderUri(context);
    }
}
