package com.xdja.multichip.param;

import android.os.Binder;
import android.os.IBinder;
import android.util.Log;
import android.util.Pair;

import com.xdja.multichip.ISkfChip;
import com.xdja.multichip.jar.MultiChipJar;
import com.xdja.skfapi.BlockCipherParam;
import com.xdja.skfapi.DevInfo;
import com.xdja.skfapi.EccCipherBlob;
import com.xdja.skfapi.EccPrivateKeyBlob;
import com.xdja.skfapi.EccPublicKeyBlob;
import com.xdja.skfapi.EccSignatureBlob;
import com.xdja.skfapi.EnvelopedKeyBlob;
import com.xdja.skfapi.FileAttribute;
import com.xdja.skfapi.RsaPrivateKeyBlob;
import com.xdja.skfapi.RsaPublicKeyBlob;
import com.xdja.skfapi.SkfApiCode;

/**
 * Created by xwy on 2017/6/7.
 */

public class SuperChip {

    private ISkfChip skfChipBinder = null;


    /***************国密接口开始********************/
    //-------------------------设备管理系列函数--------------------------------------

    /**
     * 等待设备插拔事件<br>
     * 功能描述：该函数等待设备插入或者拔除事件。szDevName 返回发生事件的设备名称。
     *
     * @param szDevName     [OUT]发生事件的设备名称。
     * @param pulDevNameLen [IN,OUT]输入/输出参数，当输入时表示szDevName的最大长度，输出时表示szDevName的有效长度,长度包含字符串结束符。作为OUT时，pulDevNameLen[0]为结果。
     * @param pulEvent      [OUT]事件类型。1 表示插入，2 表示拔出。pulEvent[0]为结果。
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    /*public int WaitForDevEvent(byte[] szDevName, int[] pulDevNameLen, int[] pulEvent) {
        int ret;
        try {
            ret = skfChipBinder.WaitForDevEvent(szDevName, pulDevNameLen, pulEvent);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }*/

    /**
     * 取消等待设备插拔事件
     * 功能描述：该方法取消等待设备插入或者拔除事件。
     *
     * @return 0：正确；其他：参看错误码定义。
     */
    /*public int CancelWaitForDevEvent() {
        int ret;
        try {
            ret = skfChipBinder.CancelWaitForDevEvent();
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }*/

    /**
     * 枚举设备<br>
     * 功能描述：获得当前系统中的设备列表。
     *
     * @param bPresent   [IN] 为1表示取当前设备状态为存在的设备列表。为0表示取当前驱动支持的设备列表。
     * @param szNameList [OUT]设备列表名称。如果参数为NULL，将由pulSize返回所需要的空间大小。每个设备的名称以单个'\0'结束，以双'\0'表示列表结束。
     * @param pulSize    [IN,OUT]输入时表示设备名称列表的长度,输出时表示szNameList所占用的空间大小。pulSize[0]为结果。
     * @return 0：正确；其他：参看错误码定义。
     */
   /* public int EnumDev(int bPresent, byte[] szNameList, int[] pulSize) {
        int ret;
        if (szNameList == null) {
            szNameList = new byte[0];
        }
        try {
            ret = skfChipBinder.EnumDev(bPresent, szNameList, pulSize);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }*/

    /**
     * 连接设备<br>
     * 功能描述：通过设备名连接设备，返回设备的句柄。
     *
     * @param szName [IN]设备名称。
     * @param phDdev [OUT]  返回设备操作句柄。phDdev[0]为结果。
     * @return 0：正确；其他：参看错误码定义。
     */
    /*public int ConnectDev(byte[] szName, long[] phDdev) {
        int ret;
        try {
            ret = skfChipBinder.ConnectDev(szName, phDdev);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }*/

    /**
     * 断开连接<br>
     * 功能描述：断开一个已经连接的设备，并释放句柄。<br>
     * 备注：如果该设备已被锁定，函数应首先解锁该设备。断开连接操作并不影响设备的权限状态。
     *
     * @param hDev [IN] 连接设备时返回的设备句柄。
     * @return 0：正确；其他：参看错误码定义。
     */
    /*public int DisConnectDev(long hDev) {
        int ret;
        try {
            ret = skfChipBinder.DisConnectDev(hDev);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }*/

    /**
     * 获取设备状态<br>
     * 功能描述：获取设备是否存在的状态。
     *
     * @param szDevName [IN] 设备名称。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是设备状态<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    /*public Pair<Integer, Integer> GetDevState(byte[] szDevName) {
        if (isEmpty(szDevName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }

        int ret;
        int[] pulDevState = new int[1];
        try {
            ret = skfChipBinder.GetDevState(szDevName, pulDevState);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pulDevState[0]);
    }*/

    /**
     * 设置设备标签<br>
     * 功能描述：设置设备标签。
     *
     * @param szLabel [IN] 设备标签字符串。该字符串应小于 32 字节。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int SetLabel(byte[] szLabel) {
        if (isEmpty(szLabel)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        try {
            ret = skfChipBinder.SetLabel(hDev, szLabel);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 获取设备信息<br>
     * 功能描述：获取设备的一些特征信息，包括设备标签、厂商信息、支持的算法等。
     *
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的设备信息<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, DevInfo> GetDevInfo() {
        int ret;
        long hDev = 0;
        DevInfo pDevInfo = new DevInfo();
        try {
            ret = skfChipBinder.GetDevInfo(hDev, pDevInfo);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pDevInfo);
    }

    /**
     * 锁定设备<br>
     * 功能描述：获得设备的独占使用权。
     *
     * @param ulTimeOut [IN] 超时时间，单位为毫秒。如果为 0xFFFFFFFF 表示无限等待。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int LockDev(int ulTimeOut) {
        int ret;
        long hDev = 0;
        try {
            ret = skfChipBinder.LockDev(hDev, ulTimeOut);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 解锁设备<br>
     * 功能描述：释放对设备的独占使用权。
     *
     * @return 0：正确；其他：参看错误码定义。
     */
    public int UnlockDev() {
        int ret;
        long hDev = 0;
        try {
            ret = skfChipBinder.UnlockDev(hDev);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 设备命令传输
     * 将命令直接发送给设备，并返回结果。
     *
     * @param pbCommand    [IN] 设备命令。
     * @param ulCommandLen [IN] 命令长度。
     * @param pbData       [OUT]返回结果数据。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int Transmit(byte[] pbCommand, int ulCommandLen, byte[] pbData) {
        if (isEmpty(pbCommand, pbData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }

        int ret;
        long hDev = 0;
        int[] pulDataLen = {pbData.length};
        try {
            ret = skfChipBinder.Transmit(hDev, pbCommand, ulCommandLen, pbData, pulDataLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }


    //-------------------------访问控制系列函数--------------------------------------

    /**
     * 修改设备认证密钥<br>
     * 权限要求：设备认证成功后才能使用。
     *
     * @param pbKeyValue [IN]密钥值。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ChangeDevAuthKey(byte[] pbKeyValue) {
        if (isEmpty(pbKeyValue)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        int ulKeyLen = pbKeyValue.length;
        try {
            ret = skfChipBinder.ChangeDevAuthKey(hDev, pbKeyValue, ulKeyLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 设备认证<br>
     * 认证过程：被认证方获取8字节随机数RND,填充0至16字节得到D0,加密得到结果D1,设备收到D1验证是否正确。
     *
     * @param pbAuthData [IN]认证数据。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int DevAuth(byte[] pbAuthData) {
        if (isEmpty(pbAuthData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        int ulLen = pbAuthData.length;
        try {
            ret = skfChipBinder.DevAuth(hDev, pbAuthData, ulLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 修改PIN<br>
     * 功能描述：调用该函数可以修改Administrator PIN和User PIN。<br>
     *
     * @param hApplication [IN]应用句柄。
     * @param ulPINType    [IN]PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @param szOldPin     [IN]原PIN值。
     * @param szNewPin     [IN]新PIN值。
     * @return 其中：Pair.first 是错误码。0x0A000024:原PIN码不正确；0x0A000025：原PIN码被锁死；
     * 0x0A000026：原PIN码无效；0x0A000027：原PIN码长度错误等。
     * Pair.second 是PIN码验证不正确后返回的重试次数。<br/>
     * <p>
     * 只有当fist为0x0A000024的时候,second才有意义。
     */

    public Pair<Integer, Integer> ChangePIN(long hApplication, int ulPINType,
                                            byte[] szOldPin, byte[] szNewPin) {
        if (isEmpty(szOldPin, szNewPin)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int[] pulRetryCount = new int[1];
        try {
            ret = skfChipBinder.ChangePIN(hApplication, ulPINType, szOldPin, szNewPin, pulRetryCount);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pulRetryCount[0]);
    }

    /**
     * 获取PIN信息<br>
     * 功能描述：获取PIN码信息，包括最大重试次数、当前剩余重试次数，以及当前PIN码是否为出厂默认PIN码。
     *
     * @param hApplication [IN]应用句柄。
     * @param ulPINType    [IN]PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的结果Bean。<br/>
     * Bean.result[0]表示最大重试次数。注：信大捷安的卡此参数无效。<br/>
     * Bean.result[1]表示当前剩余重试次数，当为0时表示已锁死。<br/>
     * Bean.result[2]是否为出厂默认PIN码。注：信大捷安的卡此参数无效。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, ResultBean<Integer[]>> GetPINInfo(long hApplication, int ulPINType) {
        int ret;
        int[] pulMaxRetryCount = new int[1];
        int[] pulRemainRetryCount = new int[1];
        int[] pbDefaultPin = new int[1];
        ResultBean<Integer[]> resultBean = new ResultBean<>();
        try {
            ret = skfChipBinder.GetPINInfo(hApplication, ulPINType, pulMaxRetryCount,
                    pulRemainRetryCount, pbDefaultPin);
            resultBean.result = new Integer[3];
            resultBean.result[0] = pulMaxRetryCount[0];
            resultBean.result[1] = pulRemainRetryCount[0];
            resultBean.result[2] = pbDefaultPin[0];
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, resultBean);
    }

    /**
     * 校验PIN码<br>
     * 功能描述：校验成功后，会获得相应的权限，如果PIN码错误，会返回PIN码重试次数，当次数为0时表示PIN码已锁死。
     *
     * @param hApplication [IN]应用句柄。
     * @param ulPINType    [IN]PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @param szPIN        [IN]PIN码。
     * @return 其中：Pair.first 是错误码。0x0A000024:PIN码不正确；0x0A000025：PIN码被锁死；
     * 0x0A000026：PIN码无效；0x0A000027：PIN码长度错误等。
     * Pair.second 是PIN码验证不正确后返回的重试次数。<br/>
     * <p>
     * 只有当fist为0x0A000024的时候,second才有意义。
     */
    public Pair<Integer, Integer> VerifyPIN(long hApplication, int ulPINType, byte[] szPIN) {
        if (isEmpty(szPIN)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int[] pulRetryCount = new int[1];
        try {
            ret = skfChipBinder.VerifyPIN(hApplication, ulPINType, szPIN, pulRetryCount);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pulRetryCount[0]);
    }

    /**
     * 解锁PIN<br>
     * 功能描述：当用户的 PIN 码锁死后，通过调用该函数来解锁用户 PIN 码。 解锁后，用户 PIN 码被设置成新值，
     * 用户 PIN 码的重试次数也恢复到原值。<br>
     * <p>
     * 权限要求：验证完管理员 PIN 才能够解锁用户 PIN 码，如果输入的 Administrator PIN不正确或者已经锁死，
     * 会调用失败，并返回 Administrator PIN 的重试次数。
     *
     * @param hApplication [IN]应用句柄。
     * @param szAdminPIN   [IN]管理员 PIN 码。
     * @param szNewUserPIN [IN]新的用户 PIN 码。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是管理员 PIN 码错误时，返回剩余重试次数。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Integer> UnblockPIN(long hApplication, byte[] szAdminPIN, byte[] szNewUserPIN) {
        if (isEmpty(szAdminPIN, szNewUserPIN)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        //modified by xwy on 2017.7.11 数组长度不应该为0至少为1
        //int[] pulRetryCount = new int[0];
        int[] pulRetryCount = new int[1];
        try {
            ret = skfChipBinder.UnblockPIN(hApplication, szAdminPIN, szNewUserPIN, pulRetryCount);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pulRetryCount[0]);
    }

    /**
     * 清除应用当前的安全状态
     *
     * @param hApplication [IN]应用句柄。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ClearSecureState(long hApplication) {
        int ret;
        try {
            ret = skfChipBinder.ClearSecureState(hApplication);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    ///----------------------------应用管理系列函数-------------------------------

    /**
     * 创建应用<br>
     * 权限要求：需要设备权限
     *
     * @param szAppName            [IN]应用名称。
     * @param szAdminPin           [IN]管理员PIN。
     * @param dwAdminPinRetryCount [IN]管理员PIN最大重试次数。
     * @param szUserPin            [IN]用户PIN。
     * @param dwUserPinRetryCount  [IN]用户PIN最大重试次数。
     * @param dwCreateFileRights   [IN]在该应用下创建文件和容器的权限。
     *                             值可参看：<br>
     *                             {@link SkfApiCode#SECURE_NEVER_ACCOUNT},<br>
     *                             {@link SkfApiCode#SECURE_ADM_ACCOUNT},<br>
     *                             {@link SkfApiCode#SECURE_USER_ACCOUNT},<br>
     *                             {@link SkfApiCode#SECURE_ANYONE_ACCOUNT}</pre>
     * @return 其中：Pair.first 是错误码
     * Pair.second 是应用的句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> CreateApplication(byte[] szAppName, byte[] szAdminPin,
                                                 int dwAdminPinRetryCount, byte[] szUserPin,
                                                 int dwUserPinRetryCount, int dwCreateFileRights) {
        long hDev = 0;
        if (isEmpty(szAppName, szAdminPin, szUserPin)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] phApplication = new long[1];
        try {
            ret = skfChipBinder.CreateApplication(hDev, szAppName, szAdminPin,
                    dwAdminPinRetryCount, szUserPin, dwUserPinRetryCount, dwCreateFileRights, phApplication);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phApplication[0]);
    }

    /**
     * 枚举应用
     *
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回应用名称列表。每个应用以单个'\0'结束,以双'\0'表示列表结束。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> EnumApplication() {
        int ret;
        long hDev = 0;
        byte[] szAppName = null;
        int[] pulSize = new int[1];
        try {
            ret = skfChipBinder.EnumApplication(hDev, new byte[0], pulSize);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            szAppName = new byte[pulSize[0]];
            ret = skfChipBinder.EnumApplication(hDev, szAppName, pulSize);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, szAppName);
    }

    /**
     * 删除应用<br>
     * 权限要求：需要设备权限
     *
     * @param szAppName [IN]应用名称。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int DeleteApplication(byte[] szAppName) {
        if (isEmpty(szAppName)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;

        try {
            ret = skfChipBinder.DeleteApplication(hDev, szAppName);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 打开应用
     *
     * @param szAppName [IN]应用名称。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是应用的句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> OpenApplication(byte[] szAppName) {
        if (isEmpty(szAppName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;

        long[] phApplication = new long[1];
        try {
            ret = skfChipBinder.OpenApplication(hDev, szAppName, phApplication);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phApplication[0]);
    }


    /**
     * 关闭应用<br>
     * 备注：此函数不影响应用安全状态。
     *
     * @param hApplication [IN]应用句柄。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int CloseApplication(long hApplication) {
        int ret;
        try {
            ret = skfChipBinder.CloseApplication(hApplication);
        } catch (Exception e) {
            ret = handleException(e);
        }

        return ret;
    }


    ///-----------------------------文件管理系列函数---------------------------------

    /**
     * 创建文件<br>
     * 功能描述：创建文件时要指定文件的名称、大小，以及读写权限。<br>
     * 权限要求：创建文件需要应用指定的创建文件权限。
     *
     * @param hApplication  [IN]应用句柄。
     * @param szFileName    [IN]文件名称，长度不得大于32个字节。
     * @param ulFileSize    [IN]文件大小。
     * @param ulReadRights  [IN]文件读权限。
     *                      值可参看：<br>
     *                      {@link SkfApiCode#SECURE_NEVER_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_ADM_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_USER_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_ANYONE_ACCOUNT}
     * @param ulWriteRights [IN]文件写权限。
     *                      值可参看：<br>
     *                      {@link SkfApiCode#SECURE_NEVER_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_ADM_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_USER_ACCOUNT},<br>
     *                      {@link SkfApiCode#SECURE_ANYONE_ACCOUNT}
     * @return 0：正确；其他：参看错误码定义。
     */
    public int CreateFile(long hApplication, byte[] szFileName, int ulFileSize,
                          int ulReadRights, int ulWriteRights) {
        if (isEmpty(szFileName)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.CreateFile(hApplication, szFileName, ulFileSize, ulReadRights, ulWriteRights);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 删除文件<br>
     * 功能描述：文件删除后，文件中写入的所有信息将丢失。文件在设备中占用的空间将被释放。<br>
     * 权限要求：删除一个文件应具有对该文件的创建权限。
     *
     * @param hApplication [IN]要删除的文件所在的应用句柄。
     * @param szFileName   [IN]要删除的文件的名称。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int DeleteFile(long hApplication, byte[] szFileName) {
        if (isEmpty(szFileName)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.DeleteFile(hApplication, szFileName);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 枚举一个应用下存在的所有文件
     *
     * @param hApplication [IN]应用句柄。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回文件名称列表。每个文件名称以单个'\0'结束,以双'\0'表示列表结束。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> EnumFiles(long hApplication) {
        int ret;
        byte[] szFileList = null;
        int[] pulSize = new int[1];
        try {
            ret = skfChipBinder.EnumFiles(hApplication, new byte[0], pulSize);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            szFileList = new byte[pulSize[0]];
            ret = skfChipBinder.EnumFiles(hApplication, szFileList, pulSize);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, szFileList);
    }

    /**
     * 获取文件属性: 获取应用文件的属性信息，例如文件的大小、权限等。
     *
     * @param hApplication [IN]文件所在应用句柄。
     * @param szFileName   [IN]文件名称。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是文件信息。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, FileAttribute> GetFileInfo(long hApplication, byte[] szFileName) {
        if (isEmpty(szFileName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        FileAttribute pFileInfo = new FileAttribute();
        try {
            ret = skfChipBinder.GetFileInfo(hApplication, szFileName, pFileInfo);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pFileInfo);
    }

    /**
     * 读文件<br>
     * 功能描述：读取文件内容。<br>
     * 权限要求：须具备该文件的读权限。
     *
     * @param hApplication [IN]应用句柄。
     * @param szFileName   [IN]文件名称。
     * @param ulOfffset    [IN]文件读取偏移位置。
     * @param ulSize       [IN]要读取文件的长度。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ReadFile(long hApplication, byte[] szFileName, int ulOfffset, int ulSize) {
        if (isEmpty(szFileName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        byte[] pbOutData = new byte[ulSize];
        int[] pulOutLen = new int[1];
        try {
            ret = skfChipBinder.ReadFile(hApplication, szFileName, ulOfffset, ulSize, pbOutData, pulOutLen);
            if (pulOutLen[0] < ulSize) {
                byte[] pbOutDataTmp = new byte[pulOutLen[0]];
                System.arraycopy(pbOutData, 0, pbOutDataTmp, 0, pulOutLen[0]);
                pbOutData = pbOutDataTmp;
            }
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbOutData);
    }

    /**
     * 写文件<br>
     * 功能描述：写数据到文件中。<br>
     * 权限要求：须具备该文件的写权限。
     *
     * @param hApplication [IN]应用句柄。
     * @param szFileName   [IN]文件名称。
     * @param ulOfffset    [IN]写入文件的偏移量。
     * @param pbData       [IN]写入数据。
     * @param ulSize       [IN]写入数据的大小。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int WriteFile(long hApplication, byte[] szFileName, int ulOfffset, byte[] pbData, int ulSize) {
        if (isEmpty(szFileName, pbData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.WriteFile(hApplication, szFileName, ulOfffset, pbData, ulSize);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    ///-----------------------------容器管理系列函数----------------------------------

    /**
     * 创建容器<br>
     * 功能描述：在应用下建立指定名称的容器并返回容器句柄。<br>
     * 权限要求：需要用户权限
     *
     * @param hApplication    [IN]应用句柄。
     * @param szContainerName [IN]容器名称,ASCII字符串，最大长度不超过64字节。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是所建立容器的容器句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> CreateContainer(long hApplication, byte[] szContainerName) {
        if (isEmpty(szContainerName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] pbContainer = new long[1];
        try {
            ret = skfChipBinder.CreateContainer(hApplication, szContainerName, pbContainer);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbContainer[0]);
    }

    /**
     * 删除容器<br>
     * 功能描述：在应用下删除指定名称的容器并释放容器相关的资源。<br>
     * 权限要求：需要用户权限
     *
     * @param hApplication    [IN]应用句柄。
     * @param szContainerName [IN]删除容器的名称。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int DeleteContainer(long hApplication, byte[] szContainerName) {
        if (isEmpty(szContainerName)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.DeleteContainer(hApplication, szContainerName);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 枚举容器<br>
     * 功能描述：枚举应用下的所有容器并返回容器名称列表。
     *
     * @param hApplication [IN]应用句柄。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是容器名称列表，每个容器名以单个‘\0’为结束，列表以双‘\0’结束。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> EnumContainer(long hApplication) {
        int ret;
        byte[] szContainerName = null;
        int[] pulSize = new int[1];
        try {
            ret = skfChipBinder.EnumContainer(hApplication, new byte[0], pulSize);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            szContainerName = new byte[pulSize[0]];
            ret = skfChipBinder.EnumContainer(hApplication, szContainerName, pulSize);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, szContainerName);
    }

    /**
     * 打开容器<br>
     * 功能描述：获取容器句柄
     *
     * @param hApplication    [IN]应用句柄。
     * @param szContainerName [IN]容器名称。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回打开的的容器句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> OpenContainer(long hApplication, byte[] szContainerName) {
        if (isEmpty(szContainerName)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] pbContainer = new long[1];
        try {
            ret = skfChipBinder.OpenContainer(hApplication, szContainerName, pbContainer);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbContainer[0]);
    }

    /**
     * 关闭容器<br>
     * 功能描述：关闭容器句柄，并释放容器句柄相关资源。
     *
     * @param hContainer [IN]容器句柄。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int CloseContainer(long hContainer) {
        int ret;
        try {
            ret = skfChipBinder.CloseContainer(hContainer);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 获得容器类型
     *
     * @param hContainer [IN]容器句柄。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是容器类型，pulContainerType[0]为结果，当值为0表示未定、尚未分配类型或者空容器，
     * 为1表示RSA容器，为2表示SM2容器。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Integer> GetContainerType(long hContainer) {
        int ret;
        int[] pulContainerType = new int[1];
        try {
            ret = skfChipBinder.GetContainerType(hContainer, pulContainerType);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pulContainerType[0]);
    }

    /**
     * 导入数字证书<br>
     * 功能描述：向容器内导入数字证书
     *
     * @param hContainer [IN]容器句柄。
     * @param bSignFlag  [IN]{@link SkfApiCode#TRUE}表示签名证书，{@link SkfApiCode#FALSE}表示加密证书。
     * @param pbCert     [IN]证书内容。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ImportCertificate(long hContainer, int bSignFlag, byte[] pbCert) {
        if (isEmpty(pbCert)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        int ulCertLen = pbCert.length;
        try {
            ret = skfChipBinder.ImportCertificate(hContainer, bSignFlag, pbCert, ulCertLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 导出数字证书<br>
     * 功能描述：从容器内导出数字证书
     *
     * @param hContainer [IN]容器句柄。
     * @param bSignFlag  [IN]{@link SkfApiCode#TRUE}表示签名证书，{@link SkfApiCode#FALSE}表示加密证书。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是证书内容。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ExportCertificate(long hContainer, int bSignFlag) {
        int ret;
        byte[] pbCert = null;
        int[] pulCertLen = new int[1];
        try {
            ret = skfChipBinder.ExportCertificate(hContainer, bSignFlag, new byte[0], pulCertLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }

            Log.i("SuperChip","ExportCertificate len: " + pulCertLen[0]);

            pbCert = new byte[pulCertLen[0]];
            ret = skfChipBinder.ExportCertificate(hContainer, bSignFlag, pbCert, pulCertLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbCert);
    }


    ///------------------------------密码服务系列函数-----------------------------------

    /**
     * 生成随机数<br>
     * 功能描述：产生指定长度的随机数
     *
     * @param ulRandomLen [IN] 随机数长度。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的随机数。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public Pair<Integer, byte[]> GenRandom(int ulRandomLen) {
        int ret;
        long hDev = 0;
        byte[] pbRandom = new byte[ulRandomLen];
        try {
            ret = skfChipBinder.GenRandom(hDev, pbRandom, ulRandomLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbRandom);
    }

    /**
     * 生成外部RSA密钥对<br>
     * 由设备生成 RSA 密钥对并明文输出。<br>
     * 生成的私钥只用于输出，接口内不做保留和计算。
     *
     * @param ulBitsLen [IN]密钥模长。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的私钥数据结构。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public Pair<Integer, RsaPrivateKeyBlob> GenExtRSAKey(int ulBitsLen) {
        int ret;
        long hDev = 0;
        RsaPrivateKeyBlob pBlob = new RsaPrivateKeyBlob();
        try {
            ret = skfChipBinder.GenExtRSAKey(hDev, ulBitsLen, pBlob);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pBlob);
    }

    /**
     * 生成RSA签名密钥对<br>
     * 功能描述：生成RSA签名密钥对并输出签名公钥。<br>
     * 权限要求：须具备用户权限。
     *
     * @param hContainer [IN]容器句柄。
     * @param ulBitsLen  [IN] 密钥模长。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的RSA公钥数据结构。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public Pair<Integer, RsaPublicKeyBlob> GenRSAKeyPair(long hContainer, int ulBitsLen) {
        int ret;
        RsaPublicKeyBlob pBlob = new RsaPublicKeyBlob();
        try {
            ret = skfChipBinder.GenRSAKeyPair(hContainer, ulBitsLen, pBlob);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pBlob);
    }


    /**
     * 导入RSA加密密钥对<br>
     * 功能描述：导入RSA加密公私钥对<br>
     * 权限要求：须具备用户权限。
     *
     * @param hContainer      [IN] 容器句柄。
     * @param ulSymAlgId      [IN] 对称算法密钥标识。
     * @param pbWrappedKey    [IN] 使用该容器内签名公钥保护的对称算法密钥。
     * @param pbEncryptedData [IN] 对称算法密钥保护的 RSA 加密私钥。私钥的格式遵循 PKCS#1 v2.1:
     *                        RSA Cryptography Standard 中的私钥格式定义。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ImportRSAKeyPair(long hContainer, int ulSymAlgId, byte[] pbWrappedKey, byte[] pbEncryptedData) {
        if (isEmpty(pbWrappedKey, pbEncryptedData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        int ulWrappedKeyLen = pbWrappedKey.length;
        int ulEncryptedDataLen = pbEncryptedData.length;
        try {
            ret = skfChipBinder.ImportRSAKeyPair(hContainer, ulSymAlgId, pbWrappedKey,
                    ulWrappedKeyLen, pbEncryptedData, ulEncryptedDataLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * RSA签名<br>
     * 功能描述：使用hContainer指定容器的签名私钥，对指定数据 pbData 进行数字签名。
     * 签名后的结果存放到 pbSignature ，设置 pulSignLen 为签名的长度。<br>
     * <p>
     * 权限要求：须具备用户权限。
     *
     * @param hContainer [IN] 用来签名的私钥所在容器句柄。
     * @param pbData     [IN] 被签名的数据,签名数据长度，应不大于 RSA 密钥模长-11。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是存放签名结果。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> RSASignData(long hContainer, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;
        byte[] pbSignatureTmp = new byte[245];
        int[] pulSignLen = new int[1];
        byte[] pbSignature = null;
        try {
            ret = skfChipBinder.RSASignData(hContainer, pbData, ulDataLen, pbSignatureTmp, pulSignLen);
            pbSignature = new byte[pulSignLen[0]];
            System.arraycopy(pbSignatureTmp, 0, pbSignature, 0, pulSignLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbSignature);
    }

    /**
     * RSA验签<br>
     * 功能描述：验证RSA签名。用 pRSAPubKeyBlob 内的公钥值对待验签数据进行验签。
     *
     * @param pRSAPubKeyBlob [IN]公钥数据结构。
     * @param pbData         [IN]待验证签名的数据。
     * @param pbSignature    [IN]待验证的签名值。签名值长度，必须为公钥模长。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int RSAVerify(RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbData,
                         byte[] pbSignature) {
        if (isEmpty(pRSAPubKeyBlob, pbData, pbSignature)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        int ulDataLen = pbData.length;
        int ulSignLen = pbSignature.length;
        try {
            ret = skfChipBinder.RSAVerify(hDev, pRSAPubKeyBlob, pbData, ulDataLen, pbSignature, ulSignLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * RSA生成并导出会话密钥<br>
     * 功能描述：生成会话密钥并用外部公钥加密输出。
     *
     * @param hContainer [IN] 容器句柄。
     * @param ulAlgId    [IN] 会话密钥算法标识。
     * @param pPubKey    [IN] 加密会话密钥的 RSA 公钥数据结构。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的Bean对象。其中：<br/>
     * Bean.resule为导出的加密会话密钥密文，按照 PKCS#1v1.5 要求封装。<br/>
     * Bean.retHandle为导出的密钥句柄<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, ResultBean<byte[]>> RSAExportSessionKey(long hContainer, int ulAlgId,
                                                                 RsaPublicKeyBlob pPubKey) {
        if (isEmpty(pPubKey)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        //start:modified by xwy on 2017-7-11 10:06:35 RSA公钥运算结果长度为模长
        /*byte[] pbData = new byte[16];
        int[] pulDataLen = {pbData.length};*/
        byte[] pbData = new byte[pPubKey.bitLen / 8];//todo
        int[] pulDataLen = new int[1];
        //end
        long[] phSessionKey = new long[1];
        ResultBean<byte[]> resultBean = new ResultBean<>();
        try {
            ret = skfChipBinder.RSAExportSessionKey(hContainer, ulAlgId, pPubKey, pbData, pulDataLen, phSessionKey);
            resultBean.result = pbData;
            resultBean.retHandle = phSessionKey[0];
            /*pbData = new byte[pulDataLen[0]];
            System.arraycopy(pbDataTmp, 0, pbData, 0, pulDataLen[0]);*/
        } catch (Exception e) {
            ret = handleException(e);
        }
        
        return Pair.create(ret, resultBean);
    }

    /**
     * RSA外来公钥运算<br>
     * 功能描述：使用外部传入的 RSA 公钥对输入数据做公钥运算并输出结果。
     *
     * @param pRSAPubKeyBlob [IN] RSA 公钥数据结构。
     * @param pbInput        [IN]待运算的原始数据,长度必须为公钥模长。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是RSA 公钥运算结果。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ExtRSAPubKeyOperation(RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbInput) {
        if (isEmpty(pRSAPubKeyBlob, pbInput)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;
        //pRSAPubKeyBlob.bitLen为模长的实际位数，必须除以8才是实际的模长
        byte[] pbOutput = new byte[pRSAPubKeyBlob.bitLen / 8];
        int[] pulOutputLen = new int[1];
        try {
            ret = skfChipBinder.ExtRSAPubKeyOperation(hDev, pRSAPubKeyBlob, pbInput, pbInput.length,
                    pbOutput, pulOutputLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbOutput);
    }

    /**
     * RSA外来私钥运算<br>
     * 功能描述：直接使用外部传入的 RSA 私钥对输入数据做私钥运算并输出结果。
     *
     * @param pRSAPriKeyBlob [IN]RSA私钥数据结构,需要填充公钥信息。
     * @param pbInput        [IN]待运算数据。长度，必须为公钥模长。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是RSA私钥运算结果。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ExtRSAPriKeyOperation(RsaPrivateKeyBlob pRSAPriKeyBlob,
                                                       byte[] pbInput) {
        if (isEmpty(pRSAPriKeyBlob, pbInput)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;
        int ulInputLen = pbInput.length;
        byte[] pbOutputTmp = new byte[pRSAPriKeyBlob.bitLen];
        int[] pulOutputLen = new int[1];
        byte[] pbOutput = null;
        try {
            ret = skfChipBinder.ExtRSAPriKeyOperation(hDev, pRSAPriKeyBlob, pbInput, ulInputLen, pbOutputTmp, pulOutputLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbOutput = new byte[pulOutputLen[0]];
            System.arraycopy(pbOutputTmp, 0, pbOutput, 0, pulOutputLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbOutput);
    }

    /**
     * 生成ECC签名密钥对<br>
     * 功能描述：生成ECC签名密钥对并输出签名公钥<br>
     * 权限要求：需要用户权限
     *
     * @param hContainer [IN]密钥容器句柄。
     * @param ulAlgId    [IN]算法标识，只支持SGD_SM2_1算法。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的ECC公钥数据结构。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, EccPublicKeyBlob> GenECCKeyPair(long hContainer, int ulAlgId) {
        int ret;
        EccPublicKeyBlob pBlob = new EccPublicKeyBlob();
        try {
            ret = skfChipBinder.GenECCKeyPair(hContainer, ulAlgId, pBlob);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pBlob);
    }

    /**
     * 导入ECC公私钥对<br>
     * 权限要求：需要用户权限
     *
     * @param hContainer        [IN]密钥容器句柄。
     * @param pEenvlopedKeyBlob [IN]受保护的加密密钥对。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ImportECCKeyPair(long hContainer, EnvelopedKeyBlob pEenvlopedKeyBlob) {
        if (isEmpty(pEenvlopedKeyBlob)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.ImportECCKeyPair(hContainer, pEenvlopedKeyBlob);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * ECC签名<br>
     * 功能描述：采用ECC算法和指定容器hContainer，对指定数据pbData进行数字签名，签名后的结果存放到pSignature中。<br>
     * 权限要求：需要用户权限。
     * 备注：输入数据为待签数据的杂凑值,当使用SM2算法时,该输入数据为待签名数据经过SM2签名预处理的结果，预处理过程遵循GM/T0009。<br>
     *
     * @param hContainer [IN]密钥容器句柄。
     * @param pbData     [IN]待签名的数据,长度必须小于密钥模长。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的签名值。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, EccSignatureBlob> ECCSignData(long hContainer, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;
        EccSignatureBlob pSignature = new EccSignatureBlob();
        try {
            ret = skfChipBinder.ECCSignData(hContainer, pbData, ulDataLen, pSignature);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pSignature);
    }

    /**
     * ECC验签<br>
     * 功能描述：用ECC公钥对数据进行验签。<br>
     * 备注：输入数据为待签数据的杂凑值,当使用SM2算法时,该输入数据为待签名数据经过SM2签名预处理的结果，预处理过程遵循GM/T0009。
     *
     * @param pECCPubKeyBlob [IN]ECC公钥数据结构。
     * @param pbData         [IN]待验证签名的数据。
     * @param pSignature     [IN]待验证签名值。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ECCVerify(EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData,
                         EccSignatureBlob pSignature) {
        if (isEmpty(pECCPubKeyBlob, pbData, pSignature)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        int ulDataLen = pbData.length;
        try {
            ret = skfChipBinder.ECCVerify(hDev, pECCPubKeyBlob, pbData, ulDataLen, pSignature);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * ECC生成并导出会话密钥<br>
     * 功能描述：生成会话密钥并用外部公钥加密导出。
     *
     * @param hContainer [IN]容器句柄。
     * @param ulAlgId    [IN]会话密钥算法标识。
     * @param pPubKey    [IN]外部输入的公钥结构。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的Bean对象。其中：<br/>
     * Bean.resule为会话密钥密文。<br/>
     * Bean.retHandle为会话密钥句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义
     */
    public Pair<Integer, ResultBean<EccCipherBlob>> ECCExportSessionKey(long hContainer, int ulAlgId,
                                                                        EccPublicKeyBlob pPubKey) {
        if (isEmpty(pPubKey)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        EccCipherBlob pData = new EccCipherBlob();
        long[] phSessionKey = new long[1];
        ResultBean<EccCipherBlob> resultBean = null;
        try {
            ret = skfChipBinder.ECCExportSessionKey(hContainer, ulAlgId, pPubKey, pData, phSessionKey);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            resultBean = new ResultBean<>();
            resultBean.result = pData;
            resultBean.retHandle = phSessionKey[0];
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, resultBean);
    }

    /**
     * ECC外来公钥加密<br>
     * 功能描述：使用外部传入的ECC公钥对输入数据做加密运算并输出结果。
     *
     * @param pECCPubkeyBlob [IN]ECC公钥数据结构。
     * @param pbPlainText    [IN]待加密的明文数据。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是密文数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, EccCipherBlob> ExtECCEncrypt(EccPublicKeyBlob pECCPubkeyBlob,
                                                      byte[] pbPlainText) {
        if (isEmpty(pECCPubkeyBlob, pbPlainText)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;
        int ulPlainTextLen = pbPlainText.length;
        EccCipherBlob pCipherText = new EccCipherBlob();
        try {
            ret = skfChipBinder.ExtECCEncrypt(hDev, pECCPubkeyBlob, pbPlainText, ulPlainTextLen, pCipherText);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pCipherText);
    }

    /**
     * ECC外来私钥解密<br>
     * 功能描述:使用外部传入的ECC私钥对输入数据做解密运算并输出结果。
     *
     * @param pECCPriKeyBlob [IN]ECC私钥数据结构。
     * @param pCipherText    [IN]待解密的密文数据。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的明文数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ExtECCDecrypt(EccPrivateKeyBlob pECCPriKeyBlob, EccCipherBlob pCipherText) {
        if (isEmpty(pECCPriKeyBlob, pCipherText)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;
        byte[] pbPlainTextTmp = new byte[pCipherText.cipherLen];
        int[] pulPlainTextLen = new int[1];
        byte[] pbPlainText = null;

        try {
            ret = skfChipBinder.ExtECCDecrypt(hDev, pECCPriKeyBlob, pCipherText, pbPlainTextTmp, pulPlainTextLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbPlainText = new byte[pulPlainTextLen[0]];
            System.arraycopy(pbPlainTextTmp, 0, pbPlainText, 0, pulPlainTextLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbPlainText);
    }

    /**
     * ECC外来私钥签名<br>
     * 功能描述：使用外部传入的ECC私钥对输入数据做签名运算并输出结果。
     *
     * @param pECCPriKeyBlob [IN]ECC私钥数据结构。
     * @param pbData         [IN]待签名数据。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是签名值。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, EccSignatureBlob> ExtECCSign(EccPrivateKeyBlob pECCPriKeyBlob, byte[] pbData) {
        if (isEmpty(pECCPriKeyBlob, pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long hDev = 0;
        int ulDataLen = pbData.length;
        EccSignatureBlob pSignature = new EccSignatureBlob();
        try {
            ret = skfChipBinder.ExtECCSign(hDev, pECCPriKeyBlob, pbData, ulDataLen, pSignature);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pSignature);
    }

    /**
     * ECC外来公钥验签<br>
     * 功能描述：使用外部传入的ECC公钥做签名验证。
     *
     * @param pECCPubKeyBlob [IN]ECC公钥数据结构。
     * @param pbData         [IN]待验证数据。
     * @param pSignature     [IN]签名值。
     * @return 0：正确；其他：参看错误码定义。
     */
    public int ExtECCVerify(EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData, EccSignatureBlob pSignature) {
        if (isEmpty(pECCPubKeyBlob, pbData, pSignature)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        long hDev = 0;
        int ulDataLen = pbData.length;
        try {
            ret = skfChipBinder.ExtECCVerify(hDev, pECCPubKeyBlob, pbData, ulDataLen, pSignature);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * ECC生成密钥协商参数并输出<br>
     * 功能描述：使用ECC密钥协商算法，为计算会话密钥而产生协商参数，返回临时ECC密钥对的公钥及协商句柄。<br>
     * 备注：为协商会话密钥，协商的发起方应首先调用本函数
     *
     * @param hContainer [IN]容器句柄。
     * @param ulAlgId    [IN]会话密钥算法标识。
     * @param pbID       [IN]发起方的ID。长度不大于32。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的Bean对象。其中：<br/>
     * Bean.resule为发起方临时ECC公钥。<br/>
     * Bean.retHandle为密钥协商句柄。<br/>
     */
    public Pair<Integer, ResultBean<EccPublicKeyBlob>> GenerateAgreementDataWithECC(long hContainer, int ulAlgId, byte[] pbID) {
        if (isEmpty(pbID)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulIDLen = pbID.length;
        EccPublicKeyBlob pTempECCPubKeyBlob = new EccPublicKeyBlob();
        long[] phAgreementHandle = new long[1];
        ResultBean<EccPublicKeyBlob> resultBean = null;
        try {
            ret = skfChipBinder.GenerateAgreementDataWithECC(hContainer, ulAlgId, pTempECCPubKeyBlob,
                    pbID, ulIDLen, phAgreementHandle);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            resultBean = new ResultBean<>();
            resultBean.retHandle = phAgreementHandle[0];
            resultBean.result = pTempECCPubKeyBlob;
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, resultBean);
    }

    /**
     * ECC产生协商数据并计算会话密钥<br>
     * 功能描述：使用ECC密钥协商算法，产生协商参数并计算会话密钥，输出临时ECC密钥对公钥，并返回产生的密钥句柄。<br>
     * 备注：本函数由响应方调用
     *
     * @param hContainer                [IN]容器句柄。
     * @param ulAlgId                   [IN]会话密钥算法标识。
     * @param pSponsorECCPubKeyBlob     [IN]发起方的ECC公钥。
     * @param pSponsorTempECCPubKeyBlob [IN]发起方的临时ECC公钥。
     * @param pbID                      [IN]响应方的ID。长度不大于32。
     * @param pbSponsorID               [IN]发起方的ID。长度不大于32。
     * @return 其中：Pair.first 是错误码
     * Pair.second 是返回的Bean对象。其中：<br/>
     * Bean.resule为响应方的临时ECC公钥。<br/>
     * Bean.retHandle为返回的对称算法密钥句柄。<br/>
     */
    public Pair<Integer, ResultBean<EccPublicKeyBlob>>
    GenerateAgreementDataAndKeyWithECC(long hContainer, int ulAlgId,
                                       EccPublicKeyBlob pSponsorECCPubKeyBlob,
                                       EccPublicKeyBlob pSponsorTempECCPubKeyBlob,
                                       byte[] pbID, byte[] pbSponsorID) {
        if (isEmpty(pSponsorECCPubKeyBlob, pSponsorTempECCPubKeyBlob, pbID, pbSponsorID)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulIDLen = pbID.length;
        int ulSponsorIDLen = pbSponsorID.length;
        EccPublicKeyBlob pTempECCPubKeyBlob = new EccPublicKeyBlob();
        long[] phKeyHandle = new long[1];
        ResultBean<EccPublicKeyBlob> resultBean = null;
        try {
            ret = skfChipBinder.GenerateAgreementDataAndKeyWithECC(hContainer, ulAlgId, pSponsorECCPubKeyBlob, pSponsorTempECCPubKeyBlob, pTempECCPubKeyBlob, pbID, ulIDLen, pbSponsorID, ulSponsorIDLen, phKeyHandle);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            resultBean = new ResultBean<>();
            resultBean.retHandle = phKeyHandle[0];
            resultBean.result = pTempECCPubKeyBlob;
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, resultBean);
    }

    /**
     * ECC计算会话密钥<br>
     * 功能描述：使用ECC密钥协商算法，使用自身协商句柄和响应方的协商参数计算会话密钥，同时返回会话密钥句柄。<br>
     * 备注：协商的发起方获得响应方的协商参数后调用本函数，计算会话密钥。
     *
     * @param hAgreementHandle   [IN]密钥协商句柄。
     * @param pECCPubKeyBlob     [IN]外部输入的响应方ECC公钥。
     * @param pTempECCPubKeyBlob [IN]外部输入的响应方临时ECC公钥。
     * @param pbID               [IN]响应方的ID。长度不大于32。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是返回的密钥句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> GenerateKeyWithECC(long hAgreementHandle, EccPublicKeyBlob pECCPubKeyBlob,
                                                  EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID) {
        if (isEmpty(pECCPubKeyBlob, pTempECCPubKeyBlob, pbID)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulIDLen = pbID.length;
        long[] phKeyHandle = new long[1];
        try {
            ret = skfChipBinder.GenerateKeyWithECC(hAgreementHandle, pECCPubKeyBlob, pTempECCPubKeyBlob, pbID, ulIDLen, phKeyHandle);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phKeyHandle[0]);
    }

    /**
     * 导出公钥<br/>
     * 功能描述：导出容器中的签名公钥或加密公钥
     *
     * @param hContainer [IN]密钥容器句柄。
     * @param bSignFlag  [IN]{@link SkfApiCode#TRUE}表示导出签名公钥，{@link SkfApiCode#FALSE}表示导出加密公钥。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是RSA公钥结构或者ECC公钥结构。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> ExportPublicKey(long hContainer, int bSignFlag) {
        int ret;
        byte[] pbBlobTmp = new byte[512];
        int[] pulBlobLen = new int[1];
        byte[] pbBlob = null;
        try {
            ret = skfChipBinder.ExportPublicKey(hContainer, bSignFlag, pbBlobTmp, pulBlobLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbBlob = new byte[pulBlobLen[0]];
            System.arraycopy(pbBlobTmp, 0, pbBlob, 0, pulBlobLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbBlob);
    }

    /**
     * 导入会话密钥<br/>
     * 功能描述：导入会话密钥密文，使用容器中加密私钥解密得到会话密钥。<br/>
     * 权限要求：用户权限。
     *
     * @param hContainer   [IN]容器句柄。
     * @param ulAlgId      [IN]会话密钥算法标识。
     * @param pbWrapedData [IN]要导入的会话密钥密文。当容器为ECC类型时，此参数为ECCCIPHERBLOB密文数据,当容器为RSA类型时，此参数为RSA公钥加密后的数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是会话密钥句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> ImportSessionKey(long hContainer, int ulAlgId, byte[] pbWrapedData) {
        if (isEmpty(pbWrapedData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulWrapedLen = pbWrapedData.length;
        long[] phKey = new long[1];
        try {
            ret = skfChipBinder.ImportSessionKey(hContainer, ulAlgId, pbWrapedData, ulWrapedLen, phKey);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phKey[0]);
    }

    /**
     * 明文导入会话密码<br/>
     * 功能描述：设置明文对称密钥，返回密钥句柄。
     *
     * @param pbKey   [IN]会话密钥值。
     * @param ulAlgID [IN]会话密钥算法标识。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是会话密钥句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> SetSymmKey(byte[] pbKey, int ulAlgID) {
        if (isEmpty(pbKey)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] phKey = new long[1];
        long hDev = 0;
        try {
            ret = skfChipBinder.SetSymmKey(hDev, pbKey, ulAlgID, phKey);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phKey[0]);
    }

    /**
     * 加密初始化<br/>
     * 功能描述：数据加密初始化。设置数据加密的算法相关参数。
     *
     * @param hKey         [IN]加密密钥句柄。
     * @param EncryptParam [IN]分组密钥算法相关参数：初始向量、初始向量长度、填充方法、反馈值的位长度
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    public int EncryptInit(long hKey, BlockCipherParam EncryptParam) {
        if (isEmpty(EncryptParam)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.EncryptInit(hKey, EncryptParam);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 单组数据加密<br/>
     * 功能描述：单一分组数据的加密操作。用指定加密密钥对指定数据进行加密，被加密的数据只包含一个分组，加密
     * 后的密文保存到指定的缓冲区中。该方法只对单个分组数据进行加密，在调用Encrypt之前必须先调用EncryptInit初始化。
     * Encrypt等价于先调用EncryptUpdate再调用EncryptFinal
     *
     * @param hKey   [IN]加密密钥句柄。
     * @param pbData [IN]待加密数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是加密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> Encrypt(long hKey, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;
        byte[] pbEncryptedDataTmp = new byte[ulDataLen + 16];
        int[] pulEncryptedLen = new int[1];
        byte[] pbEncryptedData = null;
        try {
            ret = skfChipBinder.Encrypt(hKey, pbData, ulDataLen, pbEncryptedDataTmp, pulEncryptedLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbEncryptedData = new byte[pulEncryptedLen[0]];
            System.arraycopy(pbEncryptedDataTmp, 0, pbEncryptedData, 0, pulEncryptedLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbEncryptedData);
    }

    /**
     * 多组数据加密<br/>
     * 功能描述：对多个分组数据进行加密。用指定加密密钥对指定数据进行加密，被加密的数据包含多个分组，加密后的密文保存到指定的缓冲区中。
     * 在调用EncryptUpdate之前必须先调用EncryptInit初始化加密操作，在调用EncryptUpdate之后，必须调用EncryptFinal结束加密操作。
     *
     * @param hKey   [IN]加密密钥句柄。
     * @param pbData [IN]待加密数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是加密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> EncryptUpdate(long hKey, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;

        //start : modified by xwy on 2017-7-11 08:46:58 update过程中如果数据组合够一个分组那么输出数据为一个分组
        //byte[] pbEncryptedDataTmp = new byte[ulDataLen];
        byte[] pbEncryptedDataTmp = new byte[ulDataLen + 16];
        //end : modified by xwy on 2017-7-11 08:46:58 update过程中如果数据组合够一个分组那么输出数据为一个分组
        int[] pulEncryptedLen = new int[1];
        byte[] pbEncryptedData = null;
        try {
            ret = skfChipBinder.EncryptUpdate(hKey, pbData, ulDataLen, pbEncryptedDataTmp, pulEncryptedLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbEncryptedData = new byte[pulEncryptedLen[0]];
            System.arraycopy(pbEncryptedDataTmp, 0, pbEncryptedData, 0, pulEncryptedLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbEncryptedData);
    }

    /**
     * 结束加密<br/>
     * 功能描述：结束多个分组的数据加密，返回剩余加密结果。先调用EncryptInit初始化，再调用EncryptUpdate对多个分组加密，最后调用EncryptFinal结束多个分组数据加密。
     *
     * @param hKey [IN]加密密钥句柄。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是加密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> EncryptFinal(long hKey) {
        int ret;
        byte[] pbEncryptedDataTmp = new byte[16];
        int[] pulEncryptedLen = new int[1];
        byte[] pbEncryptedData = null;
        try {
            ret = skfChipBinder.EncryptFinal(hKey, pbEncryptedDataTmp, pulEncryptedLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbEncryptedData = new byte[pulEncryptedLen[0]];
            System.arraycopy(pbEncryptedDataTmp, 0, pbEncryptedData, 0, pulEncryptedLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbEncryptedData);
    }

    /**
     * 解密初始化<br/>
     * 功能描述：设置数据解密的算法相关参数。调用DecryptInit之后，可以调用Decrypt对单个分组数据进行解密，
     * 也可以多次调用DecryptUpdate之后再调用DecryptFinal完成对分组数据的解密
     *
     * @param hKey         [IN]解密密钥句柄。
     * @param EncryptParam [IN]分组密钥算法相关参数。
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    public int DecryptInit(long hKey, BlockCipherParam EncryptParam) {
        if (isEmpty(EncryptParam)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.DecryptInit(hKey, EncryptParam);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 单组数据解密<br/>
     * 功能描述：单一分组数据的解密操作。用指定解密密钥对指定数据进行解密，被解密的数据只包含一个分组，解密后的明文保存到指定的缓冲区中。
     * 在调用Decrypt之前必须先调用DecryptInit初始化解密操作。Decrypt等价于先调用DecryptUpdate，再调用DecryptFinal。
     *
     * @param hKey            [IN]解密密钥句柄。
     * @param pbEncryptedData [IN]待解密数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是解密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> Decrypt(long hKey, byte[] pbEncryptedData) {
        if (isEmpty(pbEncryptedData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulEncryptedLen = pbEncryptedData.length;
        byte[] pbDataTmp = new byte[ulEncryptedLen];
        int[] pulDataLen = new int[1];
        byte[] pbData = null;

        try {
            ret = skfChipBinder.Decrypt(hKey, pbEncryptedData, ulEncryptedLen, pbDataTmp, pulDataLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbData = new byte[pulDataLen[0]];
            System.arraycopy(pbDataTmp, 0, pbData, 0, pulDataLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbData);
    }

    /**
     * 分组数据解密<br/>
     * 功能描述：多个分组数据进行解密操作。用指定解密密钥对指定数据进行解密，被解密的数据包含多个分组，解密后的明文保存到指定的缓存区中。
     * 在调用DecryptUpdate之前，必须先调用DecryptInit初始化解密操作；在调用DecryptUpdate之后，必须调用DecryptFinal结束解密操作。
     *
     * @param hKey            [IN]解密密钥句柄。
     * @param pbEncryptedData [IN]待解密数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是解密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> DecryptUpdate(long hKey, byte[] pbEncryptedData) {
        if (isEmpty(pbEncryptedData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulEncryptedLen = pbEncryptedData.length;
        //start : modified by xwy on 2017-7-11 08:46:58 update过程中如果数据组合够一个分组那么输出数据为一个分组
        //byte[] pbDataTmp = new byte[ulEncryptedLen];
        byte[] pbDataTmp = new byte[ulEncryptedLen + 16];
        //end : modified by xwy on 2017-7-11 08:46:58 update过程中如果数据组合够一个分组那么输出数据为一个分组
        int[] pulDataLen = new int[1];
        byte[] pbData = null;

        try {
            ret = skfChipBinder.DecryptUpdate(hKey, pbEncryptedData, ulEncryptedLen, pbDataTmp, pulDataLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbData = new byte[pulDataLen[0]];
            System.arraycopy(pbDataTmp, 0, pbData, 0, pulDataLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbData);
    }

    /**
     * 结束多个分组的解密<br/>
     * 功能描述：先调用DecryptInit初始化解密操作，再调用DecryptUpdate对多个分组进行解密，
     * 最后调用DecryptFinal结束多个分组数据的解密。
     *
     * @param hKey [IN]解密密钥句柄。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是解密后的数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> DecryptFinal(long hKey) {
        int ret;
        byte[] pbDecryptedDataTmp = new byte[16];
        int[] pulDecryptedDataLen = new int[1];
        byte[] pbDecryptedData = null;

        try {
            ret = skfChipBinder.DecryptFinal(hKey, pbDecryptedDataTmp, pulDecryptedDataLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbDecryptedData = new byte[pulDecryptedDataLen[0]];
            System.arraycopy(pbDecryptedDataTmp, 0, pbDecryptedData, 0, pulDecryptedDataLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbDecryptedData);
    }

    /**
     * 密码杂凑初始化<br/>
     * 功能描述：初始化密码杂凑计算操作，指定计算密码杂凑的算法。
     * 备注：当ulAlgID为SGD_SM3且ulIDLen不为0的情况下，pPubKey、pucID有效，执行SM2算法签名预处理1操作。
     *
     * @param ulAlgID [IN]密码杂凑算法标识。
     * @param pPubKey [IN]签名者公钥。当ulAlgID为SGD_SM3时有效。
     * @param pucID   [IN]签名者ID值。当ulAlgID为SGD_SM3时有效。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是密码杂凑对象句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> DigestInit(int ulAlgID, EccPublicKeyBlob pPubKey, byte[] pucID) {
        if (isEmpty(pPubKey, pucID)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] phHash = new long[1];
        int ulIDLen = pucID.length;
        long hDev = 0;
        try {
            ret = skfChipBinder.DigestInit(hDev, ulAlgID, pPubKey, pucID, ulIDLen, phHash);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phHash[0]);
    }

    /**
     * 单组数据密码杂凑<br/>
     * 功能描述：对单一分组的消息进行密码杂凑计算。调用Digest之前必须调用DigestInit初始化。<br/>
     * 调用 Digest 之前，必须调用 DigestInit 初始化杂凑计算操作。Digest等价于多次调用 DigestUpdate 之后再调用 DigestFinal。
     *
     * @param hHash  [IN]密码杂凑对象句柄。
     * @param pbData [IN]消息数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是密码杂凑数据。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> Digest(long hHash, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;
        //各种杂凑运算的结果最大的结果长度为64字节
        byte[] phHashDataTmp = new byte[64];
        int[] pulHashLen = new int[1];
        byte[] phHashData = null;
        try {
            ret = skfChipBinder.Digest(hHash, pbData, ulDataLen, phHashDataTmp, pulHashLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            phHashData = new byte[pulHashLen[0]];
            System.arraycopy(phHashDataTmp, 0, phHashData, 0, pulHashLen[0]);

        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phHashData);
    }

    /**
     * 分组数据密码杂凑<br/>
     * 功能描述：对多个分组数据进行密码杂凑运算。
     * 调用 DigestUpdate 之前，必须调用 DigestInit 初始化杂凑计算操作；调用 DigestUpdate 之后，必须调用DigestFinal 结束杂凑计算操作。
     *
     * @param hHash  [IN]密码杂凑对象句柄。
     * @param pbData [IN]消息数据。
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    public int DigestUpdate(long hHash, byte[] pbData) {
        if (isEmpty(pbData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        int ulDataLen = pbData.length;
        try {
            ret = skfChipBinder.DigestUpdate(hHash, pbData, ulDataLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 结束密码杂凑<br/>
     * 功能描述：结束多个分组的密码杂凑计算，将结果保存到指定的缓冲区。<br/>
     * DigestFinal 必须用于 DigestUpdate 之后。
     *
     * @param hHash [IN]密码杂凑对象句柄。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是杂凑结果。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> DigestFinal(long hHash) {
        int ret;
        byte[] pHashDataTmp = new byte[64];
        int[] pulHashLen = new int[1];
        byte[] pHashData = null;
        try {
            ret = skfChipBinder.DigestFinal(hHash, pHashDataTmp, pulHashLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pHashData = new byte[pulHashLen[0]];
            System.arraycopy(pHashDataTmp, 0, pHashData, 0, pulHashLen[0]);

        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pHashData);
    }

    /**
     * 消息鉴别码运算初始化<br/>
     * 功能描述：初始化消息鉴别码计算操作，设置计算消息鉴别码的密钥参数，并返回消息鉴别码句柄。<br/>
     * 备注：消息鉴别码计算采用分组加密算法的 CBC 模式，将加密结果的最后一块作为计算结果。待计算数据的长度必须是分组加密算法块长的倍数，接口内部不作数据填充。
     *
     * @param hKey      [IN]计算消息鉴别码的密钥句柄。
     * @param pMacParam [IN]消息认证计算相关参数，包括初始向量、初始向量长度、填充方法等。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是消息鉴别码对象句柄。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, Long> MacInit(long hKey, BlockCipherParam pMacParam) {
        if (isEmpty(pMacParam)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        long[] phMac = new long[1];
        try {
            ret = skfChipBinder.MacInit(hKey, pMacParam, phMac);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, phMac[0]);
    }

    /**
     * 单组数据消息鉴别码运算<br/>
     * 功能描述：Mac 计算单一分组数据的消息鉴别码。<br/>
     * 备注：调用 Mac 之前，必须调用 MacInit 初始化消息鉴别码计算操作。Mac等价于多次调用 MacUpdate 之后再调用 MacFinal。
     *
     * @param hMac   [IN]消息鉴别码句柄。
     * @param pbData [IN]待计算数据。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是计算后的 Mac 结果。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> Mac(long hMac, byte[] pbData) {
        if (isEmpty(pbData)) {
            return Pair.create(MultiChipJar.SKF_PARAM_ERROR, null);
        }
        int ret;
        int ulDataLen = pbData.length;
        byte[] pbMacDataTmp = new byte[16];
        int[] pulMacLen = new int[1];
        byte[] pbMacData = null;

        try {
            ret = skfChipBinder.Mac(hMac, pbData, ulDataLen, pbMacDataTmp, pulMacLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbMacData = new byte[pulMacLen[0]];
            System.arraycopy(pbMacDataTmp, 0, pbMacData, 0, pulMacLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbMacData);
    }

    /**
     * 多组数据消息鉴别码运算<br/>
     * 功能描述：计算多个分组数据的消息鉴别码。<br/>
     * 备注：调用 MacUpdate 之前，必须调用 MacInit 初始化消息鉴别码计算操作；调用 MacUpdate 之后，必须调用 MacFinal 结束多个分组数据的消息鉴别码计算操作。
     *
     * @param hMac   [IN]消息鉴别码句柄。
     * @param pbData [IN]待计算数据。
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    public int MacUpdate(long hMac, byte[] pbData) {
        if (isEmpty(pbData)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        int ulDataLen = pbData.length;
        try {
            ret = skfChipBinder.MacUpdate(hMac, pbData, ulDataLen);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 结束消息鉴别码运算<br/>
     * 功能描述：结束多个分组数据的消息鉴别码计算操作。
     * 备注：MacFinal 必须用于 MacUpdate 之后。
     *
     * @param hMac [IN]消息鉴别码句柄。
     * @return 其中：Pair.first 是错误码。
     * Pair.second 是消息鉴别码。<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, byte[]> MacFinal(long hMac) {
        int ret;
        byte[] pbMacDataTmp = new byte[16];
        int[] pulMacDataLen = new int[1];
        byte[] pbMacData = null;
        try {
            ret = skfChipBinder.MacFinal(hMac, pbMacDataTmp, pulMacDataLen);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            pbMacData = new byte[pulMacDataLen[0]];
            System.arraycopy(pbMacDataTmp, 0, pbMacData, 0, pulMacDataLen[0]);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return Pair.create(ret, pbMacData);
    }

    /**
     * 关闭密码对象句柄<br/>
     * 功能描述：关闭会话密钥、密码杂凑对象、MAC对象、ECC密钥协商等句柄<br/>
     *
     * @param hHandle [IN]要关闭的对象句柄。
     * @return 0:  成功。
     * 其他：参看错误码定义。
     */
    public int CloseHandle(long hHandle) {
        int ret;
        try {
            ret = skfChipBinder.CloseHandle(hHandle);
        } catch (Exception e) {
            ret = handleException(e);
        }

        return ret;
    }

    /***************国密接口结束********************/
    /**
     * 信大捷安自己的接口。<br>
     * 用于解锁信大捷安的卡。
     *
     * @param role   要解锁的PIN码角色
     * @param key    解锁时用到的key
     * @param newPin 解锁后的PIN码
     */
    public int XdjaUnlockPin(int role, byte[] key, byte[] newPin) {
        if (isEmpty(key, newPin)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.XdjaUnlockPin(role, key, newPin);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    /**
     * 信大捷安自己的接口。<br>
     * 用于解锁信大捷安的卡,使用usn和动态解锁码解锁。
     *
     * @param role        要解锁的PIN码角色
     * @param usn         usn
     * @param usnlockCode usnlockCode
     * @param newPin      解锁后的PIN码
     */
    public int XdjaUnlockPinUseUsn(int role, byte[] usn, byte[] usnlockCode, byte[] newPin) {
        if (isEmpty(usn, usnlockCode, newPin)) {
            return MultiChipJar.SKF_PARAM_ERROR;
        }
        int ret;
        try {
            ret = skfChipBinder.XdjaUnlockPinUseUsn(role, usn, usnlockCode, newPin);
        } catch (Exception e) {
            ret = handleException(e);
        }
        return ret;
    }

    //add 2017年7月29日16:29:38 weizg
    /**
     * 获取芯片的cos信息
     * @return 芯片cos信息
     */
    public String XdjaGetCosInfo(){
        String cosInfo = "";
        try {
            cosInfo = skfChipBinder.XdjaGetCosInfo();
        } catch (Exception e) {
            handleException(e);
        }

        return cosInfo;
    }


    private int handleException(Exception e) {
        e.printStackTrace();
        return -1;
    }

    private SuperChip(ISkfChip skfChipBinder1) {

        this.skfChipBinder = skfChipBinder1;
        try {
            this.skfChipBinder.asBinder().linkToDeath(new IBinder.DeathRecipient() {
                @Override
                public void binderDied() {
                    skfChipBinder = null;
                }
            }, Binder.FLAG_ONEWAY);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 构造SuperChipBinder对象的建造类
     */
    public static final class Builder {
        /**
         * 构造方法
         *
         * @return 每次都返回一个SuperChipBinder对象。如果context为空，则返回null
         */
        public SuperChip build(ISkfChip skfChipBinder) {
            if (skfChipBinder == null) {
                return null;
            }
            return new SuperChip(skfChipBinder);
        }
    }

    private boolean isEmpty(Object... objects) {
        Object object = null;
        for (Object object1 : objects) {
            object = object1;
            if (object == null) {
                return true;
            } else if (object instanceof byte[] && ((byte[]) object).length == 0) {
                return true;
            }
        }
        return false;
    }

}
