package com.xdja.multichip.jniapi;

import android.content.Context;
import android.os.Bundle;
import android.os.IBinder;
import android.os.ParcelUuid;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.multichip.IGetMultiJniApi;
import com.xdja.multichip.IMultiJniApi;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.param.ParamKeywords;
import com.xdja.multichip.param.VhsmInitParam;
import com.xdja.multichip.param.VhsmServerInfo;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_GET_BINDER_FAIL;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/1/15 <br/>
 */
public class JarMultiJniApiVhsmManager {

    private static JarMultiJniApiVhsmManager instance;
    private static final String TAG = "JarMultiJniApiVhsm";

    private static final String KEY_RET = "ret";

    private JarMultiJniApiVhsmManager() {

    }

    public static JarMultiJniApiVhsmManager getInstance() {
        if (instance == null) {
            synchronized (JarMultiJniApiVhsmManager.class) {
                if (instance == null) {
                    instance = new JarMultiJniApiVhsmManager();
                }
            }
        }
        return instance;
    }

    /**
     * 设置vhsm路径到私有路径，如果芯片管家存在就初始化到芯片管家私有目录，如果安全接入存在就初始化到安全接入私有目录
     */
    public int setVhsmPriPath(Context context) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return RET_GET_BINDER_FAIL;
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_VHSM_CUSTOM_PATH);
        bundle.putString(ParamKeywords.KEY_VHSM_PATH, ParamKeywords.VHSM_PRI_PATH);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            Log.e(TAG, "set custom vhsm pri path result" + result);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 设置vhsm路径
     */
    public int setVhsmCustomPath(Context context, String path) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return RET_GET_BINDER_FAIL;
        }
        if (TextUtils.isEmpty(path)) {
            path = "";
        }
        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_VHSM_CUSTOM_PATH);
        bundle.putString(ParamKeywords.KEY_VHSM_PATH, path);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            Log.e(TAG, "set custom vhsm path result" + result);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 初始化私有云Vhsm
     *
     * @return
     */

    public int initVhsm(Context context, int type, VhsmInitParam param) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return RET_GET_BINDER_FAIL;
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_INIT_VHSM);
        bundle.putInt(ParamKeywords.KEY_INIT_VHSM_TYPE, type);
        bundle.putParcelable(ParamKeywords.KEY_INIT_VHSM_PARAM, param);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            Log.e(TAG, "vhsm init callmethod=" + result);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            int ret = result.getInt(ParamKeywords.KEY_int_ret);
            Log.e(TAG, "vhsm init ret=" + ret);
            return ret;
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 设置服务器信息
     */
    public int setVhsmServerInfo(Context context, VhsmServerInfo serverInfo) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return RET_GET_BINDER_FAIL;
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_SET_SERVER_INFO);
        bundle.putParcelable(ParamKeywords.KEY_SERVER_INFO, serverInfo);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            Log.e(TAG, "vhsm set server info  callMethod=" + result);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            return result.getInt(ParamKeywords.KEY_int_ret);
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 未真正初始化成功之前，会生成一个卡id，这个卡id可以提供给外界展示
     *
     * @param context
     * @return
     */
    public String getVhsmVirtuallyId(Context context) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return "";
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_GET_VHSM_VIRTUALLYID);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            Log.e(TAG, "vhsm getVhsmVirtuallyId  callMethod=" + result);
            if (result == null) {
                return "";
            }
            int ret = result.getInt(ParamKeywords.KEY_int_ret);
            if (ret != 0) {
                Log.e(TAG, "vhsm getVhsmVirtuallyId  result=" + ret);
                return "";
            }
            return result.getString(ParamKeywords.KEY_String_cardId);

        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 获取服务器信息
     */
    public VhsmServerInfo getServerInfo(Context context) {
        final Pair<Integer, JarJniApiProxy> make = JarMultiJniApiManager.getInstance().make(context, JniApiParam.TYPE_VHSM_NET);
        if (make.first != 0 || make.second == null) {
            Log.e(TAG, "get vhsm  serverinfo: proxy is empty ret=" + make.first);
            return null;
        }
        JarJniApiProxy proxy = make.second;
        Bundle extras = new Bundle();
        extras.putString(ParamKeywords.KEY_String_method, "getExistVhsmServerInfo");

        Bundle bundle = proxy.callMethod(extras);
        if (bundle == null) {
            return null;
        }
        int ret = bundle.getInt(KEY_RET);
        if (ret == JNIAPI.XKR_OK) {
            Log.e(TAG, "get vhsm server info success:serverType=" + bundle.getInt("serverType"));
            VhsmServerInfo vhsmServerInfo = new VhsmServerInfo();
            vhsmServerInfo.setServerType(bundle.getInt("serverType"));
            vhsmServerInfo.setAddrType(bundle.getInt("addrType"));
            vhsmServerInfo.setPort(bundle.getInt("port"));
            vhsmServerInfo.setIp(bundle.getString("addr"));
            vhsmServerInfo.setDn(bundle.getString("addr"));
            return vhsmServerInfo;
        } else {
            Log.e(TAG, "get vhsm  serverinfo ret=" + make.first);
            return null;
        }
    }

    /**
     * 更新服务器信息
     * @param context
     * @param isIP
     * @param ip
     * @param port
     * @return
     */
    public int updateServerInfo(Context context, boolean isIP,String ip, int port) {
        final Pair<Integer, JarJniApiProxy> make = JarMultiJniApiManager.getInstance().make(context, JniApiParam.TYPE_VHSM_NET);
        if (make.first != 0 || make.second == null) {
            Log.e(TAG, "update serverinfo: proxy is empty ret=" + make.first);
            return RET_GET_BINDER_FAIL;
        }
        JarJniApiProxy proxy = make.second;
        Bundle extras = new Bundle();
        extras.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_UPDATE_SERVER_INFO);
        extras.putString("ip", ip);
        extras.putBoolean("isIP", isIP);
        extras.putInt("port", port);

        Bundle bundle = proxy.callMethod(extras);
        if (bundle == null) {
            return JarMultiJniApiErrorCode.RET_EXCEPTION;
        }
        int ret = bundle.getInt(KEY_RET);
        if (ret == JNIAPI.XKR_OK) {
            Log.e(TAG, "update serverinfo success ip=" + ip + " isIP=" + isIP + " port=" + port);
            return 0;
        } else {
            Log.e(TAG, "pdate serverinfo ret=" + ret);
            return ret;
        }
    }

    /**
     * 创建默认虚拟卡
     *
     * @param context
     * @return
     */
    public Pair<Integer, Pair<JniApiParam, JarJniApiProxy>> createDefaultVhsm(Context context) {
        IGetMultiJniApi multiBinder = JarMultiJniApiManager.getInstance().getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return Pair.create(RET_GET_BINDER_FAIL, null);
        }
        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_CreateDefaultVhsm);
        try {
            Bundle result = multiBinder.callMethod(bundle);
            if (result == null) {
                return Pair.create(JarMultiJniApiErrorCode.RET_EXCEPTION, null);
            }
            result.setClassLoader(JniApiParam.class.getClassLoader());
            int ret = result.getInt(ParamKeywords.KEY_int_ret);
            if (ret != 0) {
                return Pair.create(ret, null);
            }
            JniApiParam jniApiParam = result.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
            final IBinder binder = result.getBinder(ParamKeywords.KEY_Binder_Binder);
            JarJniApiTypeProxy proxy = new JarJniApiTypeProxy(context, IMultiJniApi.Stub.asInterface(binder), jniApiParam.chipType);
            return Pair.create(ret, Pair.create(jniApiParam, (JarJniApiProxy) proxy));
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return Pair.create(JarMultiJniApiErrorCode.RET_EXCEPTION, null);
    }

    /**
     * 贴膜卡SM4,目前接口不支持CBC模式
     *
     * @param proxy
     * @param dataIn 要加密或解密的数据（填充过的且长度是16的倍数）
     * @param flag   加密模式
     * @param kID    加密用0x08，解密用0x09
     * @param IV     初始向量，当CBC_MODE时需要
     * @return
     */
    public int SM4(JarJniApiProxy proxy, byte[] dataIn, int dataLen, int flag, byte[] dataOut,
                   byte kID, byte[] IV) {
        if (proxy == null || dataIn == null || dataIn.length < dataLen || dataOut == null ||
                dataOut.length < dataLen) {
            return JarMultiJniApiErrorCode.RET_PARAM_ERROR;
        }
        if (IV != null && IV.length != 16) {
            return JarMultiJniApiErrorCode.RET_PARAM_ERROR;
        }

        Bundle extras = new Bundle();
        extras.putString(ParamKeywords.KEY_String_method, "SM4");
        extras.putByteArray("dataIn", dataIn);
        extras.putInt("dataLen", dataLen);
        extras.putInt("flag", flag);
        extras.putByte("kID", kID);
        extras.putByteArray("IV", IV);

        Bundle bundle = proxy.callMethod(extras);
        if (bundle == null) {
            return JarMultiJniApiErrorCode.RET_PARAM_ERROR;
        }
        int ret = bundle.getInt(KEY_RET);
        byte[] dataOutTmp = null;
        if (ret == JNIAPI.XKR_OK) {
            dataOutTmp = bundle.getByteArray("dataOut");
            System.arraycopy(dataOutTmp, 0, dataOut, 0, dataOutTmp.length);
            byte[] IVTmp = bundle.getByteArray("IV");
            if (IVTmp != null && IV != null) {
                System.arraycopy(IVTmp, 0, IV, 0, IVTmp.length);
            }
        }
        return ret;
    }
}
