package com.xdja.multichip.jniapi;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Pair;

import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.jar.forcallprovider.ForCallProvider;
import com.xdja.multichip.IGetMultiJniApi;
import com.xdja.multichip.IMultiJniApi;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.param.ParamKeywords;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_EXCEPTION;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_GET_BINDER_FAIL;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_MAKE_FAILE;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_PARAM_ERROR;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/7/24 <br/>
 */
public class JarMultiJniApiManager {

    public static final int RET_OK = 0;

    private static JarMultiJniApiManager instance;
    private IGetMultiJniApi getMultiJniApiBinder = null;

    private HashMap<String, IBinder> cardIdBinderMap = new HashMap<>();
    private HashMap<IBinder, JarJniApiProxy> binderProxyMap = new HashMap<>();
    /**
     * 通过 {@link #make(Context, int)} 生成的 {@link JarJniApiTypeProxy} 的map；
     * 其中Key是卡类型
     */
    private HashMap<Integer, JarJniApiProxy> typeProxyMap = new HashMap<>();


    private JarMultiJniApiManager() {

    }

    public static JarMultiJniApiManager getInstance() {
        if (instance == null) {
            synchronized (JarMultiJniApiManager.class) {
                if (instance == null) {
                    instance = new JarMultiJniApiManager();
                }
            }
        }
        return instance;
    }

    /**
     * 得到设备上所有的卡的相关参数
     *
     * @param context
     * @return
     */
    public Pair<Integer, List<JniApiParam>> getAll(Context context) {
        if (context == null) {
            return Pair.create(RET_PARAM_ERROR, null);
        }
        IGetMultiJniApi multiBinder = getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return Pair.create(RET_GET_BINDER_FAIL, null);
        }
        try {
            List<Bundle> result = multiBinder.getMultiJniApi();
            List<JniApiParam> list = new ArrayList<>();
            if (result == null || result.size() == 0) {
                return Pair.create(RET_OK, list);
            }
            for (Bundle bundle : result) {
                bundle.setClassLoader(JniApiParam.class.getClassLoader());
                JniApiParam jniApiParam = bundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
                final IBinder binder = bundle.getBinder(ParamKeywords.KEY_Binder_Binder);
                final String cardId = jniApiParam.cardId;
                cardIdBinderMap.put(cardId, binder);
                binder.linkToDeath(new IBinder.DeathRecipient() {
                    @Override
                    public void binderDied() {
                        cardIdBinderMap.remove(cardId);
                        binderProxyMap.remove(binder);
                    }
                }, 0);
                list.add(jniApiParam);
            }
            return Pair.create(0, list);
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return Pair.create(RET_EXCEPTION, null);
    }

    /**
     * 根据类型获取代理
     *
     * @param context
     * @param cardType
     * @return
     */
    public Pair<Integer, JarJniApiProxy> make(Context context, final int cardType) {
        if (context == null) {
            return Pair.create(RET_PARAM_ERROR, null);
        }
        if (cardType != JniApiParam.TYPE_ONBOARD &&
                cardType != JniApiParam.TYPE_TF &&
                cardType != JniApiParam.TYPE_COVERED &&
                cardType != JniApiParam.TYPE_BLUETOOTH) {
            return Pair.create(RET_PARAM_ERROR, null);
        }

        if (typeProxyMap.containsKey(cardType)) {
            return Pair.create(0, typeProxyMap.get(cardType));
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_makeByType);
        bundle.putInt(ParamKeywords.KEY_int_cardType, cardType);
        IGetMultiJniApi multiBinder = getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return Pair.create(RET_GET_BINDER_FAIL, null);
        }
        try {
            Bundle result = multiBinder.callMethod(bundle);
            if (result != null) {
                int ret = result.getInt(ParamKeywords.KEY_int_ret);
                if (ret != 0) {
                    return Pair.create(ret, null);
                }
                ArrayList<Bundle> bundleList = result.getParcelableArrayList(ParamKeywords.KEY_ParcelableArrayList_Bundle);
                if (bundleList == null || bundleList.size() == 0) {
                    return Pair.create(RET_MAKE_FAILE, null);
                }
                Bundle typeBundle = bundleList.get(0);
                typeBundle.setClassLoader(JniApiParam.class.getClassLoader());

                JniApiParam jniApiParam = typeBundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
                IBinder binder = typeBundle.getBinder(ParamKeywords.KEY_Binder_Binder);
                JarJniApiTypeProxy proxy = new JarJniApiTypeProxy(context, IMultiJniApi.Stub.asInterface(binder), jniApiParam.chipType);
                typeProxyMap.put(cardType, proxy);
                binder.linkToDeath(new IBinder.DeathRecipient() {
                    @Override
                    public void binderDied() {
                        typeProxyMap.remove(cardType);
                    }
                }, 0);
                return Pair.create(0, (JarJniApiProxy) proxy);
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return Pair.create(RET_EXCEPTION, null);
    }

    /**
     * 得到操作卡的对象;
     *
     * @param context
     * @param cardId
     * @return
     */
    public Pair<Integer, JarJniApiProxy> make(Context context, String cardId) {
        if (context == null || TextUtils.isEmpty(cardId)) {
            return Pair.create(RET_PARAM_ERROR, null);
        }

        IBinder binder = cardIdBinderMap.get(cardId);
        if (binder == null) {
            getAll(context);
            binder = cardIdBinderMap.get(cardId);
        }

        if (binder != null) {
            XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
            try {
                int ret = IMultiJniApi.Stub.asInterface(binder).GetDevInfo(0, devinfo);
                if (ret == 0) {
                    String getCardId = new String(devinfo.cardid, 0, 32);
                    if (cardId.equals(getCardId)) {
                        if (binderProxyMap.containsKey(binder)) {
                            return Pair.create(0, binderProxyMap.get(binder));
                        } else {
                            JarJniApiProxy jniApiJarProxy = new JarJniApiProxy(context, IMultiJniApi.Stub.asInterface(binder), getCardId);
                            binderProxyMap.put(binder, jniApiJarProxy);
                            return Pair.create(0, jniApiJarProxy);
                        }
                    } else {
                        cardIdBinderMap.remove(cardId);
                        binderProxyMap.remove(binder);
                    }
                }
            } catch (RemoteException e) {
                e.printStackTrace();
                return Pair.create(RET_EXCEPTION, null);
            }
            return Pair.create(RET_MAKE_FAILE, null);
        }
        return Pair.create(RET_GET_BINDER_FAIL, null);
    }

    HashMap<String, IBinder> getCardIdBinderMap() {
        return cardIdBinderMap;
    }

    HashMap<IBinder, JarJniApiProxy> getBinderProxyMap() {
        return binderProxyMap;
    }

    /**
     * 得到Binder代理对象
     *
     * @param context
     * @return
     */
    private IGetMultiJniApi getGetMultiJniApiBinder(Context context) {
        if (getMultiJniApiBinder == null) {
            synchronized (JarMultiJniApiManager.class) {
                if (getMultiJniApiBinder == null) {
                    ContentResolver resolver = context.getContentResolver();
                    Uri uri = getUri(context);
                    if (uri == null) {
                        return null;
                    }
                    String method = "GetBinder";
                    String arg = "GetMultiJniApiBinder";
                    Bundle bundle = resolver.call(uri, method, arg, null);
                    if (bundle != null) {
                        int ret = bundle.getInt("ret");
                        if (ret == 0) {
                            IBinder binder = bundle.getBinder("Binder");
                            if (binder != null) {
                                getMultiJniApiBinder = IGetMultiJniApi.Stub.asInterface(binder);
                                try {
                                    binder.linkToDeath(new IBinder.DeathRecipient() {
                                        @Override
                                        public void binderDied() {
                                            getMultiJniApiBinder = null;
                                        }
                                    }, 0);
                                } catch (RemoteException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
                }
            }
        }
        return getMultiJniApiBinder;
    }

    public interface ConnectBluetoothKeyCallBack {
        void callBack(int ret, JarJniApiProxy proxy);
    }

    private static ArrayList<String> findList = new ArrayList<>();

    /**
     * 得到Uri
     *
     * @param context
     * @return
     */
    private Uri getUri(Context context) {
        return ForCallProvider.getGetProxyProviderUri(context);
    }
}
