/**
 * Arithmetic.java
 *
 * @author zhangxiaolong@xdja.com
 * 2015-7-9
 */
package com.xdja.multichip.jniapi;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAKey;

public class Arithmetic {
    /**
     * 产生指定长度随机数
     *
     * @param len 要产生数组的长度
     */
    public static byte[] genrateRandomByteArray(int len) {
        byte[] result = new byte[len];
        int i = 0;
        java.util.Random r = new java.util.Random();
        for (i = 0; i < len; i++) {
            result[i] = (byte) (r.nextInt(255) - 127);
        }

        return result;
    }

    /**
     * 得到 X509Certificate
     *
     * @param certContent [in]
     * @param certLen
     * @return
     */
    public static X509Certificate getX509Certificate(byte[] certContent, int certLen) {
        ByteArrayInputStream in = new ByteArrayInputStream(certContent, 0, certLen);

        CertificateFactory certFactory = null;
        X509Certificate x509Cert = null;
        try {
            //创建X509工厂类
            certFactory = CertificateFactory.getInstance("X.509");

            //创建X509Certificate类
            x509Cert = (X509Certificate) certFactory.generateCertificate(in);
        } catch (CertificateException e) {
            e.printStackTrace();
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return x509Cert;
    }

    /**
     * 容器名称 1
     */
    public final static int CONTAINER_NAME = 1;

    /**
     * 交换证书 2
     */
    public final static int EXC_CERT = 2;

    /**
     * 交换公钥 3
     */
    public final static int EXC_PUBKEY = 3;

    /**
     * 交换私钥 4
     */
    public final static int EXC_PRIKEY = 4;

    /**
     * 签名证书 5
     */
    public final static int SIGN_CERT = 5;

    /**
     * 签名公钥 6
     */
    public final static int SIGN_PUBKEY = 6;

    /**
     * 签名私钥 7
     */
    public final static int SIGN_PRIKEY = 7;

    /**
     * 将 容器号 和 类型 转换成 fid
     *
     * @param containNo 容器号
     * @param certType  类型(容器名称、交换证书、交换公钥、交换私钥、签名证书、签名公钥、签名私钥)
     * @return
     */
    public static byte[] convertToFid(int containNo, int certType) {
        byte[] certId = new byte[2];

        certId[0] = 0x00;
        certId[1] = (byte) (0x27 + certType + containNo * 7);

        return certId;
    }


    public static final int CERT_TYPE_RSA1024 = 0;
    public static final int CERT_TYPE_RSA2048 = 1;
    public static final int CERT_TYPE_SM2 = 2;

    /**
     * 得到证书类型
     *
     * @param x509
     * @return 0:RSA-1024
     * 1:RSA-2048
     * 2:SM2
     */
    public static int getCertType(X509Certificate x509) {
        String oid = x509.getSigAlgOID();
        if (oid.equals("1.2.156.197.1.501") || oid.equals("1.2.156.10197.1.501")) {
            return CERT_TYPE_SM2;
        }
        RSAKey publicKey = (RSAKey) x509.getPublicKey();
        int bits = publicKey.getModulus().bitLength();
        if (1024 == bits) {
            return CERT_TYPE_RSA1024;
        } else {
            return CERT_TYPE_RSA2048;
        }
    }


    /**
     * 交换证书 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getExchangeCertFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 1);
        return fid;
    }

    /**
     * 交换公钥 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getExchangePubKeyFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 2);
        return fid;
    }

    /**
     * 交换私钥 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getExchangePriKeyFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 3);
        return fid;
    }

    /**
     * 签名证书 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getSignCertFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 4);
        return fid;
    }

    /**
     * 签名公钥 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getSignPubKeyFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 5);
        return fid;
    }

    /**
     * 签名私钥 fid
     *
     * @param containerNo
     * @return
     */
    public static byte[] getSignPriKeyFid(int containerNo) {
        byte[] fid = new byte[2];
        fid[0] = 0x00;
        fid[1] = (byte) (0x28 + containerNo * 7 + 6);
        return fid;
    }
}