package com.xdja.multichip.jniapi;

import android.content.BroadcastReceiver;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.util.Log;

import com.xdja.jar.forcallprovider.ForCallProvider;
import com.xdja.multichip.ICardStatusBinder;
import com.xdja.multichip.param.ParamKeywords;

import java.util.ArrayList;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_EXCEPTION;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_GET_BINDER_FAIL;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_NO_POWER;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_PARAM_ERROR;


/**
 * @author: zhangxiaolong@xdja.com <br/>
 * date:   2018/5/16 <br/>
 */
public class JarMultiChipStatusManager {
    public static final String TAG = JarMultiChipStatusManager.class.getSimpleName();
    /**
     * 存在
     */
    public static final int CARD_STATUS_EXIST = 1;

    /**
     * 不存在
     */
    public static final int CARD_STATUS_NOT_EXIST = -1;

    public static final int MULTI_CHIP_STATUS_MSG_WHAT_1 = 1;

    private static JarMultiChipStatusManager instance;

    private ICardStatusBinder cardStatusBinder = null;

    private JarMultiChipStatusManager() {

    }

    public static JarMultiChipStatusManager getInstance() {
        if (instance == null) {
            synchronized (JarMultiChipStatusManager.class) {
                if (instance == null) {
                    instance = new JarMultiChipStatusManager();
                }
            }
        }
        return instance;
    }

    /**
     * 发送卡状态
     *
     * @param context
     * @param cardId
     * @param cardType {@link #CARD_STATUS_EXIST},{@link #CARD_STATUS_NOT_EXIST}
     * @param status
     * @return
     */
    public int sendCardStatus(Context context, String cardId, int cardType, int status) {
        if (context == null) {
            return RET_PARAM_ERROR;
        }
        if (!context.getPackageName().equals(ForCallProvider.getCallPkg(context))) {
            return RET_NO_POWER;
        }

        ICardStatusBinder binder = getCardStatusBinder(context);
        if (binder == null) {
            return RET_GET_BINDER_FAIL;
        }
        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_sendSafeCardStatus);
        bundle.putString(ParamKeywords.KEY_String_cardId, cardId);
        bundle.putInt(ParamKeywords.KEY_int_cardType, cardType);
        bundle.putInt(ParamKeywords.KEY_int_status, status);
        try {
            Bundle result = binder.callMethod(bundle);
            if (result != null) {
                int ret = result.getInt(ParamKeywords.KEY_int_ret);
                return ret;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return RET_EXCEPTION;
    }

    private ArrayList<MultiChipStatusListener> listenerList = new ArrayList<>();
    private Messenger messenger = null;

    private void initMessenger(Context context) {
        messenger = new Messenger(new Handler(context.getMainLooper()) {
            @Override
            public void handleMessage(Message msg) {
                super.handleMessage(msg);
                if (msg.what == MULTI_CHIP_STATUS_MSG_WHAT_1) {
                    Bundle bundle = (Bundle) msg.obj;
                    String cardId = bundle.getString(ParamKeywords.KEY_String_cardId);
                    int cardType = bundle.getInt(ParamKeywords.KEY_int_cardType);
                    int status = bundle.getInt(ParamKeywords.KEY_int_status);
                    if (status == CARD_STATUS_EXIST) {
                        notifyListenerChipAdd(cardId, cardType);
                    } else if (status == CARD_STATUS_NOT_EXIST) {
                        notifyListenerChipRemove(cardType);
                    }
                }
            }
        });
    }

    /**
     * 注册监听
     *
     * @param context
     * @param listener
     * @return
     */
    public synchronized int registerListener(Context context, MultiChipStatusListener listener) {
        if (listener == null || context == null) {
            return RET_PARAM_ERROR;
        }
        if (listenerList.contains(listener)) {
            return 0;
        }
        listenerList.add(listener);

        registerReceiveServiceStart(context);

        return registerToService(context);
    }

    /**
     * 想服务端注册
     *
     * @param context
     * @return
     */
    private int registerToService(Context context) {
        if (messenger == null) {
            initMessenger(context);
        }

        ICardStatusBinder binder = getCardStatusBinder(context);
        if (binder == null) {
            return RET_GET_BINDER_FAIL;
        }
        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_registerListener);
        bundle.putBinder(ParamKeywords.KEY_Binder_messenger, messenger.getBinder());
        try {
            Bundle result = binder.callMethod(bundle);
            if (result != null){
                int ret = result.getInt(ParamKeywords.KEY_int_ret);
                return ret;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return RET_EXCEPTION;
    }

    /**
     * 取消注册监听
     *
     * @param listener
     * @return
     */
    public int unregisterListener(MultiChipStatusListener listener) {
        if (listenerList.contains(listener)) {
            listenerList.remove(listener);
        }
        return 0;
    }

    /**
     * 得到Binder代理对象
     *
     * @param context
     * @return
     */
    private ICardStatusBinder getCardStatusBinder(Context context) {
        if (cardStatusBinder == null) {
            synchronized (JarMultiJniApiManager.class) {
                if (cardStatusBinder == null) {
                    ContentResolver resolver = context.getContentResolver();
                    Uri uri = getUri(context);
                    if (uri == null) {
                        return null;
                    }
                    String method = ParamKeywords.KEY_METHOD_GetBinder;
                    String arg = ParamKeywords.KEY_METHOD_CardStatusBinder;
                    Bundle bundle = resolver.call(uri, method, arg, null);
                    if (bundle != null) {
                        int ret = bundle.getInt(ParamKeywords.KEY_int_ret);
                        if (ret == 0) {
                            IBinder binder = bundle.getBinder(ParamKeywords.KEY_Binder_Binder);
                            if (binder != null) {
                                cardStatusBinder = ICardStatusBinder.Stub.asInterface(binder);
                                try {
                                    binder.linkToDeath(new IBinder.DeathRecipient() {
                                        @Override
                                        public void binderDied() {
                                            cardStatusBinder = null;
                                            notifyServiceDeath();
                                        }
                                    }, 0);
                                } catch (RemoteException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
                }
            }
        }
        return cardStatusBinder;
    }

    /**
     * 得到Uri
     *
     * @param context
     * @return
     */
    private Uri getUri(Context context) {
        return ForCallProvider.getGetProxyProviderUri(context);
    }

    /**
     * 注册这样的广播监听，是为了这样的场景：
     * 应用 先向 多芯片卡服务 注册了 状态监听，但多芯片卡服务死掉了。
     * 当多芯片卡服务又启动之后，发一个广播，
     * 这样应用就不需要再主动向多芯片卡服务在Jar包中完成
     *
     * @param context
     */
    private void registerReceiveServiceStart(Context context) {
        try {
            IntentFilter filter = new IntentFilter();
            filter.addAction(ParamKeywords.ACTION_CardStatusServiceStart);
            //modify by zhangxiaolong 2018年7月30日 因为在v890手机上，会崩溃
            //改为getReceiver 和 加上 try catch
            context.registerReceiver(getReceiver(), filter);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private BroadcastReceiver receiver;

    private synchronized BroadcastReceiver getReceiver() {
        if (receiver == null) {
            receiver = new BroadcastReceiver() {
                @Override
                public void onReceive(Context context, Intent intent) {
                    Log.e(TAG, "receiver: " + intent);
                    if (listenerList != null && listenerList.size() > 0) {
                        registerToService(context);
                    }
                    notifyServiceStart();
                }
            };
        }
        return receiver;
    }


    private void notifyListenerChipAdd(String cardId, int cardType) {
        for (MultiChipStatusListener listener : listenerList) {
            listener.chipAdd(cardId, cardType);
        }
    }

    private void notifyListenerChipRemove(int cardType) {
        for (MultiChipStatusListener listener : listenerList) {
            listener.chipRemove(cardType);
        }
    }

    private void notifyServiceDeath() {
        for (MultiChipStatusListener listener : listenerList) {
            listener.serviceDeath();
        }
    }

    private void notifyServiceStart() {
        for (MultiChipStatusListener listener : listenerList) {
            listener.serviceStart();
        }
    }

    public interface MultiChipStatusListener {
        /**
         * 有芯片增加
         *
         * @param cardId
         * @param cardType {@link com.xdja.multichip.param.JniApiParam#TYPE_BLUETOOTH},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_COVERED},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_ONBOARD},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_TF}
         */
        void chipAdd(String cardId, int cardType);

        /**
         * 有芯片卸载
         *
         * @param cardType {@link com.xdja.multichip.param.JniApiParam#TYPE_BLUETOOTH},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_COVERED},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_ONBOARD},
         *                 {@link com.xdja.multichip.param.JniApiParam#TYPE_TF}
         */
        void chipRemove(int cardType);

        /**
         * 远端服务死掉
         */
        void serviceDeath();

        /**
         * 远端服务启动了
         */
        void serviceStart();
    }
}
