package com.xdja.multichip.jniapi;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.jar.forcallprovider.ForCallProvider;
import com.xdja.multichip.IGetMultiJniApi;
import com.xdja.multichip.IMultiJniApi;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.param.ParamKeywords;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_EXCEPTION;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_GET_BINDER_FAIL;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_MAKE_FAILE;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_PARAM_ERROR;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/7/24 <br/>
 */
public class JarMultiJniApiManager {

    public static final int RET_OK = 0;

    private static JarMultiJniApiManager instance;
    private IGetMultiJniApi getMultiJniApiBinder = null;

    private HashMap<String, IBinder> cardIdBinderMap = new HashMap<>();
    private HashMap<IBinder, JarJniApiProxy> binderProxyMap = new HashMap<>();
    /**
     * 通过 {@link #make(Context, int)} 生成的 {@link JarJniApiTypeProxy} 的map；
     * 其中Key是卡类型
     */
    private HashMap<Integer, JarJniApiProxy> typeProxyMap = new HashMap<>();

    /**
     * 卡号 和 卡类型 对应的map<br>
     * key： 卡号<br>
     * value： 卡类型{@link JniApiParam#TYPE_ONBOARD},
     * {@link JniApiParam#TYPE_TF},
     * {@link JniApiParam#TYPE_COVERED},
     * {@link JniApiParam#TYPE_BLUETOOTH}
     */
    private HashMap<String, Integer> cardIdTypeMap = new HashMap<>();

    private JarMultiJniApiManager() {

    }

    public static JarMultiJniApiManager getInstance() {
        if (instance == null) {
            synchronized (JarMultiJniApiManager.class) {
                if (instance == null) {
                    instance = new JarMultiJniApiManager();
                }
            }
        }
        return instance;
    }

    /**
     * 得到设备上所有的卡的相关参数
     *
     * @param context
     * @return
     */
    public Pair<Integer, List<JniApiParam>> getAll(Context context) {
        if (context == null) {
            return Pair.create(RET_PARAM_ERROR, null);
        }
        IGetMultiJniApi multiBinder = getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return Pair.create(RET_GET_BINDER_FAIL, null);
        }


        //modify by zhangxiaolong 2019-1-24   --------------start
        //在北京市局项目mate8上，会出现在调用过程中远程服务被kill的情况，
        //如果出现这种情况，重试3次。
        boolean flagException = false;
        List<Bundle> result = null;
        for (int i = 0; i < 3; i++) {
            try {
                result = multiBinder.getMultiJniApi();
                flagException = false;
            } catch (RemoteException e) {
                e.printStackTrace();
                flagException = true;
                SystemClock.sleep(30);
            }
            if (!flagException) {
                break;
            }
        }
        if (flagException) {
            return Pair.create(RET_EXCEPTION, null);
        }
        //modify by zhangxiaolong 2019-1-24   --------------end

        List<JniApiParam> list = new ArrayList<>();
        if (result == null || result.size() == 0) {
            return Pair.create(RET_OK, list);
        }
        for (Bundle bundle : result) {
            bundle.setClassLoader(JniApiParam.class.getClassLoader());
            JniApiParam jniApiParam = bundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
            final IBinder binder = bundle.getBinder(ParamKeywords.KEY_Binder_Binder);
            final String cardId = jniApiParam.cardId.trim();
            cardIdBinderMap.put(cardId, binder);
            cardIdTypeMap.put(cardId, jniApiParam.chipType);

            //add 2018年11月19日14:29:26 持久化类型与卡号对应关系
            CardTypeUtil.saveCardIdAndType(context, cardId, jniApiParam.chipType);

            binderLinkToDeath(binder, cardId, jniApiParam.chipType);
            list.add(jniApiParam);
        }
        return Pair.create(0, list);


    }

    /**
     * 根据类型获取代理
     *
     * @param context
     * @param cardType
     * @return
     */
    public Pair<Integer, JarJniApiProxy> make(Context context, final int cardType) {
        if (context == null) {
            return Pair.create(RET_PARAM_ERROR, null);
        }
        if (cardType != JniApiParam.TYPE_ONBOARD &&
                cardType != JniApiParam.TYPE_TF &&
                cardType != JniApiParam.TYPE_COVERED &&
                cardType != JniApiParam.TYPE_BLUETOOTH &&
                cardType != JniApiParam.TYPE_VHSM &&
                cardType != JniApiParam.TYPE_VHSM_NET) {
            return Pair.create(RET_PARAM_ERROR, null);
        }

        if (typeProxyMap.containsKey(cardType)) {
            return Pair.create(0, typeProxyMap.get(cardType));
        }

        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_makeByType);
        bundle.putInt(ParamKeywords.KEY_int_cardType, cardType);
        IGetMultiJniApi multiBinder = getGetMultiJniApiBinder(context);
        if (multiBinder == null) {
            return Pair.create(RET_GET_BINDER_FAIL, null);
        }
        try {
            Bundle result = multiBinder.callMethod(bundle);
            if (result != null) {
                int ret = result.getInt(ParamKeywords.KEY_int_ret);
                if (ret != 0) {
                    return Pair.create(ret, null);
                }
                ArrayList<Bundle> bundleList = result.getParcelableArrayList(ParamKeywords.KEY_ParcelableArrayList_Bundle);
                if (bundleList == null || bundleList.size() == 0) {
                    return Pair.create(RET_MAKE_FAILE, null);
                }
                Bundle typeBundle = bundleList.get(0);
                typeBundle.setClassLoader(JniApiParam.class.getClassLoader());

                JniApiParam jniApiParam = typeBundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
                final IBinder binder = typeBundle.getBinder(ParamKeywords.KEY_Binder_Binder);
                JarJniApiTypeProxy proxy = new JarJniApiTypeProxy(context, IMultiJniApi.Stub.asInterface(binder), jniApiParam.chipType);
                typeProxyMap.put(cardType, proxy);

                //add 2018年11月19日14:29:26 持久化类型与卡号对应关系
                final String cardId = jniApiParam.cardId.trim();
                cardIdBinderMap.put(cardId, binder);
                cardIdTypeMap.put(cardId, jniApiParam.chipType);

                CardTypeUtil.saveCardIdAndType(context, jniApiParam.cardId.trim(), jniApiParam.chipType);

                binderLinkToDeath(binder, cardId, cardType);
                return Pair.create(0, (JarJniApiProxy) proxy);
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return Pair.create(RET_EXCEPTION, null);
    }

    /**
     * 得到操作卡的对象;
     *
     * @param context
     * @param cardId
     * @return
     */
    public Pair<Integer, JarJniApiProxy> make(Context context, String cardId) {
        if (context == null || TextUtils.isEmpty(cardId)) {
            return Pair.create(RET_PARAM_ERROR, null);
        }

        cardId = cardId.trim();
        IBinder binder = cardIdBinderMap.get(cardId);
        if (binder == null) {

            //add 2018年11月19日14:33:50 新增首先获取卡号对应的类型，减少调用getAll
            int cardType = CardTypeUtil.getCardType(context, cardId);
            if (cardType == -1) {
                getAll(context);
            } else {
                make(context, cardType);
            }

            binder = cardIdBinderMap.get(cardId);
        }

        if (binder != null) {
            XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
            try {
                int ret = IMultiJniApi.Stub.asInterface(binder).GetDevInfo(0, devinfo);
                if (ret == 0) {
                    String getCardId = new String(devinfo.cardid).trim();
                    if (cardId.equals(getCardId)) {
                        if (binderProxyMap.containsKey(binder)) {
                            return Pair.create(0, binderProxyMap.get(binder));
                        } else {
                            int cardType = cardIdTypeMap.get(getCardId);
                            JarJniApiProxy jniApiJarProxy = new JarJniApiProxy(context, IMultiJniApi.Stub.asInterface(binder), getCardId, cardType);
                            binderProxyMap.put(binder, jniApiJarProxy);
                            return Pair.create(0, jniApiJarProxy);
                        }
                    } else {
                        cardIdBinderMap.remove(cardId);
                        binderProxyMap.remove(binder);
                    }
                }
            } catch (RemoteException e) {
                e.printStackTrace();
                return Pair.create(RET_EXCEPTION, null);
            }
            return Pair.create(RET_MAKE_FAILE, null);
        }
        return Pair.create(RET_GET_BINDER_FAIL, null);
    }

    HashMap<String, IBinder> getCardIdBinderMap() {
        return cardIdBinderMap;
    }

    HashMap<IBinder, JarJniApiProxy> getBinderProxyMap() {
        return binderProxyMap;
    }

    /**
     * 得到Binder代理对象
     *
     * @param context
     * @return
     */
    IGetMultiJniApi getGetMultiJniApiBinder(Context context) {
        if (getMultiJniApiBinder == null) {
            synchronized (JarMultiJniApiManager.class) {
                if (getMultiJniApiBinder == null) {
                    ContentResolver resolver = context.getContentResolver();
                    Uri uri = getUri(context);
                    if (uri == null) {
                        return null;
                    }
                    String method = "GetBinder";
                    String arg = "GetMultiJniApiBinder";
                    Bundle bundle = resolver.call(uri, method, arg, null);
                    if (bundle != null) {
                        int ret = bundle.getInt("ret");
                        if (ret == 0) {
                            IBinder binder = bundle.getBinder("Binder");
                            if (binder != null) {
                                getMultiJniApiBinder = IGetMultiJniApi.Stub.asInterface(binder);
                                try {
                                    binder.linkToDeath(new IBinder.DeathRecipient() {
                                        @Override
                                        public void binderDied() {
                                            getMultiJniApiBinder = null;
                                        }
                                    }, 0);
                                } catch (RemoteException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
                }
            }
        }
        return getMultiJniApiBinder;
    }

    public interface ConnectBluetoothKeyCallBack {
        void callBack(int ret, JarJniApiProxy proxy);
    }

    /**
     * 得到Uri
     *
     * @param context
     * @return
     */
    private Uri getUri(Context context) {
        return ForCallProvider.getGetProxyProviderUri(context);
    }

    /**
     * 死亡通知
     *
     * @param binder
     * @param cardId
     * @param cardType
     */
    private void binderLinkToDeath(final IBinder binder, final String cardId, final int cardType) {
        ProxyToDeathBean linkToDeathBean = obtain(binder,cardId,cardType);
        if (!linkToDeathBeans.contains(linkToDeathBean)){
            linkToDeathBeans.add(linkToDeathBean);
        }
    }

    private HashMap<String, ProxyToDeathBean> proxyToDeathBeanMap = new HashMap<>();

    private ProxyToDeathBean obtain(final IBinder binder, final String cardId, final int cardType) {
        String key = binder.toString() + cardId;
        if (proxyToDeathBeanMap.containsKey(key)) {
            return proxyToDeathBeanMap.get(key);
        }
        ProxyToDeathBean linkToDeathBean = new ProxyToDeathBean(binder, cardId, cardType);
        proxyToDeathBeanMap.put(key, linkToDeathBean);
        return linkToDeathBean;
    }

    ArrayList<ProxyToDeathBean> linkToDeathBeans = new ArrayList<>();

    private void remove(final IBinder binder, final String cardId, final int cardType) {
        cardIdBinderMap.remove(cardId);
        typeProxyMap.remove(cardType);
        binderProxyMap.remove(binder);
        String key = binder.toString() + cardId;
        proxyToDeathBeanMap.remove(key);
    }

    class ProxyToDeathBean {
        IBinder binder;
        String cardId;
        int cardType;
        Messenger messenger;

        ProxyToDeathBean(final IBinder binder, final String cardId, final int cardType) {
            this.binder = binder;
            try {
                this.binder.linkToDeath(new IBinder.DeathRecipient() {
                    @Override
                    public void binderDied() {
                        remove(binder, cardId, cardType);
                        linkToDeathBeans.remove(ProxyToDeathBean.this);
                    }
                }, 0);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
            this.cardId = cardId;
            this.cardType = cardType;
            messenger = new Messenger(new Handler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message msg) {
                    super.handleMessage(msg);
                    Log.d("clientProxy", "Client handleMessage what = " + msg.what);
                    if (msg.what == ParamKeywords.FLAG_MSG_SERVER_DEATH) {
                        remove(binder, cardId, cardType);
                        linkToDeathBeans.remove(ProxyToDeathBean.this);
                    }
                }
            });
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_SendClientProxy);
            bundle.putBinder(ParamKeywords.KEY_Binder_messenger, messenger.getBinder());
            try {
                IMultiJniApi.Stub.asInterface(binder).callMethod(bundle);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }
    }

}
