package com.xdja.uniteauth.jar;

import android.app.Activity;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Process;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2018/7/31 <br/>
 * 第三方应用调用所使用的类
 */
public class UniteAuth {

    private final static int JAR_VERSION = 1;
    private Uri uri;
    private IUniteAuth uniteAuthBinder;

    private UniteAuth(Uri uri) {
        this.uri = uri;
    }

    /**
     * 获取授权码
     *
     * @param context   上下文
     * @param randomStr 由第三方应用生成的随机数，不能为空，不能大于512个字节
     * @param listener  回调是在主线程
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    public int getAuthenticationCode(Context context, String randomStr, ICallbackListener listener) {
        return getAuthenticationCode(context,
                randomStr,
                null,
                -1,
                listener);
    }

    /**
     * 获取授权码
     *
     * @param context   上下文
     * @param randomStr 由第三方应用生成的随机数，不能为空，不能大于512个字节
     * @param cardId    卡号
     * @param listener  回调是在主线程
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    public int getAuthenticationCode(Context context, String randomStr, String cardId, ICallbackListener listener) {
        if (TextUtils.isEmpty(cardId)) {
            throw new NullPointerException("cardId is null");
        }
        return getAuthenticationCode(context,
                randomStr,
                cardId,
                -1,
                listener);
    }

    /**
     * 获取授权码
     *
     * @param context   上下文
     * @param randomStr 由第三方应用生成的随机数，不能为空，不能大于512个字节
     * @param cardType  卡类型，可参看jar_multi_jniapi.jar中的
     *                  com.xdja.multichip.param.JniApiParam类中的
     *                  TYPE_ONBOARD、TYPE_TF、TYPE_BLUETOOTH、TYPE_COVERED
     * @param listener  回调是在主线程
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    public int getAuthenticationCode(Context context, String randomStr, int cardType, ICallbackListener listener) {
        if (cardType <= 0) {
            return ErrorCode.RET_NOT_SUPPORT_CARDTYPE;
        }

        return getAuthenticationCode(context,
                randomStr,
                null,
                cardType,
                listener);
    }

    /**
     * 内部使用方法
     *
     * @param context
     * @param randomStr
     * @param cardId
     * @param cardType
     * @param listener
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    private int getAuthenticationCode(Context context,
                                      String randomStr,
                                      String cardId,
                                      int cardType,
                                      ICallbackListener listener) {
        if (context == null) {
            throw new NullPointerException("context is null");
        }

        if (listener == null) {
            throw new NullPointerException("listener is null");
        }

        if (TextUtils.isEmpty(randomStr)) {
            randomStr = "";
        }

        try {
            IUniteAuth binder = getBinder(context);
            if (binder == null) {
                return ErrorCode.RET_CANNOT_CALL_SERVER;
            }
            Bundle bundle = BundleHandle.getBundleForGetAuthCode(randomStr, cardId, cardType, listener);
            Bundle result = binder.call(bundle);
            int ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (ArithmeticException e) {
            try {
                int ret = Integer.valueOf(e.getMessage());
                return ret;
            } catch (Exception exception) {
                return ErrorCode.RET_EXCEPTION;
            }
        } catch (RemoteException e) {
            e.printStackTrace();
            return ErrorCode.RET_EXCEPTION;
        }
    }

    /**
     * 得到Binder。
     * 如果返回null，表示无法调用到binder server端。
     *
     * @param context
     * @return
     */
    private synchronized IUniteAuth getBinder(Context context) {
        if (uniteAuthBinder != null) {
            return uniteAuthBinder;
        }
        if (context == null) {
            throw new NullPointerException("context is null");
        }

        ContentResolver resolver = context.getContentResolver();
        Bundle extra = new Bundle();
        extra.putInt(ParamKeywords.KEY_int_version, JAR_VERSION);
        Bundle bundle;
        try{
            bundle = resolver.call(uri, ParamKeywords.KEY_METHOD_GetBinder, ParamKeywords.KEY_METHOD_UniteAuthBinder, extra);
        }catch (IllegalArgumentException e){
            return null;
        }
        if (bundle == null) {
            return null;
        }
        int ret = bundle.getInt(ParamKeywords.KEY_int_ret);
        if (ret != 0) {
            throw new ArithmeticException("" + ret);
        }
        IBinder binder = bundle.getBinder(ParamKeywords.KEY_Binder_Binder);
        if (binder == null) {
            return null;
        }
        uniteAuthBinder = IUniteAuth.Stub.asInterface(binder);
        try {
            binder.linkToDeath(new IBinder.DeathRecipient() {
                @Override
                public void binderDied() {
                    uniteAuthBinder = null;
                }
            }, 0);
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return uniteAuthBinder;
    }

    /**
     * 创建者
     */
    public static final class Builder {
        private Uri uri;
        private static final Uri URI_DEFAULT = Uri.parse("content://com.xdja.uniteauth.UniteAuthProvider");

        public Builder baseUri(Uri uri) {
            if (uri == null) {
                throw new NullPointerException("uri is null!");
            }
            this.uri = uri;
            return this;
        }

        /**
         * 创建
         *
         * @return 实例
         */
        public UniteAuth create() {
            if (uri == null) {
                uri = URI_DEFAULT;
            }
            return new UniteAuth(uri);
        }
    }

    /**
     * 获取授权码
     *
     * @param context   上下文
     * @param randomStr 由第三方应用生成的随机数，不能为空，不能大于512个字节
     * @param listener  回调是在主线程
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    public int getAuthenticationCodeSync(Context context, String randomStr, ICallbackListener listener) {
        return getAuthenticationCodeByStartActivity(context,
                randomStr,
                null,
                -1,
                listener);
    }


    /**
     * 内部使用方法
     *
     * @param context
     * @param randomStr
     * @param cardId
     * @param cardType
     * @param listener
     * @return 0:表示已经调用到了Binder服务端；
     * 还会有的返回值为：
     * {@link ErrorCode#RET_EXCEPTION},
     * {@link ErrorCode#RET_CANNOT_CALL_SERVER},
     * {@link ErrorCode#RET_NO_POWER}
     */
    private int getAuthenticationCodeByStartActivity(Context context,
                                      String randomStr,
                                      String cardId,
                                      int cardType,
                                      ICallbackListener listener) {
        if (context == null) {
            throw new NullPointerException("context is null");
        }

        if (listener == null) {
            throw new NullPointerException("listener is null");
        }

        if (TextUtils.isEmpty(randomStr)) {
            randomStr = "";
        }


        try {
            IUniteAuth binder = getBinder(context);
            if (binder == null) {
                return ErrorCode.RET_CANNOT_CALL_SERVER;
            }

            //1.首先获取当前统一认证aar所在的包名
            Bundle bundle4GetPackageName = new Bundle();
            bundle4GetPackageName.putString(ParamKeywords.KEY_METHOD, ParamKeywords.KEY_METHOD_getAarPackageName);
            Bundle result = binder.call(bundle4GetPackageName);
            int getPackageName_ret = result.getInt(ParamKeywords.KEY_int_ret,-1);
            String arrPackageName = result.getString(ParamKeywords.KEY_String_pkgName);

            if(getPackageName_ret !=0 || TextUtils.isEmpty(arrPackageName)){
                Log.d("AuthJar", "arrPackageName is null");
                return ErrorCode.RET_CANNOT_CALL_SERVER;
            }

            //2.启动统一认证界面
            Bundle bundle = BundleHandle.getBundleForGetAuthCode(randomStr, cardId, cardType, listener);

            int uid = Process.myUid();
            int pid = Process.myPid();
            String pkgName = context.getPackageName();
            if (TextUtils.isEmpty(pkgName)) {
                return ErrorCode.RET_NO_POWER;
            }

            bundle.putInt(ParamKeywords.KEY_int_uid, uid);
            bundle.putInt(ParamKeywords.KEY_int_pid, pid);
            bundle.putString(ParamKeywords.KEY_String_pkgName, pkgName);


            Intent intent = new Intent();
            intent.setClassName(arrPackageName, "com.xdja.uniteauth.presenter.AuthLoginActivity");
            intent.setPackage(arrPackageName);
            intent.putExtra(ParamKeywords.KEY_Bundle_bundle,bundle);
            if (!(context instanceof Activity)) {
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            }
            context.startActivity(intent);

            return 0;
        } catch (ArithmeticException e) {
            try {
                int ret = Integer.valueOf(e.getMessage());
                return ret;
            } catch (Exception exception) {
                return ErrorCode.RET_EXCEPTION;
            }
        } catch (RemoteException e) {
            e.printStackTrace();
            return ErrorCode.RET_EXCEPTION;
        }
    }

}
