package com.rabbit.blade.compiler;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.Filer;
import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.Elements;
import javax.tools.Diagnostic;


public abstract class BladeTypeProcessor<T extends BladeTypeElement> extends AbstractProcessor {

    private Filer mFileUtils;
    private Elements elementUtils;
    private Messager messager;

    private final List<T> elements = new ArrayList<>();

    public BladeTypeProcessor() {
        super();
    }

    @Override
    public Set<String> getSupportedAnnotationTypes() {
        Set<String> supportedAnnotationTypes = new HashSet<>();
        supportedAnnotationTypes.add(getAnnotationClass().getCanonicalName());
        return supportedAnnotationTypes;
    }

    @Override
    public SourceVersion getSupportedSourceVersion() {
        return SourceVersion.latestSupported();
    }

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        mFileUtils = processingEnv.getFiler();
        elementUtils = processingEnv.getElementUtils();
        messager = processingEnv.getMessager();
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {

        this.elements.clear();

        Set<? extends Element> foundElements = roundEnv.getElementsAnnotatedWith(getAnnotationClass());
        try {
            for (Element element : foundElements) {
                if (element.getKind() != getElementKind()) {
                    error(element, "Only @%s  can be annotated with @%s", getElementKind(), getAnnotationClass().getSimpleName());
                    return true;
                }

                T t = getElementClass()
                        .getConstructor(TypeElement.class)
                        .newInstance(((TypeElement) element));
                this.elements.add(t);
            }

            for (T ele : this.elements) {
                if (ele != null) {
                    ele.generateCode(elementUtils, mFileUtils);
                }
            }

        }catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            e.printStackTrace();
            return true;
        } catch (InstantiationException e) {
            if (e.getCause() instanceof ProcessingException) {
                ProcessingException cause = (ProcessingException) e.getCause();
                error(cause.getElement(), cause.getMessage());
            }
            e.printStackTrace();
            return true;
        }
        return false;
    }

    private void error(Element e, String msg, Object... args) {
        messager.printMessage(Diagnostic.Kind.ERROR, String.format(msg, args), e);
    }

    public abstract Class<? extends Annotation> getAnnotationClass();

    public abstract ElementKind getElementKind();

    public abstract Class<T> getElementClass();
}
