package com.rabbit.blade.compiler.split;

import com.rabbit.blade.api.split.Forked;
import com.rabbit.blade.compiler.BladeTypeElement;
import com.rabbit.blade.compiler.ProcessingException;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeSpec;

import java.io.IOException;

import javax.annotation.processing.Filer;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.Elements;

/**
 * <b>Description: </b>
 * Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/11/21 9:34.
 */

public class ForkedElement extends BladeTypeElement {

    private final String tag;
    private final int priority;
    private final boolean sticky;

    public ForkedElement(TypeElement typeElement) throws ProcessingException {
        super(typeElement);

        Forked forkedAnn = typeElement.getAnnotation(Forked.class);
        this.tag = forkedAnn.tag();
        this.priority = forkedAnn.priority();
        this.sticky = forkedAnn.sticky();

        if (this.tag.equals("")) {
            throw new ProcessingException(typeElement,
                    "tag() in @%s for class %s is Empty! that's not allowed",
                    Forked.class.getSimpleName(), typeElement.getQualifiedName().toString());
        }
    }

    @Override
    public void generateCode(Elements elementUtils, Filer filer) {
        PackageElement pkg = elementUtils.getPackageOf(this.getTypeElement());
        String packageName = pkg.isUnnamed() ? null : pkg.getQualifiedName().toString();

        if (packageName != null && !packageName.equals("")) {
            ClassName forkedMakerCN = ClassName.get(packageName, getTypeElement().getSimpleName() + "Processor");
            ClassName autoServiceCN = ClassName.get("com.google.auto.service", "AutoService");

            AnnotationSpec.Builder autoAnnSpeBuilder = AnnotationSpec.builder(autoServiceCN).addMember("value","com.rabbit.blade.comm.util.Builder.class");

            TypeSpec.Builder makerBuilder = TypeSpec
                    .classBuilder(forkedMakerCN)
                    .addSuperinterface(ClassName.get("com.rabbit.blade.comm.util","Builder"))
                    .addModifiers(Modifier.PUBLIC)
                    .addAnnotation(autoAnnSpeBuilder.build());

            ClassName boundedAppName = ClassName.get(getTypeElement());
            String name = lowerFirstLetter(boundedAppName.simpleName());
            FieldSpec.Builder boundedAppFieldBuilder =
                    FieldSpec.builder(boundedAppName, name)
                            .addModifiers(Modifier.PUBLIC)
                            .addModifiers(Modifier.FINAL);
            makerBuilder.addField(boundedAppFieldBuilder.build());

            MethodSpec.Builder boundedConstructMethodBuilder = MethodSpec
                    .constructorBuilder()
                    .addModifiers(Modifier.PUBLIC)
                    .addStatement("this.$L = new $L()", name, boundedAppName);
            makerBuilder.addMethod(boundedConstructMethodBuilder.build());

            MethodSpec.Builder buildMethodBuilder = MethodSpec
                    .methodBuilder("build")
                    .addModifiers(Modifier.PUBLIC)
                    .addAnnotation(Override.class)
                    .addStatement("com.rabbit.blade.presenter.split.ForkService.registerBoundedApplication($L,this.$L)", "\"" + tag + "\"", name)
                    .addStatement("com.rabbit.blade.comm.event.EventBusProvider.instance().register(this)");
            makerBuilder.addMethod(buildMethodBuilder.build());

            ParameterSpec.Builder eventParameterBuilder = ParameterSpec
                    .builder(ClassName.get("com.rabbit.blade.presenter.split", "ForkedEvent"), "forkedEvent");
            AnnotationSpec.Builder subscriberAnnBuilder = AnnotationSpec
                    .builder(ClassName.get("org.greenrobot.eventbus", "Subscribe"))
                    .addMember("threadMode", "org.greenrobot.eventbus.ThreadMode.MAIN")
                    .addMember("sticky", String.valueOf(sticky))
                    .addMember("priority", String.valueOf(priority));

            MethodSpec.Builder handleMethodBuilder = MethodSpec
                    .methodBuilder("handleForkedEvent")
                    .addModifiers(Modifier.PUBLIC)
                    .addParameter(eventParameterBuilder.build())
                    .addAnnotation(subscriberAnnBuilder.build())
                    .addStatement("boolean result = com.rabbit.blade.presenter.split.ForkService.checkForkedEventSubscriber(getClass())")
                    .beginControlFlow("if (result)")
                    .addStatement("com.rabbit.blade.presenter.split.ForkService.handleForkedEvent(this.$L,forkedEvent)", name)
                    .endControlFlow();
            makerBuilder.addMethod(handleMethodBuilder.build());

            try {
                JavaFile.builder(packageName, makerBuilder.build()).build().writeTo(filer);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

}
