package com.xdja.aircert.sdk.device;

import android.text.TextUtils;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_FILE;
import com.xdja.aircert.sdk.AirCertSdk;
import com.xdja.aircert.sdk.bean.CertDetailInfoBean;
import com.xdja.aircert.sdk.cert.CertSKF;
import com.xdja.aircert.sdk.config.CertRule;
import com.xdja.aircert.sdk.config.CertType;
import com.xdja.aircert.sdk.config.XDJAAlgParams;
import com.xdja.aircert.sdk.util.FidUtil;
import com.xdja.aircert.sdk.util.SdkLog;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.multichip.param.CertBean;

import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;

import static com.xdja.aircert.sdk.device.DeviceWrapper.getCryptoInstance;

/**
 * @author: lyl
 * desc:读写卡操作
 * 2020/11/10
 */
public class OperateCertHelper {
    private static final int RSA_ALG_BITS = XDJAAlgParams.RSA_KEY_LEN_2048;
    /**
     * 将SKF结构的数据写入卡
     *
     * @param type    设备类型
     * @param certStr SKF结构数据
     * @return -4 标识服务端跟客户端的证书机制不一致
     */
    public int importCert(CryptoDevType type, int containerNum, String certStr) {

        String certType = AirCertSdk.getsInstance().getAirCertConfig().getCertType();

        String certRule = CertHelper.getCertRule(AirCertSdk.getsInstance().getAirCertConfig().getAlgType());


//        Log.e("jff", "522 DeviceWrapper importCert : ");
        CertSKF certSKF = splitCertStr(type, containerNum, certStr);
        if (certSKF == null) {
            return -1;
        }
        if (certSKF.getErrorCode() == 4) {
            return -4;
        }
        String pin = CertHelper.getPin();
        int role = CertHelper.getRole(AirCertSdk.getsInstance().getAirCertConfig().getContainerNum());
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        CertBean bean = null;

        if (CertHelper.getAlgType().equals(XDJAAlgParams.CA_ALG_RSA)) {
            switch (CertHelper.getCertRule(AirCertSdk.getsInstance().getAirCertConfig().getAlgType())) {
                case CertRule.SINGLE:
                    if (AirCertSdk.getsInstance().getAirCertConfig().getCertType().equals(CertType.SIGNING)
                            || TextUtils.isEmpty(AirCertSdk.getsInstance().getAirCertConfig().getCertType())) {
                        bean = CertBean.createSingleRSASignCert(role, pin, containerNum, certSKF.getSignCert());
                    } else {
                        bean = CertBean.createSingleRSAExchangeCert(role, pin, containerNum, certSKF.getEnCodeCert());
                    }
                    break;
                default:
                    break;
            }
        } else {
            switch (CertHelper.getCertRule(AirCertSdk.getsInstance().getAirCertConfig().getAlgType())) {
                case CertRule.SINGLE:
                    if (AirCertSdk.getsInstance().getAirCertConfig().getCertType().equals(CertType.SIGNING)
                            || TextUtils.isEmpty(AirCertSdk.getsInstance().getAirCertConfig().getCertType())) {

                        bean = CertBean.createSingleSM2SignCert(role, pin, containerNum, certSKF.getSignCert());
                    } else {
                        bean = CertBean.createSingleSM2ExchangeCert(role, pin, containerNum, certSKF.getEnCodeCert());
                    }
                    break;
                case CertRule.DOUBLE_CERT:
                    bean = CertBean.createDoubleSM2CertSkf(role, pin, containerNum, certSKF.getSignCert(), certSKF.getEnCodeCert(), certSKF.getKeyPair());
                    break;
                default:
                    break;
            }
        }

//        ModuleLog.e("import cert = " + bean.toString());
        int importCertRet = cryptoInstance.importCert(bean);

        if (importCertRet != 0) {
            return importCertRet;
        }

        int writeProjectLabelRet = writeProjectLabel(type, certSKF.getProjectLabel());
        if (writeProjectLabelRet != 0) {
            return writeProjectLabelRet;
        } else {
            return 0;
        }
    }

    private int writeProjectLabel(CryptoDevType type, String labStr) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        byte[] labStrBytes = labStr.getBytes();
        int writeLabel = -1;

        int createRet = createFile(type);
        if (createRet == 0 || createRet == 2033) {//创建文件成功或者文件已存在
            writeLabel = cryptoInstance.writeFile(labelFid, 0, 64, labStrBytes);
            //            ModuleLog.e("写标签内容：= " + labSm3Str.getBytes());
        }

        return writeLabel;
    }

    private int createFile(CryptoDevType type) {
        byte[] labelFid = new byte[]{0x00, (byte) 0x9f};

        XDJA_FILE file = new XDJA_FILE();
        file.read_Acl = (byte) 0xf1;
        file.write_Acl = (byte) 0xf1;
        file.room = 0x200;
        file.type = JNIAPI.FILE_BINARY;
        file.id = labelFid;


        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }

        int ret = cryptoInstance.createFile(file);
        //        ModuleLog.e("创建ret " + ret + " type " + type);
        return ret;
    }

    private CertSKF splitCertStr(CryptoDevType type, int containerNum, String certStr) {

//        ModuleLog.e(certStr);

        if (certStr.equals("") || certStr.length() <= 0) {
            return null;
        }
        //        ModuleLog.e("certStr " + certStr);

        //“cert”: “签名证书Base64#加密证书Base64#加密密钥” //证书信息
        //        certStr = certStr.replaceAll("\r", "");
        //        certStr = certStr.replaceAll("\n", "");
        certStr = certStr.replace("-----BEGIN CERTIFICATE-----", "").replace("-----END CERTIFICATE-----", "");
        certStr = certStr.replace("\r", "").replace("\n", "");
        certStr = certStr.replace("\\r", "").replace("\\n", "");

        //        certStr = certStr.replaceAll("-----BEGIN CERTIFICATE-----", "");
        //        certStr = certStr.replaceAll("-----END CERTIFICATE-----", "");
        String[] pieces = certStr.split("#");
        //        ModuleLog.e("cert分隔后数据长度 " + pieces.length);


//        for (int i = 0; i <= pieces.length - 1; i++) {
//            ModuleLog.e("第" + (i + 1) + "段\n" + pieces[i]);
//        }

        CertSKF certSKF = new CertSKF();

        String certRule = CertHelper.getCertRule(AirCertSdk.getsInstance().getAirCertConfig().getAlgType());

        //如果配置项证书个数跟服务端返回的数据长度不一致
        //todo jff 之后做证书更新这一块应该做相应的变更
        if ((certRule.equals(CertRule.SINGLE) && pieces.length != 2)
                || (certRule.equals(CertRule.DOUBLE_CERT) && pieces.length != 4)) {
            certSKF.setErrorCode(4);
        }

        switch (pieces.length) {
            case 2:  //加密证书base64#项目标签（单证/单证更新延期）
                if (!TextUtils.isEmpty(pieces[0])) {
                    if (TextUtils.isEmpty(AirCertSdk.getsInstance().getAirCertConfig().getCertType())
                            || AirCertSdk.getsInstance().getAirCertConfig().getCertType().equals(CertType.SIGNING)) {

                        certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                    } else if (AirCertSdk.getsInstance().getAirCertConfig().getCertType().equals(CertType.ENCRYPT)) {
                        certSKF.setEnCodeCert(Base64.decode(pieces[0].getBytes()));
                    }
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setProjectLabel(pieces[1]);
                }
                certSKF.setLen(2);
                break;
            case 3://签名证书base64#加密证书base64#项目标签（双证无私钥）
                if (!TextUtils.isEmpty(pieces[0])) {
                    certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setEnCodeCert(Base64.decode(pieces[1].getBytes()));
                }

                if (!TextUtils.isEmpty(pieces[2])) {
                    certSKF.setProjectLabel(pieces[2]);
                }
                certSKF.setLen(3);
            case 4://签名证书base64#加密证书base64#私钥结构体#项目标签
                if (!TextUtils.isEmpty(pieces[0])) {
                    certSKF.setSignCert(Base64.decode(pieces[0].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[1])) {
                    certSKF.setEnCodeCert(Base64.decode(pieces[1].getBytes()));
                }

                if (!TextUtils.isEmpty(pieces[2])) {
                    certSKF.setKeyPair(Base64.decode(pieces[2].getBytes()));
                }
                if (!TextUtils.isEmpty(pieces[3])) {
                    certSKF.setProjectLabel(pieces[3]);
                }
                certSKF.setLen(4);
                break;
            default:
                break;
        }
        return certSKF;
    }


    public String getSn(CryptoDevType type, int containerNum, String certStr) {
        CertSKF certSKF = splitCertStr(type, containerNum, certStr);
        if (certSKF == null) {
            return "";
        }
        CertDetailInfoBean info;
        byte[][] fid = FidUtil.getCertFid(containerNum);
        byte[] signCert;
        byte[] encodeCert;
        if (AirCertSdk.getsInstance().getAirCertConfig().getCertType().equals(CertType.SIGNING)
                || TextUtils.isEmpty(AirCertSdk.getsInstance().getAirCertConfig().getCertType())) {

            signCert = certSKF.getSignCert();
            int signCertLen = signCert != null ? signCert.length : 0;
            int[] signCodeCertLen = new int[]{signCertLen};
            info = readBytesToCert(fid[0], signCert, signCodeCertLen);
        } else {
            encodeCert = certSKF.getEnCodeCert();
            int encodeCertLen = encodeCert != null ? encodeCert.length : 0;
            int[] enCodeCertLen = new int[]{encodeCertLen};
            info = readBytesToCert(fid[1], encodeCert, enCodeCertLen);
        }

        if (info == null) {
            return "";
        }
        return info.getSn();
    }

    /**
     * 将证书byte[]解析为可读的字符串
     * author jff
     *
     * @param fid     设备内证书id
     * @param certBuf 待解析的证书byte[]
     * @return 证书实体
     */
    private static CertDetailInfoBean readBytesToCert(byte[] fid, byte[] certBuf, int[] certLen) {
        if (fid.length == 0 || certBuf.length == 0 || certLen.length == 0) {
            return null;
        }
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd");
            SimpleDateFormat formatterValid = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss");
            ByteArrayInputStream bin = new ByteArrayInputStream(certBuf);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(bin);
            CertDetailInfoBean certDetailInfo = new CertDetailInfoBean();

            certDetailInfo.setIssuerOrg(cert.getIssuerX500Principal().getName());
            certDetailInfo.setIssuerCn(cert.getIssuerDN().getName());
            certDetailInfo.setSubjectOrg(cert.getSubjectX500Principal().getName());
            certDetailInfo.setSubjectCn(cert.getSubjectDN().getName());

            certDetailInfo.setAlg(cert.getSigAlgName());
            certDetailInfo.setVer(cert.getVersion());


            System.arraycopy(fid, 0, certDetailInfo.getCertId(), 0, 2);
            certDetailInfo.setIndate(formatter.format(cert.getNotBefore()) + " - " + formatter.format(cert.getNotAfter()));
            certDetailInfo.setNotBefore(cert.getNotBefore());
            certDetailInfo.setNotAfter(cert.getNotAfter());


            //sn应该是16进制的
            certDetailInfo.setSn(cert.getSerialNumber().toString(16));
            certDetailInfo.setCertLen(certLen);
            bin.close();

            return certDetailInfo;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * 将数据用指定设备内0号容器内私钥签名
     *
     * @param type 指定私钥对应的设备
     * @param str  待签名内容
     * @return 签名后的内容
     */
    public String signData(CryptoDevType type, int containerNum, String str) {

        //        ModuleLog.e("待签名信息 str = " + str);
        if (str.equals("")) {
            return null;
        }

        byte[][] keyPair = FidUtil.getKeyPairFid(type, containerNum);
        byte[] pubFid = keyPair[0];
        byte[] priFid = keyPair[1];

        byte[] signData;
        int[] signDataLen = new int[1];
        byte[] certInfoBytes = str.getBytes();
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            SdkLog.loge("待签名信息 cryptoInstance = null");
            return null;
        }

        int ret = DeviceWrapper.verifyPin(type, containerNum, "");
        if (ret != 0) {
            //            ModuleLog.d(" 验证Pin码result = " + ret + CryptoDevManager.getInstance().getErrorText(ret));
            return "errorCode" + CryptoDevManager.getInstance().getErrorText(ret);
        }

        if (CertHelper.getAlgType().equals(XDJAAlgParams.CA_ALG_RSA)) {
            signData = new byte[256];
            ret = cryptoInstance.RSASign(RSA_ALG_BITS, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);
        } else {
            signData = new byte[64];
            ret = cryptoInstance.SM2Sign(pubFid, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);
        }
        byte[] data = new byte[signDataLen[0]];

        /*
        测试代码导出证书
        SM2PubKey2PubKey(type, containerNum);

        ModuleLog.e("签名前数据str =" + str);
        Log.e("jff", "863 DeviceWrapper signData : " + signData.toString());
        AlgSignUtils.getAlgSignUtils().verifySM2Sign(type, str, signData);
        */
        System.arraycopy(signData, 0, data, 0, data.length);
        //        ModuleLog.e("签名后 数据 = " + new String(Base64.encode(data)));

//        int success = AlgSignUtils.getAlgSignUtils().verifySM2Sign(type, str, signData);
//        int success = AlgSignUtils.getAlgSignUtils().verifyRSASign(type, str, signData);
//        ModuleLog.e("本地签名验签" + success);

        if (ret == 0) {
            return new String(Base64.encode(data));
        }

        return "";
    }

    /**
     * 从容其中读取证书
     *
     * @param type        设备类型
     * @param containerNo 容器ID
     * @param certType    证书类型    0  签名证书       1 交换证书
     * @return
     */
    public static CertDetailInfoBean readCertFromContainer(CryptoDevType type, int containerNo, String certType) {

        byte[] fid = null;
        byte[][] certFid = FidUtil.getCertFid(containerNo);

        if (certType.equals(CertType.SIGNING)) {
            fid = certFid[0];
            return readCertFromFid(type, fid);
        } else if (certType.equals(CertType.ENCRYPT)) {
            fid = certFid[1];
            return readCertFromFid(type, fid);
        } else {
            CertDetailInfoBean info;
            info = readCertFromFid(type, certFid[0]);
            if (info == null) {
                info = readCertFromFid(type, certFid[1]);
            }
            return info;
        }

    }
    private static CertDetailInfoBean readCertFromFid(CryptoDevType type, byte[] fid) {
        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        byte[] bytes = readCertBytes(type, fid, certBuf, certLen);
        if (bytes == null) {
            return null;
        }

        return readBytesToCert(fid, certBuf, certLen);
    }

    /**
     * 读取指定设备0号容器内的签名证书
     *
     * @param type 要读取的设备类型
     * @param fid  读取的位置
     * @return
     */
    public static byte[] readCertBytes(CryptoDevType type, byte[] fid, byte[] certBuf, int[] certLen) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = cryptoInstance.readCert(fid, certBuf, certLen);
        cryptoInstance.close();

        if (result != 0) {
            return null;
        }
        return certBuf;
    }
}
