package com.xdja.cssp.acs.service.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.xdja.cssp.acs.bean.asset.ProductAsset;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import com.xdja.cssp.acs.IAssetService;
import com.xdja.cssp.acs.bean.asset.Asset;
import com.xdja.cssp.acs.bean.asset.SearchBean;
import com.xdja.cssp.acs.bean.asset.Type;
import com.xdja.cssp.acs.service.Constants;
import com.xdja.cssp.acs.service.handler.AssetBeanMapHandler;
import com.xdja.platform.common.lite.kit.StrKit;
import com.xdja.platform.common.lite.kit.prop.Prop;
import com.xdja.platform.common.lite.kit.prop.PropKit;
import com.xdja.platform.microservice.ServiceException;
import com.xdja.platform.microservice.db.Dao;
import com.xdja.platform.microservice.db.SqlHelper;

/**
 * 资产服务实现
 * @date 2015-2-3
 * @author rrx
 * 
 */
@Component
public class AssetServiceImpl implements IAssetService, InitializingBean {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	private final Dao acsDao = Dao.use(Constants.DB_ACS);
	private final Dao amsDao = Dao.use(Constants.DB_AMSCORE);
	
	/**
	 * 检查指定资产是否存在的sql语句
	 */
	private static String ASSET_EXISTS_SQL = null;
	@Override
	public boolean exists(String assetIdentity) {
		if (StrKit.isBlank(assetIdentity)) {
			throw new IllegalArgumentException("参数assetIdentity为空");
		}
		
		try {
			return acsDao.queryForLong(ASSET_EXISTS_SQL, assetIdentity) > 0;
		} catch (SQLException e) {
			throw ServiceException.create(String.format("判断标识为%s的资产是否存在时出现异常", assetIdentity), e);
		}
	}

	/**
	 * 查询资产信息的sql
	 */
	private static String QUERY_ASSET_SQL = null;
	/**
	 * 查询资产私有属性的sql，在配置文件中的键必须以 query.asset. 开头
	 */
	private static List<String> QUERY_ASSET_PROP_SQLS = new ArrayList<String>();
	@Override
	public List<Asset> queryAssets(String... assetIdentity) {
		if (null == assetIdentity || assetIdentity.length <= 0) {
			return Collections.emptyList();
		}
		
		Map<Long, Asset> assetMap = null;
		//查询资产基本信息
		try {
			String sql = SqlHelper.prepare(QUERY_ASSET_SQL, "#identity", assetIdentity.length);
			assetMap = acsDao.queryForMap(sql, new AssetBeanMapHandler(), Arrays.asList(assetIdentity).toArray());
		} catch (SQLException e) {
			throw ServiceException.create(String.format("查询指定资产信息（%s）时出现异常", Arrays.toString(assetIdentity)), e);
		}
		
		//查询资产私有属性
		try {
			if (null != assetMap && !assetMap.isEmpty()) {
				for (String sql : QUERY_ASSET_PROP_SQLS) {
					sql = SqlHelper.prepare(sql, "#assetId", assetMap.size());
					for (Map<String, Object> assetProp : acsDao.queryForList(sql, assetMap.keySet().toArray())) {
						assetMap.get(assetProp.get("assetId")).setInfo(assetProp);
					}
				}
			}
		} catch (SQLException e) {
			e.printStackTrace();
			throw ServiceException.create(String.format("查询指定资产信息（%s）时出现异常", Arrays.toString(assetIdentity)), e);
		}
		
		List<Asset> result = new ArrayList<Asset>();
		if (null != assetMap && !assetMap.isEmpty()) {
			result.addAll(assetMap.values());
		}
		return result;
	}

	/**
	 * 查询关联资产标识的sql
	 */
	private static String QUERY_RELATED_SQL = null;
	@Override
	public String queryRelatedByIdentity(String assetIdentity) {
		if (StrKit.isBlank(assetIdentity)) {
			throw new IllegalArgumentException("参数assetIdentity为空");
		}
		
		String related = null;
		try {
			related = acsDao.queryForStr(QUERY_RELATED_SQL, assetIdentity);
		} catch (SQLException e) {
			throw ServiceException.create(String.format("查询指定资产（%s）关联的资产标识时出现异常", assetIdentity), e);
		}
		
		if (null == related) {
			related = "";
		}
		
		return related;
	}

	@Override
	public String queryUnlockByUId(String uId) {
		if (StrKit.isBlank(uId)) {
			throw new IllegalArgumentException("参数uId为空");
		}
		try {
			return acsDao.queryForStr("SELECT c_unlock_code FROM t_terminal_unlock_code WHERE c_card_uid=?", uId);
		} catch (SQLException e) {
			throw ServiceException.create(String.format("根据安全卡UID=%s查询加密解锁码时出现异常", uId), e);
		}
	}
	
	@Override
	public String queryUnlockByCardNo(String cardNo) {
		if (StrKit.isBlank(cardNo)) {
			throw new IllegalArgumentException("参数cardNo为空");
		}
		try {
			return acsDao.queryForStr("SELECT c_unlock_code FROM t_terminal_unlock_code WHERE c_card_no=?", cardNo);
		} catch (SQLException e) {
			throw ServiceException.create(String.format("根据安全卡cardNo=%s查询加密解锁码时出现异常", cardNo), e);
		}
	}
	
	@Override
	public String queryCardNoByUId(String uId) {
		if (StrKit.isBlank(uId)) {
			throw new IllegalArgumentException("参数uId为空");
		}
		
		try {
			return acsDao.queryForStr("SELECT c_card_no FROM t_terminal_unlock_code WHERE c_card_uid=?", uId);
		} catch (SQLException e) {
			throw ServiceException.create(String.format("根据UID查询加密解锁码时出现异常,uId=%s", uId), e);
		}
	}
	
	@Override
	public List<Asset> queryAssets(SearchBean searchBean) {
		if (null == searchBean
				|| (StringUtils.isBlank(searchBean.getSn())
					&& StringUtils.isBlank(searchBean.getImei())
					&& StringUtils.isBlank(searchBean.getSerialCode()))) {
			return Collections.emptyList();
		}
		
		try {
			List<String> params = new ArrayList<>();
			
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT info.n_id AS id, info.c_asset_identify AS identity, info.n_asset_type AS type, info.c_card_no AS cardNo,");
			sql.append(" info.c_serial_code AS serialCode, info.n_time AS time, info.c_relation_asset_identify AS relatedAsset,");
			sql.append(" info.n_status AS status, mobile.n_asset_id AS assetId, mobile.c_model AS model, mobile.c_chip_code AS chipCode,");
			sql.append(" mobile.c_os_name AS osName, mobile.c_os_version AS osVersion, mobile.c_imei AS imei, mobile.c_sn_no AS snNo");
			sql.append(" FROM t_asset_info info");
			sql.append(" JOIN t_asset_mobiles_info mobile ON info.n_id = mobile.n_asset_id WHERE 1=1");
			
			if (StringUtils.isNotBlank(searchBean.getSn())) {
				sql.append(" AND mobile.c_sn_no LIKE ?");
				params.add("%" + searchBean.getSn() + "%");
			}
			
			if (StringUtils.isNotBlank(searchBean.getImei())) {
				sql.append(" AND mobile.c_imei LIKE ?");
				params.add("%" + searchBean.getImei() + "%");
			}
			
			if (StringUtils.isNotBlank(searchBean.getSerialCode())) {
				sql.append(" OR info.c_serial_code LIKE ?");
				params.add("%" + searchBean.getSerialCode() + "%");
			}
			
			return acsDao.queryForList(sql.toString(), new ResultSetHandler<List<Asset>> () {
				@Override
				public List<Asset> handle(ResultSet rs) throws SQLException {
					List<Asset> result = new ArrayList<>();
					Map<String, Object> info = null;
					Asset asset = null;
					
					while (rs.next()) {
						asset = new Asset();
						asset.setIdentity(rs.getString("identity"));
						asset.setType(Type.valueOf(rs.getInt("type")));
						asset.setCardNo(rs.getString("cardNo"));
						asset.setSerialCode(rs.getString("serialCode"));
						asset.setTime(rs.getLong("time"));
						asset.setRelatedAsset(rs.getString("relatedAsset"));
						asset.setStatus(rs.getInt("status"));
						
						if (StringUtils.isNotBlank(rs.getString("snNo"))) {
							info = new HashMap<String, Object>();
							info.put("model", rs.getString("model"));
							info.put("chipCode", rs.getString("chipCode"));
							info.put("osName", rs.getString("osName"));
							info.put("osVersion", rs.getString("osVersion"));
							info.put("imei", rs.getString("imei"));
							info.put("snNo", rs.getString("snNo"));
							
							asset.setInfo(info);
						}
						
						result.add(asset);
					}
					
					return result;
				}
			}, params.toArray());
		} catch (Exception e) {
			throw ServiceException.create("根据设备扩展属性查询设备信息时出现异常", e);
		}
	}
	
	@Override
	public Map<String, Object> queryUnLockInfo(String cardNo, String uId) {
		if ((StringUtils.isBlank(cardNo) && StringUtils.isBlank(uId))) {
			throw new IllegalArgumentException("参数非法，cardNo,uId至少有一个不能为空");
		}
		
		try {
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT c_card_no cardNo, c_card_uid uId, c_unlock_code unLockCode, c_disk_code diskCode");
			sql.append(" FROM t_terminal_unlock_code WHERE 1=1");
			
			List<String> params = new ArrayList<>();
			
			if (StringUtils.isNotBlank(cardNo)) {
				sql.append(" AND c_card_no = ?");
				params.add(cardNo);
			}
			
			if (StringUtils.isNotBlank(uId)) {
				sql.append(" AND c_card_uid = ?");
				params.add(uId);
			}
		
			return acsDao.queryForMap(sql.toString(), params.toArray());
		} catch (Exception e) {
			throw ServiceException.create(String.format("查询解锁码信息时出现异常，请求参数cardNo=%s , uId=%s", cardNo, uId), e);
		}
	}
	
	@Override
	public void afterPropertiesSet() throws Exception {
		Prop sqlProp = PropKit.use("sql/asset");
		
		ASSET_EXISTS_SQL = sqlProp.get("asset.exists");
		if (StrKit.isBlank(ASSET_EXISTS_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "asset.exists"));
		}
		
		QUERY_ASSET_SQL = sqlProp.get("query.asset");
		if (StrKit.isBlank(QUERY_ASSET_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.asset"));
		}
		
		QUERY_RELATED_SQL = sqlProp.get("query.related");
		if (StrKit.isBlank(QUERY_RELATED_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.related"));
		}
		
		for (Object key : sqlProp.getProperties().keySet()) {
			if (key.toString().startsWith("query.asset.")) {
				QUERY_ASSET_PROP_SQLS.add(sqlProp.get(key.toString()));
			}
		}
		
		QUERY_USBKEY_TYPE  = sqlProp.get("query.usbkey.type");
		if (StrKit.isBlank(QUERY_USBKEY_TYPE)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.usbkey.type"));
		}
	}

	
	/**
	 * 查询资产用途类型 0-用户卡 1-管理卡 2-备份卡
	 */
	private static String QUERY_USBKEY_TYPE = null;
	
	@Override
	public boolean queryUsbkeyIsBC(String sn, Integer caAlg) {
		if (StrKit.isBlank(sn) || null == caAlg) {
			throw new IllegalArgumentException("参数为空");
		}
		
		try {
			return acsDao.queryForLong(QUERY_USBKEY_TYPE, sn, caAlg) > 0;
		} catch (SQLException e) {
			throw ServiceException.create(String.format("判断sn为%s的资产是否具有备份恢复功能时出现异常", sn), e);
		}
	}


	@Override
	public ProductAsset queryAssetByIdentity(String assetIdentify) {
		if (StringUtils.isBlank(assetIdentify)) {
			throw new IllegalArgumentException("参数非法");
		}
		try {
			StringBuffer sqlBuffer = new StringBuffer();
			sqlBuffer.append("SELECT ai.c_asset_identify AS identity, ai.c_asset_type AS type, ai.c_card_no AS cardNo, ai.c_serial_code AS serialCode, ai.n_status AS status, ");
			sqlBuffer.append("tp.c_name AS projectName, tc.c_name AS customerName, tc.c_customer_tax AS customerTax, tor.c_name AS orderName, ");
			sqlBuffer.append("tpi.c_product_category AS productCategory, tpi.c_product_name AS productName, tpi.c_product_model AS productModel, tpi.c_program_version AS programVersion, tpi.c_system_version AS systemVersion ");
			sqlBuffer.append("FROM t_asset_info ai ");
			sqlBuffer.append("LEFT JOIN t_project tp ON ai.n_customer_id = tp.n_customer_id ");
			sqlBuffer.append("LEFT JOIN t_customer tc ON ai.n_customer_id = tc.n_id ");
			sqlBuffer.append("LEFT JOIN t_order tor ON ai.n_customer_id = tor.n_customer_id ");
			sqlBuffer.append("LEFT JOIN t_product_info tpi ON ai.c_asset_identify = tpi.c_asset_identify WHERE 1=1 ");
			sqlBuffer.append("AND (ai.c_asset_identify = ? OR ai.c_serial_code = ?) ");
			List<String> params = new ArrayList<>();
			params.add(assetIdentify);
			params.add(assetIdentify);
			return amsDao.queryForObject(ProductAsset.class, sqlBuffer.toString(), params.toArray());
		}catch (Exception e){
			throw ServiceException.create(MessageFormat.format("查询资产信息出现异常，请求参数assetIdentify = {0}", assetIdentify), e);
		}
	}

	@Override
	public ProductAsset queryAssetByCustomerInfo(String customerName, String customerTax) {
		if (StringUtils.isBlank(customerName) && StringUtils.isBlank(customerTax)) {
			throw new IllegalArgumentException("参数非法, 客户名称和客户税号至少一个不为空");
		}
		try {
			StringBuffer sqlBuffer = new StringBuffer();
			sqlBuffer.append("SELECT ai.c_asset_identify AS identity, ai.c_asset_type AS type, ai.c_card_no AS cardNo, ai.c_serial_code AS serialCode, ai.n_status AS status, ");
			sqlBuffer.append("tp.c_name AS projectName, tc.c_name AS customerName, tc.c_customer_tax AS customerTax, tor.c_name AS orderName, ");
			sqlBuffer.append("tpi.c_product_category AS productCategory, tpi.c_product_name AS productName, tpi.c_product_model AS productModel, tpi.c_program_version AS programVersion, tpi.c_system_version AS systemVersion ");
			sqlBuffer.append("FROM t_customer tc ");
			sqlBuffer.append("LEFT JOIN t_project tp ON tc.n_id = tp.n_customer_id ");
			sqlBuffer.append("LEFT JOIN t_asset_info ai ON ai.n_customer_id = tc.n_id ");
			sqlBuffer.append("LEFT JOIN t_order tor ON tc.n_id = tor.n_customer_id ");
			sqlBuffer.append("LEFT JOIN t_product_info tpi ON tc.n_id = tpi.n_customer_id WHERE 1=1 ");
			List<String> params = new ArrayList<>();

			if (StringUtils.isNotBlank(customerName)) {
				sqlBuffer.append("AND tc.c_name = ? ");
				params.add(customerName);
			}

			if (StringUtils.isNotBlank(customerTax)) {
				sqlBuffer.append("AND tc.c_customer_tax = ? ");
				params.add(customerTax);
			}

			return amsDao.queryForObject(ProductAsset.class, sqlBuffer.toString(), params.toArray());
		}catch (Exception e){
			throw ServiceException.create(MessageFormat.format("查询资产信息出现异常，请求参数customerName = {0}, customerTax = {1}", customerName, customerTax), e);
		}
	}
}
