package com.xdja.cssp.acs.service.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import com.xdja.cssp.acs.ICertService;
import com.xdja.cssp.acs.bean.cert.Cert;
import com.xdja.cssp.acs.bean.cert.Cert.CaAlg;
import com.xdja.cssp.acs.bean.cert.Cert.CertType;
import com.xdja.cssp.acs.bean.cert.CertStatus;
import com.xdja.cssp.acs.bean.cert.PublicKey;
import com.xdja.cssp.acs.service.Constants;
import com.xdja.cssp.acs.service.bean.CheckStateCert;
import com.xdja.cssp.acs.service.handler.CertBeanListHandler;
import com.xdja.cssp.acs.service.handler.CertBeanMapHandler;
import com.xdja.cssp.acs.service.handler.CheckStateCertMapHandler;
import com.xdja.cssp.acs.service.handler.PublicKeyBeanListHandler;
import com.xdja.cssp.acs.service.handler.PublicKeyBeanMapHandler;
import com.xdja.platform.common.lite.kit.StrKit;
import com.xdja.platform.common.lite.kit.prop.Prop;
import com.xdja.platform.common.lite.kit.prop.PropKit;
import com.xdja.platform.microservice.ServiceException;
import com.xdja.platform.microservice.db.Dao;
import com.xdja.platform.microservice.db.SqlHelper;

/**
 * 证书服务实现
 * @date 2015-2-3
 * @author rrx
 * 
 */
@Component
public class CertServiceImpl implements ICertService, InitializingBean {

	private final Dao acsDao = Dao.use(Constants.DB_ACS);
	private final Dao amsDao = Dao.use(Constants.DB_AMSCORE);
	
	/**
	 * 根据卡号获取证书信息的sql语句
	 */
	private static String CHECK_CERT_BY_CARDNO_SQL = null;
	@Override
	public CertStatus checkCertStatus(String cardNo, String sn, int caAlg) {
		if (StrKit.isBlank(cardNo)) {
			throw new IllegalArgumentException("参数cardNo为空");
		}
		if (StrKit.isBlank(sn)) {
			throw new IllegalArgumentException("参数sn为空");
		}
		if (caAlg <= 0 || caAlg >2) {
			throw new IllegalArgumentException("参数caAlg的取值范围为[1-RSA，2-SM2]");
		}
		
		CertStatus certStatus = CertStatus.SUCCESS;
		try {
			Map<String, CheckStateCert> certStateMap = acsDao.queryForMap(CHECK_CERT_BY_CARDNO_SQL, new CheckStateCertMapHandler(), cardNo, caAlg);
			if (null == certStateMap || certStateMap.isEmpty()) {
				certStatus = CertStatus.CARD_NOT_EXISTS;
			} else {
				if (!certStateMap.containsKey(sn)) {
					certStatus = CertStatus.CARD_SN_DISACCORD;
				} else if (CheckStateCert.CERT_STATE_FREEZE == certStateMap.get(sn).getState()) {
					certStatus = CertStatus.CERT_FREEZE;
				}
			}
		} catch (SQLException e) {
			throw ServiceException.create(String.format("根据卡号%s查询对应的证书信息出错", cardNo), e);
		}
		
		return certStatus;
	}

	/**
	 * 根据sn列表获取证书公钥信息的sql语句
	 */
	private static String QUERY_PUBLICKEY_BY_SNLIST_SQL = null;
	@Override
	public List<PublicKey> queryPublicKeyBySnList(List<String> snList, int caAlg) {
		if (null == snList || snList.isEmpty()) {
			throw new IllegalArgumentException("参数snList为空");
		}
		
		Object[] params = new Object[0];
		StringBuffer sql = new StringBuffer(QUERY_PUBLICKEY_BY_SNLIST_SQL).append(" WHERE ");
		if (caAlg != 0) {
			params = ArrayUtils.add(params, caAlg);
			sql.append("n_ca_alg=? AND ");
		}
		params = ArrayUtils.addAll(params, snList.toArray());
		sql.append(SqlHelper.prepare("c_cert_sn IN (#snList)", "#snList", snList.size()));
		
		try {
			return acsDao.queryForList(sql.toString(), new PublicKeyBeanListHandler(), params);
		} catch (SQLException e) {
			throw ServiceException.create("根据证书列表查询指定算法和类型的公钥信息时出错", e);
		}
	}
	
	/**
	 * 根据卡号列表获取证书公钥信息的sql语句
	 */
	private static String QUERY_PUBLICKEY_BY_CARDNOLIST_SQL = null;
	@Override
	public Map<String, List<PublicKey>> queryPublicKeyByCardNos(List<String> cardNoList, int caAlg, int certType) {
		if (null == cardNoList || cardNoList.isEmpty()) {
			throw new IllegalArgumentException("参数cardNoList为空");
		}
		
		Object[] params = new Object[0];
		StringBuffer sql = new StringBuffer(QUERY_PUBLICKEY_BY_CARDNOLIST_SQL).append(" WHERE ");
		if (caAlg != 0) {
			params = ArrayUtils.add(params, caAlg);
			sql.append("n_ca_alg=? AND ");
		}
		if (certType != 0) {
			params = ArrayUtils.add(params, certType);
			params = ArrayUtils.add(params, PublicKey.CertType.SINGLE.value);
			sql.append("(n_cert_type=? OR n_cert_type=?) AND ");
		}
		params = ArrayUtils.addAll(params, cardNoList.toArray());
		sql.append(SqlHelper.prepare("c_card_no IN (#cardNoList)", "#cardNoList", cardNoList.size()));
		
		try {
			return acsDao.queryForMap(sql.toString(), new PublicKeyBeanMapHandler(), params);
		} catch (SQLException e) {
			throw ServiceException.create("根据证书列表查询指定算法和类型的公钥信息时出错", e);
		}
	}

	/**
	 * 根据sn列表获取证书信息的sql语句
	 */
	private static String QUERY_CERT_BY_SNLIST_SQL = null;
	@Override
	public List<Cert> queryCertBySnList(List<String> snList, int caAlg) {
		if (null == snList || snList.isEmpty()) {
			throw new IllegalArgumentException("参数snList为空");
		}
		
		Object[] params = new Object[0];
		StringBuffer sql = new StringBuffer(QUERY_CERT_BY_SNLIST_SQL).append(" WHERE ");
		if (caAlg != 0) {
			params = ArrayUtils.add(params, caAlg);
			sql.append("n_ca_alg=? AND ");
		}
		params = ArrayUtils.addAll(params, snList.toArray());
		sql.append(SqlHelper.prepare("c_cert_sn IN (#snList)", "#snList", snList.size()));
		
		try {
			return acsDao.queryForList(sql.toString(), new CertBeanListHandler(), params);
		} catch (SQLException e) {
			throw ServiceException.create("根据证书列表查询指定算法和类型的公钥信息时出错", e);
		}
	}

	/**
	 * 根据卡号列表获取证书信息的sql语句
	 */
	private static String QUERY_CERT_BY_CARDNOLIST_SQL = null;
	@Override
	public Map<String, List<Cert>> queryCertByCardNos(List<String> cardNoList, int caAlg, int certType) {
		if (null == cardNoList || cardNoList.isEmpty()) {
			throw new IllegalArgumentException("参数cardNoList为空");
		}
		
		Object[] params = new Object[0];
		StringBuffer sql = new StringBuffer(QUERY_CERT_BY_CARDNOLIST_SQL).append(" WHERE ");
		if (caAlg != 0) {
			params = ArrayUtils.add(params, caAlg);
			sql.append("n_ca_alg=? AND ");
		}
		if (certType != 0) {
			params = ArrayUtils.add(params, certType);
			params = ArrayUtils.add(params, Cert.CertType.SINGLE.value);
			//sql.append("(n_cert_type=? OR n_cert_type=?) AND ");
			sql.append("n_cert_type IN (?,?) AND ");
		}
		params = ArrayUtils.addAll(params, cardNoList.toArray());
		sql.append(SqlHelper.prepare("c_card_no IN (#cardNoList)", "#cardNoList", cardNoList.size()));
		
		try {
			//System.out.println("从证书表中读取证书的sql:" + sql.toString());
			return acsDao.queryForMap(sql.toString(), new CertBeanMapHandler(), params);
		} catch (SQLException e) {
			throw ServiceException.create("根据证书列表查询指定算法和类型的公钥信息时出错", e);
		}
	}
	
	private static String QUERY_HIS_CERT_BY_SNLIST_SQL = null;
	@Override
	public List<Cert> queryHisCertBySnList(final List<String> snList, final int caAlg) {
		if (null == snList || snList.isEmpty()) {
			throw new IllegalArgumentException("参数snList为空");
		}
		
		Object[] params = new Object[0];
		StringBuffer sql = new StringBuffer(QUERY_HIS_CERT_BY_SNLIST_SQL).append(" WHERE ");
		
		switch (caAlg) {
			
			case Constants.CA_ALG_NORMAL:
				sql.append(SqlHelper.prepare("c_sm2_sign_sn IN (#snList) OR c_sm2_enc_sn IN (#snList) OR c_rsa_sign_sn IN (#snList) OR c_rsa_enc_sn IN (#snList)", "#snList", snList.size()));
				params = ArrayUtils.addAll(params, snList.toArray());
				params = ArrayUtils.addAll(params, snList.toArray());
				params = ArrayUtils.addAll(params, snList.toArray());
				params = ArrayUtils.addAll(params, snList.toArray());
				break;
				
			case Constants.CA_ALG_SM2:
				sql.append(SqlHelper.prepare("c_sm2_sign_sn IN (#snList) OR c_sm2_enc_sn IN (#snList)", "#snList", snList.size()));
				params = ArrayUtils.addAll(params, snList.toArray());
				params = ArrayUtils.addAll(params, snList.toArray());
				break;
				
			case Constants.CA_ALG_RAS:
				sql.append(SqlHelper.prepare("c_rsa_sign_sn IN (#snList) OR c_rsa_enc_sn IN (#snList)", "#snList", snList.size()));
				params = ArrayUtils.addAll(params, snList.toArray());
				params = ArrayUtils.addAll(params, snList.toArray());
				break;
		}
		
		try {
			return amsDao.queryForList(sql.toString(), new ResultSetHandler<List<Cert>> () {

				@Override
				public List<Cert> handle(ResultSet rs) throws SQLException {
					List<Cert> results = new ArrayList<Cert>();

			        if (!rs.next()) {
			            return results;
			        }
			        
			        Cert cert = null;
			        do {
			        	switch (caAlg) {
						
							case Constants.CA_ALG_NORMAL:

								if (snList.contains(rs.getString("c_sm2_sign_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_SM2), CertType.valueOf(Constants.CA_CERTTYPE_SIGN));
									cert.setSn(rs.getString("c_sm2_sign_sn"));
									cert.setCert(rs.getString("c_sm2_sign"));
						        	results.add(cert);
								}
					        	
								if (snList.contains(rs.getString("c_sm2_enc_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_SM2), CertType.valueOf(Constants.CA_CERTTYPE_ENCRYPT));
									cert.setSn(rs.getString("c_sm2_enc_sn"));
									cert.setCert(rs.getString("c_sm2_enc"));
						        	results.add(cert);
								}
					        	
					        	if (snList.contains(rs.getString("c_rsa_sign_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_RAS), CertType.valueOf(Constants.CA_CERTTYPE_SIGN));
									cert.setSn(rs.getString("c_rsa_sign_sn"));
									cert.setCert(rs.getString("c_rsa_sign"));
									results.add(cert);
					        	}
					        	
					        	if (snList.contains(rs.getString("c_rsa_enc_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_RAS), CertType.valueOf(Constants.CA_CERTTYPE_ENCRYPT));
									cert.setSn(rs.getString("c_rsa_enc_sn"));
									cert.setCert(rs.getString("c_rsa_enc"));
									results.add(cert);
					        	}
					        	break;
								
							case Constants.CA_ALG_SM2:
								
								if (snList.contains(rs.getString("c_sm2_sign_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_SM2), CertType.valueOf(Constants.CA_CERTTYPE_SIGN));
									cert.setSn(rs.getString("c_sm2_sign_sn"));
									cert.setCert(rs.getString("c_sm2_sign"));
						        	results.add(cert);
								}
					        	
								if (snList.contains(rs.getString("c_sm2_enc_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_SM2), CertType.valueOf(Constants.CA_CERTTYPE_ENCRYPT));
									cert.setSn(rs.getString("c_sm2_enc_sn"));
									cert.setCert(rs.getString("c_sm2_enc"));
						        	results.add(cert);
								}
					        	break;
								
							case Constants.CA_ALG_RAS:

								if (snList.contains(rs.getString("c_rsa_sign_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_RAS), CertType.valueOf(Constants.CA_CERTTYPE_SIGN));
									cert.setSn(rs.getString("c_rsa_sign_sn"));
									cert.setCert(rs.getString("c_rsa_sign"));
						        	results.add(cert);
								}
					        	
								if (snList.contains(rs.getString("c_rsa_enc_sn"))) {
									cert = new Cert(rs.getString("c_card_no"), CaAlg.valueOf(Constants.CA_ALG_RAS), CertType.valueOf(Constants.CA_CERTTYPE_ENCRYPT));
									cert.setSn(rs.getString("c_rsa_enc_sn"));
									cert.setCert(rs.getString("c_rsa_enc"));
						        	results.add(cert);
								}
					        	break;
						}
			        	
			        } while (rs.next());

			        return results;
				}
				
			}, params);
			
		} catch (Exception e) {
			throw ServiceException.create("根据证书列表查询指定算法和类型的公钥信息时出错", e);
		}
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		Prop sqlProp = PropKit.use("sql/cert");
		
		CHECK_CERT_BY_CARDNO_SQL = sqlProp.get("check.cert_byCardNo");
		if (StrKit.isBlank(CHECK_CERT_BY_CARDNO_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "check.cert_byCardNo"));
		}
		
		QUERY_PUBLICKEY_BY_CARDNOLIST_SQL = sqlProp.get("query.cert.publickey");
		if (StrKit.isBlank(QUERY_PUBLICKEY_BY_CARDNOLIST_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.cert.publickey"));
		}
		
		QUERY_PUBLICKEY_BY_SNLIST_SQL = sqlProp.get("query.cert.publickey");
		if (StrKit.isBlank(QUERY_PUBLICKEY_BY_SNLIST_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.cert.publickey"));
		}
		
		QUERY_CERT_BY_CARDNOLIST_SQL = sqlProp.get("query.cert.cert");
		if (StrKit.isBlank(QUERY_CERT_BY_CARDNOLIST_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.cert.cert"));
		}
		
		QUERY_CERT_BY_SNLIST_SQL = sqlProp.get("query.cert.cert");
		if (StrKit.isBlank(QUERY_CERT_BY_SNLIST_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.cert.cert"));
		}
		
		QUERY_HIS_CERT_BY_SNLIST_SQL = sqlProp.get("query.cert.hisCert");
		if (StrKit.isBlank(QUERY_HIS_CERT_BY_SNLIST_SQL)) {
			throw ServiceException.create(String.format("sql语句%s不存在", "query.cert.cert"));
		}
	}
}
