package com.xdja.cssp.as.ticket;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.UUID;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.ClassUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.xdja.cssp.as.service.AS;
import com.xdja.cssp.as.service.ITicketService;
import com.xdja.cssp.as.service.model.ResultBean;
import com.xdja.cssp.as.ticket.annotation.CheckTicket;
import com.xdja.cssp.restful.exception.ApiException;
import com.xdja.cssp.restful.exception.TicketNotAuthException;
import com.xdja.platform.rpc.consumer.refer.DefaultServiceRefer;

/**
 * 校验Ticket切面
 * @author wyf
 *
 */
@Aspect
@Component
public class VerifyTicketAspect {
	
	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	private ITicketService service = DefaultServiceRefer.getServiceRefer(ITicketService.class);
	
	/**
	 * HTTP消息头中ticket的Key
	 */
	public static final String TICKET_KEY = "ticket";
	
	@SuppressWarnings("unchecked")
	@Around("@annotation(checkTicket)")
	public Object verifyTicket(ProceedingJoinPoint jp, CheckTicket checkTicket) throws Throwable {
		logger.debug("开始ticket有效性验证");
		HttpServletRequest httpRequest = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
		
		AsBean asBean = new AsBean();
		
		//服务标识
		String hostId = "未设置服务标识";
		Method method = ClassUtils.getMethodIfAvailable(jp.getTarget().getClass(), "getHostId");
		if (null != method) {
			try {
				Object result = method.invoke(jp.getTarget());
				if (null != result) {
					hostId = result.toString();
					logger.debug("服务器标识：{}", hostId);
				} else {
					logger.warn("{}中的方法：public String getHostId(){}未返回服务标识", jp.getTarget().getClass());
				}
			} catch (IllegalAccessException | IllegalArgumentException
					| InvocationTargetException e) {
				logger.error("从{}中获取服务标识失败", jp.getTarget().getClass(), e);
			}
		} else {
			logger.warn("请在{}中添加方法：public String getHostId(){}返回服务标识", jp.getTarget().getClass());
		}
		asBean.setHostId(hostId);
		
		//生成请求Id
		String requestId = this.genRequestId();
		asBean.setRequestId(requestId);
		
		//验证ticket
		String ticket = httpRequest.getHeader(TICKET_KEY);
		if (StringUtils.isBlank(ticket)) {
			throw new TicketNotAuthException(hostId, requestId,
					ApiException.REQUEST_PARAMS_NOT_VALID, "ticket为空");
		}
		try {
			ResultBean resultBean = service.verifyTicket(ticket);
			if (resultBean.getResultStatus() != AS.RESULT_SUCCESS) {
				throw new TicketNotAuthException(hostId, requestId,
						ApiException.NOT_AUTHORIZED, "ticket验证未通过");
			}
			
			Map<String, Object> accountInfo = (Map<String, Object>)resultBean.getInfo();
			asBean.setAccount((String)accountInfo.get("account"));
			asBean.setCardNo((String)accountInfo.get("cardNo"));
			asBean.setTicketPeriod((Long)accountInfo.get("ticketPeriod"));
			
			logger.debug("ticket有效性验证完成");
		} catch (IllegalArgumentException e) {
			logger.error("验证ticket时参数非法", e);
			throw new TicketNotAuthException(hostId, requestId,
					ApiException.REQUEST_PARAMS_NOT_VALID, "ticket为空");
		} catch (Exception e) {
			logger.error("验证ticket失败", e);
			throw new TicketNotAuthException(hostId, requestId,
					ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		
		Object[] args = jp.getArgs();
		Object arg = null;
		for (int i = 0; i < args.length; i++) {
			arg = args[i];
			if (null != arg && arg.getClass() == AsBean.class) {
				args[i] = asBean;
				break;
			}
		}
		return jp.proceed(args);
	}
	
	/**
	 * 生成请求id
	 * @return
	 */
	private String genRequestId() {
		return UUID.randomUUID().toString();
	}

}
