package com.xdja.cssp.restful.exception.handler;

import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.xdja.cssp.restful.auth.exception.AuthException;
import com.xdja.cssp.restful.auth.exception.DecryptException;
import com.xdja.cssp.restful.auth.exception.EncryptException;
import com.xdja.cssp.restful.auth.exception.NotMatchSignatureException;
import com.xdja.cssp.restful.auth.exception.SignException;
import com.xdja.cssp.restful.auth.exception.VerifySignatureException;
import com.xdja.cssp.restful.bean.ResponseError;
import com.xdja.cssp.restful.exception.ApiException;
import com.xdja.cssp.restful.exception.ApiExceptionResponse;
import com.xdja.cssp.restful.exception.BadRequestException;
import com.xdja.cssp.restful.exception.ForbiddenException;
import com.xdja.cssp.restful.exception.InternalServerException;
import com.xdja.cssp.restful.exception.ResourceConflictException;
import com.xdja.cssp.restful.exception.ResourceNotFoundException;
import com.xdja.cssp.restful.exception.UnauthorizedException;


/**
 * 异常处理类，将实际的异常转换为异常json字符串返回给客户端，
 * 并且根据异常类型返回对应的状态码
 * @author zhouyj
 *
 */
@ControllerAdvice
public class ApiExceptionHandler {
	
	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
//	@Autowired
//	private ILogBusiness logBusiness;
	
	
	/**
	 * 处理非法请求
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(value = BadRequestException.class)
	@ResponseBody
	public Object handleBadRequest(BadRequestException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.BAD_REQUEST.value());
		return convert(exception);
	}
	
	
	/**
	 * 处理资源不存在
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(value = ResourceNotFoundException.class)
	@ResponseBody
	public Object handleNotFound(ResourceNotFoundException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.NOT_FOUND.value());
		return convert(exception);
	}
	
	
	/**
	 * 处理资源冲突
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(value = ResourceConflictException.class)
	@ResponseBody
	public Object handleConflict(ResourceConflictException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.CONFLICT.value());
		return convert(exception);
	}
	
	
	/**
	 * 处理未认证
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(value = UnauthorizedException.class)
	@ResponseBody
	public Object handleUnauthorized(UnauthorizedException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.UNAUTHORIZED.value());
		return convert(exception);
	}
	
	/**
	 * 禁止访问
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(ForbiddenException.class)
	@ResponseBody
	public Object handleForbidden(ForbiddenException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.FORBIDDEN.value());
		return convert(exception);
	}
	
	/**
	 * 服务器内部异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(InternalServerException.class)
	@ResponseBody
	public Object handleForbidden(InternalServerException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
		return convert(exception);
	}
	
	/**
	 * 转换异常为对应的异常返回对象
	 * @param exception
	 * @return
	 */
	public ApiExceptionResponse convert(ApiException exception) {
		ApiExceptionResponse response = new ApiExceptionResponse();
		response.setRequestId(exception.getRequestId());
		response.setHostId(exception.getHostId());
		response.setErrCode(exception.getErrCode());
		response.setMessage(exception.getMessage());
		return response;
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public String handleUnexpectedServerError(RuntimeException ex) {
	    return ex.getMessage();
	}
	
	/**
	 * 解密异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(DecryptException.class)
	@ResponseBody
	public Object handleDecrypt(DecryptException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.BAD_REQUEST.value());
		return convertError(exception);
	}
	
	/**
	 * 加密异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(EncryptException.class)
	@ResponseBody
	public Object handleEncrypt(EncryptException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
		return convertError(exception);
	}
	
	/**
	 * 签名异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(SignException.class)
	@ResponseBody
	public Object handleSign(SignException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
		return convertError(exception);
	}
	
	/**
	 * 验证签名异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(VerifySignatureException.class)
	@ResponseBody
	public Object handleVerifySignature(VerifySignatureException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.BAD_REQUEST.value());
		return convertError(exception);
	}

	/**
	 * 签名不匹配异常
	 * @param exception
	 * @return
	 */
	@ExceptionHandler(NotMatchSignatureException.class)
	@ResponseBody
	public Object handleNotMatchSignature(NotMatchSignatureException exception, HttpServletResponse httpservletResponse) {
		saveLog(exception);
		httpservletResponse.setStatus(HttpStatus.UNAUTHORIZED.value());
		return convertError(exception);
	}

	/**
	 * 转换异常为对应的异常返回对象
	 * @param exception
	 * @return
	 */
	public ResponseError convertError(AuthException exception) {
		ResponseError response = new ResponseError();
		response.setRequestId(exception.getRequestId());
		response.setHostId(exception.getHostId());
		response.setErrCode(exception.getErrCode());
		response.setMessage(exception.getMessage());
		return response;
	}
	
	public void saveLog(ApiException exception) {
//		Log log;
//		if(exception.getCause() == null) {
//			log = new Log(exception.getHostId(), exception.getRequestId(), null, exception.getStatus(), 
//					exception.getErrCode(), exception.getMessage(), null, System.currentTimeMillis());
//		}else {
//			log = new Log(exception.getHostId(), exception.getRequestId(), null, exception.getStatus(), 
//					exception.getErrCode(), exception.getMessage(), exception.getCause().getMessage(), System.currentTimeMillis());
//		}
//		
//		try {
//			logBusiness.saveLog(log);
//		} catch (Exception e) {
//			logger.error("记录错误日志异常", e);
//		}
	}
	
	public void saveLog(AuthException exception) {
//		Log log;
//		if(exception.getCause() == null) {
//			log = new Log(exception.getHostId(), exception.getRequestId(), null, exception.getStatus(), 
//					exception.getErrCode(), exception.getMessage(), null, System.currentTimeMillis());
//		}else {
//			log = new Log(exception.getHostId(), exception.getRequestId(), null, exception.getStatus(), 
//					exception.getErrCode(), exception.getMessage(), exception.getCause().getMessage(), System.currentTimeMillis());
//		}
//		
//		try {
//			logBusiness.saveLog(log);
//		} catch (Exception e) {
//			logger.error("记录错误日志异常", e);
//		}
	}
}
