package com.xdja.location;

import android.location.Location;
import android.os.Bundle;

import androidx.annotation.ColorRes;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.content.ContextCompat;

import com.amap.api.maps2d.AMap;
import com.amap.api.maps2d.AMapUtils;
import com.amap.api.maps2d.CameraUpdateFactory;
import com.amap.api.maps2d.UiSettings;
import com.amap.api.maps2d.model.BitmapDescriptor;
import com.amap.api.maps2d.model.BitmapDescriptorFactory;
import com.amap.api.maps2d.model.CircleOptions;
import com.amap.api.maps2d.model.LatLng;
import com.amap.api.maps2d.model.MyLocationStyle;
import com.blankj.utilcode.utils.ConvertUtils;
import com.xdja.eoa.commonutil.ArraysUtils;
import com.xdja.eoa.commonwidget.widget.CommonToolBar;
import com.xdja.location.params.RuleAddress;

import java.util.List;

/**
 * @author mob-wan
 * @date 2019/10/21
 * <p>
 * 定位详情地址展示
 * 1、可以选择是单次定位还是连续定位，默认连续定位，连续定位默认时间间隔2s
 * 2、可以选择是否展示打卡范围，默认不展示
 */
public class LocationDetailActivityV2 extends AppCompatActivity implements AMap.OnMyLocationChangeListener {

    /**
     * 标题内容
     */
    public static final String EXTRA_TITLE = "extra_title";
    /**
     * 单次定位
     */
    public static final String EXTRA_ONCE_LOCATION = "extra_once_location";
    /**
     * 连续定位时间间隔，单位：毫秒，默认值：2000毫秒
     */
    public static final String EXTRA_INTERVAL = "extra_interval";
    /**
     * 打卡范围列表
     */
    public static final String EXTRA_LIST_RULE_ADDRESS = "extra_list_rule_address";
    /**
     * 当前定位地址
     */
    public static final String EXTRA_CURRENT_LATLNG = "extra_current_location";

    /**
     * 地图控件引用
     */
    private EoaMapView mMapView;
    /**
     * 地图的控制器类，用来操作地图
     */
    private AMap aMap;

    private String mTitle;
    private boolean mOnceLocation;
    private long mInterval;
    private List<RuleAddress> mRuleAddressList;
    private LatLng mCurrentLat;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.location_activity_location_detail_v2);

        getData();
        initView();
        initMap(savedInstanceState);
        initRuleAddress();
    }

    /**
     * 打卡范围
     */
    private void initRuleAddress() {
        if (ArraysUtils.isListEmpty(mRuleAddressList)) {
            return;
        }
        for (RuleAddress item : mRuleAddressList) {
            //构造一个圆形，在地图上标注打卡范围
            LatLng ruleLatLng = new LatLng(item.getLatitude(), item.getLongitude());
            long distanceRange = item.getDistanceRange();
            @ColorRes int fillColor;
            @ColorRes int strokeColor;
            if (mCurrentLat != null && AMapUtils.calculateLineDistance(mCurrentLat, ruleLatLng) < distanceRange) {
                fillColor = R.color.location_circle_fill_color;
                strokeColor = R.color.location_circle_stroke_color;
            } else {
                fillColor = R.color.location_circle_fill_color_red;
                strokeColor = R.color.location_circle_stroke_color_red;
            }
            aMap.addCircle(new CircleOptions()
                    .center(ruleLatLng)
                    .radius(distanceRange)
                    .fillColor(ContextCompat.getColor(this, fillColor))
                    .strokeColor(ContextCompat.getColor(this, strokeColor))
                    .strokeWidth(ConvertUtils.dp2px(1)));
        }
    }

    private void initMap(Bundle savedInstanceState) {
        //在activity执行onCreate时执行mMapView.onCreate(savedInstanceState). 创建地图
        mMapView.onCreate(savedInstanceState);
        if (aMap == null) {
            aMap = mMapView.getMap();
        }
        UiSettings uiSettings = aMap.getUiSettings();
        //隐藏缩放按钮
        uiSettings.setZoomControlsEnabled(false);

        MyLocationStyle myLocationStyle = new MyLocationStyle();
        if (mOnceLocation) {
            //定位一次，且将视角移动到地图中心点
            myLocationStyle.myLocationType(MyLocationStyle.LOCATION_TYPE_LOCATE);
        } else {
            //连续定位、且将视角移动到地图中心点，定位蓝点跟随设备移动
            myLocationStyle.myLocationType(MyLocationStyle.LOCATION_TYPE_FOLLOW);
            //设置定位频次方法，单位：毫秒，默认值：2000毫秒
            myLocationStyle.interval(mInterval);
        }
        BitmapDescriptor locationIcon = BitmapDescriptorFactory.fromResource(R.drawable.location_ic_detail_marker);
        myLocationStyle.myLocationIcon(locationIcon);
        myLocationStyle.radiusFillColor(ContextCompat.getColor(this, android.R.color.transparent));
        myLocationStyle.strokeColor(ContextCompat.getColor(this, android.R.color.transparent));
        //设置定位蓝点的style
        aMap.setMyLocationStyle(myLocationStyle);
        //设置为true表示启动显示定位蓝点
        aMap.setMyLocationEnabled(true);
        //设置地图缩放级别为17f，缩放范围一般是3f-19f
        float zoomLevel = getZoomLevel(mRuleAddressList, mCurrentLat);
        aMap.animateCamera(CameraUpdateFactory.zoomTo(zoomLevel));
        aMap.setOnMyLocationChangeListener(this);
    }

    private void initView() {
        CommonToolBar toolBar = findViewById(R.id.toolbar_location_detail);
        toolBar.setTitle(mTitle);
        mMapView = findViewById(R.id.view_map);
    }

    private void getData() {
        Bundle bundle = getIntent().getExtras();
        if (bundle != null) {
            mTitle = bundle.getString(EXTRA_TITLE, getString(R.string.location_title));
            mOnceLocation = bundle.getBoolean(EXTRA_ONCE_LOCATION);
            mInterval = bundle.getLong(EXTRA_INTERVAL, 2000);
            mRuleAddressList = bundle.getParcelableArrayList(EXTRA_LIST_RULE_ADDRESS);
            mCurrentLat = bundle.getParcelable(EXTRA_CURRENT_LATLNG);
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        //在activity执行onDestroy时执行mMapView.onDestroy(). 销毁地图
        mMapView.onDestroy();
    }

    @Override
    protected void onResume() {
        super.onResume();
        //在activity执行onResume时执行mMapView.onResume(). 重新绘制加载地图
        mMapView.onResume();
    }

    @Override
    protected void onPause() {
        super.onPause();
        //在activity执行onPause时执行mMapView.onPause(). 暂停地图的绘制
        mMapView.onPause();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        //在activity执行onSaveInstanceState时执行mMapView.onSaveInstanceState(outState). 保存地图当前的状态
        mMapView.onSaveInstanceState(outState);
    }

    @Override
    public void onMyLocationChange(Location location) {
    }

    /**
     * 获取地图缩放级别（动态调整缩放比例，保证定位地址和距离最近的打卡范围在可见视图内）
     *
     * @param ruleAddressList 规则地址
     * @param currentLat      当前定位经纬度
     * @return 地图缩放级别
     */
    private float getZoomLevel(List<RuleAddress> ruleAddressList, LatLng currentLat) {
        if (ArraysUtils.isListEmpty(ruleAddressList) || mCurrentLat == null) {
            return 17f;
        }
        //遍历获取定位地址和距离最近的打卡范围 最小距离
        float minDistance = 0;
        for (RuleAddress address : ruleAddressList) {
            LatLng latLng = new LatLng(address.getLatitude(), address.getLongitude());
            float distance = AMapUtils.calculateLineDistance(latLng, currentLat);
            if (minDistance > distance || minDistance == 0) {
                minDistance = distance;
            }
        }
        //计算地图缩放级别，缩放级别19f 对应比例尺大小10
        //计算公式：比例尺大小1/比例尺大小2 == 2 ^ (缩放比例2 - 缩放比例1)
        double result = 19f - Math.log(minDistance / 10) / Math.log(2);
        float minZoomLevel = 3f;
        float maxZoomLevel = 19f;
        if (result < minZoomLevel) {
            return minZoomLevel;
        } else if (result > maxZoomLevel) {
            return maxZoomLevel;
        }
        return (float) result;
    }
}
