package com.xdja.location;

import android.app.Activity;
import android.content.Intent;
import android.text.TextUtils;
import android.view.View;

import com.amap.api.location.AMapLocation;
import com.amap.api.location.AMapLocationClient;
import com.amap.api.location.AMapLocationClientOption;
import com.amap.api.location.AMapLocationListener;
import com.amap.api.maps2d.model.LatLng;
import com.amap.api.services.core.LatLonPoint;
import com.amap.api.services.geocoder.GeocodeResult;
import com.amap.api.services.geocoder.GeocodeSearch;
import com.amap.api.services.geocoder.RegeocodeQuery;
import com.amap.api.services.geocoder.RegeocodeResult;
import com.blankj.utilcode.utils.IntentUtils;
import com.blankj.utilcode.utils.LocationUtils;
import com.blankj.utilcode.utils.LogUtils;
import com.blankj.utilcode.utils.NetworkUtils;
import com.xdja.eoa.commonutil.permission.PermissionsUtils;
import com.xdja.eoa.commonwidget.widget.CustomDialog;
import com.xm.permissions.OnPermissionResult;
import com.xm.permissions.PermissionsManager;

import java.util.Arrays;
import java.util.List;

/**
 * @author mob-wan
 * @date 2019/11/19
 * <p>
 * 定位帮助类
 */
public class LocationHelper implements AMapLocationListener, OnPermissionResult {

    /**
     * 定位权限未开启
     */
    public static final int ERROR_CODE_NO_PERMISSION = 1;
    /**
     * GPS未开启
     */
    public static final int ERROR_CODE_NO_GPS = 2;
    /**
     * 网络未连接、网络信号差、链路不通
     */
    public static final int ERROR_CODE_NO_NETWORK = 3;
    /**
     * GPS信号弱
     */
    public static final int ERROR_CODE_WEAK_GPS = 4;

    public static final int ERROR_CODE_OTHER = 5;

    private Activity mActivity;

    /**
     * 声明mLocationClient对象
     */
    private AMapLocationClient mLocationClient;

    private OnLocationCallBack mOnLocationCallBack;

    private boolean goSettings = false;

    public boolean isGoSettings() {
        return goSettings;
    }

    public void setGoSettings(boolean goSettings) {
        this.goSettings = goSettings;
    }

    public LocationHelper(Activity activity, OnLocationCallBack onLocationCallBack) {
        this.mActivity = activity;
        this.mOnLocationCallBack = onLocationCallBack;
    }

    /**
     * 获取定位权限
     */
    public void requestLocationPermission(OnPermissionResult onPermissionResult) {
        List<String> permissionList = Arrays.asList(PermissionsUtils.LOCATION_PERMISSION);
        String[] permissions = new String[permissionList.size()];
        permissionList.toArray(permissions);
        PermissionsManager permissionsManager = new PermissionsManager(mActivity);
        permissionsManager.setPermissions(permissions)
                .setOnPermissionResult(onPermissionResult)
                .request();
    }

    /**
     * 拒绝开启定位权限、且设置不再询问时，弹框提示
     */
    private void showLocPermissionDeniedDialog() {
        final CustomDialog dialog = new CustomDialog(mActivity);
        dialog.setCloseSkin(true);
        dialog.setMessage("您已拒绝开启定位权限，请打开设置手动开启", true);
        dialog.setCancelButton(R.string.common_widget_cancel, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });
        dialog.setOkButton("去设置", new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
                String packageName = mActivity.getPackageName();
                Intent appDetailsSettingsIntent = IntentUtils.getAppDetailsSettingsIntent(packageName);
                mActivity.startActivity(appDetailsSettingsIntent);
                goSettings = true;
            }
        });
        dialog.show();
    }

    /**
     * 判断是否有定位权限
     */
    public boolean hasLocPermission() {
        return PermissionsUtils.havePermission(mActivity, PermissionsUtils.LOCATION_PERMISSION);
    }

    /**
     * 获取当前位置之前，判断是否符合定位条件
     */
    private boolean beforeGetAddress() {
        if (mOnLocationCallBack != null) {
            mOnLocationCallBack.onLocating();
        }

        if (!hasLocPermission()) {
            //缺少定位权限
            if (mOnLocationCallBack != null) {
                mOnLocationCallBack.onLocationError(ERROR_CODE_NO_PERMISSION,
                        "定位权限未开启，当前无法获取定位", "去开启", new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                requestLocationPermission(LocationHelper.this);
                            }
                        });
            }
            return false;
        }
        if (!LocationUtils.isGpsEnabled()) {
            //GPS未开启
            if (mOnLocationCallBack != null) {
                mOnLocationCallBack.onLocationError(ERROR_CODE_NO_GPS,
                        "GPS未开启，无法获取精确定位", "去开启", new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                LocationUtils.openGpsSettings();
                            }
                        });
            }
            return false;
        }
        return true;
    }

    public void startLocation() {
        if (!beforeGetAddress()) {
            return;
        }
        mLocationClient = new AMapLocationClient(mActivity);
        //初始化定位参数
        AMapLocationClientOption mLocationOption = new AMapLocationClientOption();
        //设置定位监听
        mLocationClient.setLocationListener(this);
        //设置定位模式为高精度模式，Battery_Saving为低功耗模式，Device_Sensors是仅设备模式
        mLocationOption.setLocationMode(AMapLocationClientOption.AMapLocationMode.Hight_Accuracy);
        //设置定位间隔,单位毫秒,默认为2000ms
        mLocationOption.setInterval(2 * 1000);
        //需要返回地址信息
        mLocationOption.setNeedAddress(true);
        //设置定位参数
        mLocationClient.setLocationOption(mLocationOption);
        //此方法为每隔固定时间会发起一次定位请求，为了减少电量消耗或网络流量消耗，
        //注意设置合适的定位时间的间隔（最小间隔支持为1000ms），并且在合适时间调用stopLocation()方法来取消定位请求
        //在定位结束后，在合适的生命周期调用onDestroy()方法
        //在单次定位情况下，定位无论成功与否，都无需调用stopLocation()方法移除请求，定位sdk内部会移除
        //启动定位
        mLocationClient.startLocation();
    }

    public void stopLocation() {
        if (mLocationClient != null) {
            mLocationClient.stopLocation();
        }
    }

    public void onDestroy() {
        if (mLocationClient != null) {
            mLocationClient.stopLocation();
            mLocationClient.onDestroy();
        }
    }

    @Override
    public void onLocationChanged(AMapLocation aMapLocation) {
        if (mOnLocationCallBack == null) {
            return;
        }
        if (aMapLocation == null) {
            return;
        }
        if (aMapLocation.getErrorCode() == 0) {
            //定位成功回调信息，设置相关信息
            String address = aMapLocation.getAddress();
            if (TextUtils.isEmpty(address)) {
                //定位地址为空，通过经纬度反查定位地址
                searchByLatLon(aMapLocation.getLatitude(), aMapLocation.getLongitude());
                return;
            }
            LatLng latLng = new LatLng(aMapLocation.getLatitude(), aMapLocation.getLongitude());
            mOnLocationCallBack.onLocationSuccess(latLng, aMapLocation.getAddress());
            return;
        }
        //显示错误信息ErrCode是错误码，errInfo是错误信息，详见错误码表。
        LogUtils.d("AMapError", "location Error, ErrCode:"
                + aMapLocation.getErrorCode() + ", errInfo:" + aMapLocation.getErrorInfo());
        switch (aMapLocation.getErrorCode()) {
            case AMapLocation.ERROR_CODE_FAILURE_CONNECTION:
                //4 请求服务器过程中的异常，多为网络情况差，链路不通导致
                mOnLocationCallBack.onLocationError(ERROR_CODE_NO_NETWORK,
                        "当前网络未连接，连接后才能打卡", "去连接", new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                NetworkUtils.openWirelessSettings();
                            }
                        });
                break;
            case AMapLocation.ERROR_CODE_FAILURE_NOENOUGHSATELLITES:
                //14 GPS信号弱
                mOnLocationCallBack.onLocationError(ERROR_CODE_WEAK_GPS,
                        "GPS信号弱，请移到开阔的位置", "", null);
                break;
            default:
                mOnLocationCallBack.onLocationError(ERROR_CODE_OTHER,
                        "定位失败", "", null);
                break;
        }
    }

    /**
     * 定位地址为空，通过经纬度反查定位地址
     *
     * @param latitude  经度
     * @param longitude 纬度
     */
    private void searchByLatLon(double latitude, double longitude) {
        LatLonPoint latLonPoint = new LatLonPoint(latitude, longitude);
        RegeocodeQuery query = new RegeocodeQuery(latLonPoint, 200, GeocodeSearch.AMAP);
        GeocodeSearch geocodeSearch = new GeocodeSearch(mActivity);
        final LatLng latLng = new LatLng(latitude, longitude);
        geocodeSearch.setOnGeocodeSearchListener(new GeocodeSearch.OnGeocodeSearchListener() {
            @Override
            public void onRegeocodeSearched(RegeocodeResult regeocodeResult, int rCode) {
                if (rCode != 1000) {
                    mOnLocationCallBack.onLocationSuccess(latLng, "");
                    return;
                }
                if (regeocodeResult == null || regeocodeResult.getRegeocodeAddress() == null) {
                    mOnLocationCallBack.onLocationSuccess(latLng, "");
                    return;
                }
                mOnLocationCallBack.onLocationSuccess(latLng, regeocodeResult.getRegeocodeAddress().getFormatAddress());
            }

            @Override
            public void onGeocodeSearched(GeocodeResult geocodeResult, int i) {

            }
        });
        geocodeSearch.getFromLocationAsyn(query);
    }

    @Override
    public void onSuccess() {
        //定位权限申请成功
        startLocation();
    }

    @Override
    public void onFailed(boolean onRationale) {
        //用户拒绝开启定位权限
        if (onRationale) {
            //拒绝开启定位权限，并且设置不再询问
            showLocPermissionDeniedDialog();
        }
    }

    public interface OnLocationCallBack {

        /**
         * 定位中
         */
        void onLocating();

        /**
         * 定位失败
         *
         * @param errorCode       定位失败错误码
         * @param tip             错误提示，比如GPS未开启，无法获取定位等提示语
         * @param opt             错误引导，比如去开启等提示语
         * @param onClickListener 错误引导点击事件
         */
        void onLocationError(int errorCode, String tip, String opt, View.OnClickListener onClickListener);

        /**
         * 定位成功
         *
         * @param latLng  经纬度
         * @param address 详细地址
         */
        void onLocationSuccess(LatLng latLng, String address);
    }
}
