package com.xdja.pki.itsca.oer.asn1;

import com.xdja.pki.itsca.oer.app.bean.SM4CipherType;
import com.xdja.pki.itsca.oer.asn1.base.Choice;
import com.xdja.pki.itsca.oer.asn1.base.OctetString;
import com.xdja.pki.itsca.oer.utils.ByteArrayUtils;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import java.math.BigInteger;
import java.util.Vector;

/**
 * --  6.3.2.5 消息密文
 * --  AesCcmCipherText
 * SymmetricCipherText ::= CHOICE {
 * sm4Ecb  cipherText,
 * sm4Cbc  SM4CipherText,
 * sm4Cfb  SM4CipherText,
 * sm4Ofb  SM4CipherText,
 * aesCcm  AesCcmCipherText
 * }
 */
public class SymmetricCipherText extends Choice {
    private static Logger logger = LoggerFactory.getLogger(SymmetricCipherText.class);
    private CipherText sm4Ecb;
    private SM4CipherText sm4Cbc;
    private SM4CipherText sm4Cfb;
    private SM4CipherText sm4Ofb;
    private AesCcmCipherText aesCcm;

    public SymmetricCipherText(CipherText sm4Ecb) {
        this.sm4Ecb = sm4Ecb;
        this.sm4Cbc = null;
        this.sm4Cfb = null;
        this.sm4Ofb = null;
        this.aesCcm = null;
    }

    public SymmetricCipherText(SM4CipherText sm4Cipher, SM4CipherType sm4CipherType) {

        this.sm4Ecb = null;
        this.sm4Cbc = null;
        this.sm4Cfb = null;
        this.sm4Ofb = null;
        this.aesCcm = null;
        if (sm4CipherType.id == SM4CipherType.SM4_CBC.id) {
            this.addIndex(1);
            this.sm4Cbc = sm4Cipher;
        }
        if (sm4CipherType.id == SM4CipherType.SM4_CFB.id) {
            this.addIndex(2);
            this.sm4Cfb = sm4Cipher;
        }
        if (sm4CipherType.id == SM4CipherType.SM4_OFB.id) {
            this.addIndex(3);
            this.sm4Ofb = sm4Cipher;
        }
    }

    public SymmetricCipherText(AesCcmCipherText sm4Ecb) {
        this.addIndex(4);
        this.sm4Ecb = null;
        this.sm4Cbc = null;
        this.sm4Cfb = null;
        this.sm4Ofb = null;
        this.aesCcm = sm4Ecb;
    }

    public static SymmetricCipherText getInstance(byte[] data) throws Exception {
        ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText start data ", data);
        BigInteger choice = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        byte[] goal = new byte[data.length - 1];
        System.arraycopy(data, 1, goal, 0, goal.length);
        data = goal;
        SymmetricCipherText symmetricCipherText;
        if (choice.intValue() - 0x80 == 0) {
            logger.debug("SymmetricCipherText choice sm4Ecb");
            OctetString instance = CipherText.getInstance(data);
            CipherText cipherText = new CipherText();
            cipherText.setString(instance.getString());
            cipherText.setGoal(instance.getGoal());
            symmetricCipherText = new SymmetricCipherText(cipherText);
            symmetricCipherText.setGoal(cipherText.getGoal());
            ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText sm4Ecb data ", cipherText.getEncode());
        } else if (choice.intValue() - 0x80 == 1) {
            logger.debug("SymmetricCipherText choice sm4Cbc");
            SM4CipherText sm4CipherText = SM4CipherText.getInstance(data);
            symmetricCipherText = new SymmetricCipherText(sm4CipherText, SM4CipherType.SM4_CBC);
            symmetricCipherText.setGoal(sm4CipherText.getGoal());
            ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText sm4Cbc data ", sm4CipherText.getEncode());
        } else if (choice.intValue() - 0x80 == 2) {
            logger.debug("SymmetricCipherText choice sm4Cfb");
            SM4CipherText sm4CipherText = SM4CipherText.getInstance(data);
            symmetricCipherText = new SymmetricCipherText(sm4CipherText, SM4CipherType.SM4_CFB);
            symmetricCipherText.setGoal(sm4CipherText.getGoal());
            ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText sm4Cfb data ", sm4CipherText.getEncode());
        } else if (choice.intValue() - 0x80 == 3) {
            logger.debug("SymmetricCipherText choice sm4Ofb");
            SM4CipherText sm4CipherText = SM4CipherText.getInstance(data);
            symmetricCipherText = new SymmetricCipherText(sm4CipherText, SM4CipherType.SM4_OFB);
            symmetricCipherText.setGoal(sm4CipherText.getGoal());
            ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText sm4Ofb data ", sm4CipherText.getEncode());
        } else if (choice.intValue() - 0x80 == 4) {
            logger.debug("SymmetricCipherText choice aesCcm");
            AesCcmCipherText aesCcmCipherText = AesCcmCipherText.getInstance(data);
            symmetricCipherText = new SymmetricCipherText(aesCcmCipherText);
            symmetricCipherText.setGoal(aesCcmCipherText.getGoal());
            ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText aesCcm data ", aesCcmCipherText.getEncode());
        } else {
            logger.error("SymmetricCipherText choice is error " + choice.intValue());
            throw new Exception("不支持的类型");
        }
        ByteArrayUtils.printHexBinary(logger, "SymmetricCipherText lave data ", symmetricCipherText.getGoal());
        return symmetricCipherText;
    }

    public CipherText getSm4Ecb() {
        return sm4Ecb;
    }

    public SM4CipherText getSm4Cbc() {
        return sm4Cbc;
    }

    public SM4CipherText getSm4Cfb() {
        return sm4Cfb;
    }

    public SM4CipherText getSm4Ofb() {
        return sm4Ofb;
    }

    public AesCcmCipherText getAesCcm() {
        return aesCcm;
    }

    @Override
    public Vector getChoiceValues() {
        Vector vector = new Vector();
        if (null != sm4Ecb) {
            //0X80
            vector.add(sm4Ecb);
        }
        if (null != sm4Cbc) {
            //0x81
            vector.add(sm4Cbc);
        }
        if (null != sm4Cfb) {
            //0x82
            vector.add(sm4Cfb);
        }
        if (null != sm4Ofb) {
            //0x83
            vector.add(sm4Ofb);
        }
        if (null != aesCcm) {
            //0x84
            vector.add(aesCcm);
        }
        return vector;
    }
}
