package com.xdja.ca.helper;

import com.xdja.ca.asn1.*;
import com.xdja.ca.bean.BaseCMPInfo;
import com.xdja.ca.bean.SignedAndEnvelopedData;
import com.xdja.ca.constant.SdkConstants;
import com.xdja.ca.error.ErrorEnum;
import com.xdja.ca.pkcs7.Pkcs7Utils;
import com.xdja.ca.sdk.SdkResult;
import com.xdja.ca.service.CaSdkRedisCacheManagerService;
import com.xdja.ca.utils.*;
import com.xdja.ca.vo.FreeText;
import com.xdja.ca.vo.ManagerCertInfo;
import com.xdja.ca.vo.UserCertInfo;
import com.xdja.pki.apache.client.utils.json.JsonUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.cms.EncryptedContentInfo;
import org.bouncycastle.asn1.cms.IssuerAndSerialNumber;
import org.bouncycastle.asn1.cms.KeyTransRecipientInfo;
import org.bouncycastle.asn1.cms.RecipientIdentifier;
import org.bouncycastle.asn1.crmf.*;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.CertificationRequestInfo;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.jce.X509KeyUsage;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.naming.NamingException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.security.spec.RSAPublicKeySpec;
import java.util.*;

import static com.xdja.ca.error.ErrorEnum.getErrorEnumByCode;

/**
 * CMP消息处理类
 *
 * @author syg
 */
public class CmpMessageHelper {

    private static final Logger logger = LoggerFactory.getLogger(CmpMessageHelper.class);

    /**
     * 生成证书请求对象CertRequest (1)
     * @param certValidity
     * @param userCertDN
     * @param publicKey
     * @param signAlg
     * @return
     */
    public static CertRequest genCertRequest(Boolean isUKey, int certValidity, String userCertDN, byte[] publicKey, ASN1ObjectIdentifier signAlg, long certReqId, int certType) throws Exception {
        // 封装证书模板  可根据CA颁发证书模板来进行设置
        final CertTemplateBuilder myCertTemplate = new CertTemplateBuilder();
        // 有效期
        OptionalValidity myOptionalValidity = null;
        if (certValidity != 0) {
            Date notBefore = new Date();
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(notBefore);
            calendar.add(Calendar.DATE, certValidity);
            Date notAfter = calendar.getTime();
            ASN1EncodableVector optionalValidityV = new ASN1EncodableVector();
            optionalValidityV.add(new DERTaggedObject(true, 0, new Time(notBefore)));
            optionalValidityV.add(new DERTaggedObject(true, 1, new Time(notAfter)));
            myOptionalValidity = OptionalValidity.getInstance(new DERSequence(optionalValidityV));
            myCertTemplate.setValidity(myOptionalValidity);
        }

        myCertTemplate.setVersion(1);
        // 序列号 由CA侧进行补充
//        myCertTemplate.setSerialNumber(new ASN1Integer(1));
        // 证书签发者 这个由CA侧进行补充
//        myCertTemplate.setIssuer(issuerDN);

        myCertTemplate.setSigningAlg(new AlgorithmIdentifier(signAlg));
        myCertTemplate.setSubject(DnUtil.getRFC4519X500Name(userCertDN));
        SubjectPublicKeyInfo keyInfo = null;
        if (publicKey != null) {
            if (SM2ObjectIdentifiers.sm2SignWithSm3.equals(signAlg)) {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                } else {
                    if (isUKey) {
                        PublicKey pubKey = SdkCertUtils.convertSM2PublicKey(Base64.toBase64String(publicKey));
                        publicKey = pubKey.getEncoded();
                    }
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                }
            } else if (NISTObjectIdentifiers.nistSignAlgorithm.equals(signAlg)) {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                } else {
                    if (isUKey) {
                        PublicKey pubKey = SdkCertUtils.convertECPublicKey(Base64.toBase64String(publicKey), NISTNamedCurves.getName(SECObjectIdentifiers.secp256r1));
                        publicKey = pubKey.getEncoded();
                    }
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                }
            } else {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                } else {
                    if (isUKey) {
                        KeyFactory kf = KeyFactory.getInstance("RSA", new BouncyCastleProvider());
                        RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(BigIntegers.fromUnsignedByteArray(publicKey), BigInteger.valueOf(65537L));
                        PublicKey pubKey = kf.generatePublic(rsaPublicKeySpec);
                        publicKey = pubKey.getEncoded();
                    }
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                }
            }
        }
        myCertTemplate.setPublicKey(keyInfo);
        ExtensionsGenerator extgen = new ExtensionsGenerator();
        try {
            int bcku = 0;
            if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                // KeyUsage
                bcku = KeyUsage.digitalSignature | KeyUsage.nonRepudiation;
            } else if (certType == SdkConstants.CERT_TYPE_ENC_3) {
                bcku = KeyUsage.keyAgreement | KeyUsage.keyEncipherment | KeyUsage.dataEncipherment;
            }
            final X509KeyUsage ku = new X509KeyUsage(bcku);
            extgen.addExtension(Extension.keyUsage, false, ku);
        } catch (IOException e) {
            logger.info("封装CertRequest的扩展信息异常", e);
            throw new IOException();
        }
        myCertTemplate.setExtensions(extgen.generate());
        return new CertRequest(new ASN1Integer(certReqId), myCertTemplate.build(), null);
    }

    /**
     * 生成CertReqMsg (1)
     *
     * @param certRequests
     * @return
     * @throws Exception
     */
    public static CertReqMessages genCertReqMessages(CertRequest[] certRequests) {
        CertReqMsg[] certReqMsgs = new CertReqMsg[certRequests.length];
        for (int i = 0; i < certRequests.length; i++) {
            if (certRequests[i] != null) {
                CertReqMsg certReqMsg = new CertReqMsg(certRequests[i], null, null);
                certReqMsgs[i] = certReqMsg;
            }
        }
        CertReqMessages myCertReqMessages = new CertReqMessages(certReqMsgs);
        return myCertReqMessages;
    }


    /**
     * 封装PKIMessage (1)
     *
     * @param keyIndex
     * @param pwd
     * @param raDN
     * @param caDN
     * @param tagNo
     * @param recipNonce
     * @param senderNonce
     * @param asn1Encodable certReqMessages ErrorMsgContent
     * @param freeText
     * @return
     * @throws Exception
     */
    public static PKIMessage genPKIMessage(int keyIndex, String pwd, byte[] privateKey, X509Certificate caCert, String raDN, String caDN, int tagNo, byte[] recipNonce, byte[] senderNonce, String transId, ASN1Encodable asn1Encodable, String freeText, boolean isUseHsm) throws Exception {
        PKIBody myPKIBody = new PKIBody(tagNo, asn1Encodable);        // initialization request
        GeneralName raName = new GeneralName(DnUtil.getRFC4519X500Name(raDN));
        GeneralName caName = new GeneralName(DnUtil.getRFC4519X500Name(caDN));
        final PKIHeaderBuilder myPKIHeader = new PKIHeaderBuilder(PKIHeader.CMP_1999, raName, caName);
        myPKIHeader.setMessageTime(new ASN1GeneralizedTime(new Date()));
        myPKIHeader.setSenderNonce(new DEROctetString(senderNonce));
        myPKIHeader.setRecipNonce(new DEROctetString(recipNonce));
        myPKIHeader.setTransactionID(transId.getBytes());

        // 获取当前RA系统的证书签名算法
        String sigAlgName = caCert.getSigAlgName();
        ASN1ObjectIdentifier protectionAlg = null;
        if ("SM3withSM2".equalsIgnoreCase(sigAlgName)) {
            protectionAlg = SM2ObjectIdentifiers.sm2SignWithSm3;
        } else if ("SHA1withRSA".equalsIgnoreCase(sigAlgName)) {
            protectionAlg = RsaObjectIdentifiers.sha1WithRSA;
        } else if ("SHA256withRSA".equalsIgnoreCase(sigAlgName)) {
            protectionAlg = RsaObjectIdentifiers.sha256WithRSA;
        } else {
            protectionAlg = NISTObjectIdentifiers.nistSignAlgorithm;
        }
        logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：" + sigAlgName + " 算法oid为：" + protectionAlg.getId());

        myPKIHeader.setProtectionAlg(new AlgorithmIdentifier(protectionAlg));

        // 用于接收者可读的消息
        myPKIHeader.setFreeText(new PKIFreeText(freeText));

        // 调密码机进行签名
        PKIHeader myHeader = myPKIHeader.build();

        String protectionValue = null;
        // 私钥索引和访问控制码
        if (privateKey == null) {
            protectionValue = SdkHsmUtils.signByYunHsm(sigAlgName, keyIndex, pwd, Base64.toBase64String(CmpMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        } else {
            protectionValue = SdkHsmUtils.signByBC(sigAlgName, privateKey, Base64.toBase64String(CmpMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        }

//        GMSSLByteArrayUtils.printHexBinary(null,"data ",getProtectedBytes(myHeader, myPKIBody));
//        GMSSLByteArrayUtils.printHexBinary(null, "sign data", singValue.getBytes());
        if (StringUtils.isBlank(protectionValue)) {
            throw new Exception("使用密码机签名失败");
        }
        PKIMessage pkiMessage = new PKIMessage(myHeader, myPKIBody, new DERBitString(Base64.decode(protectionValue)));
        return pkiMessage;
    }

    /**
     * 封装并发送ErrorMsg消息
     *
     * @param errMsg
     * @param errCode
     * @param raDN
     * @param caDN
     * @param recipNonce
     * @param senderNonce
     * @param transId
     * @param url
     * @return
     */
    public static SdkResult genErrorPKIMsg(int keyIndex, String pwd, byte[] privateKey, X509Certificate caCert, X509Certificate[] caCerts, String raSignSn, int applyUserType, String errMsg, int errCode, String raDN, String caDN, byte[] recipNonce, byte[] senderNonce, String transId, String url, String protectAlgName, boolean isHttps, boolean isUseHsm) {
        SdkResult sdkResult = new SdkResult();
        // 封装ErrorMsgContent返回给CA
        logger.info("发送错误消息 ======== 1.封装ErrorMsgContent结构体");
        ErrorMsgContent errorMsgContent = null;
        try {
            errorMsgContent = CmpMessageHelper.genErrorMsgContent(PKIStatus.rejection, errCode, errMsg);
        } catch (Exception e) {
            logger.error("=============== 封装ErrorMsgContent异常", e);
            sdkResult.setError(ErrorEnum.ERROR_MSG_CONTENT_EXCEPTION);
            return sdkResult;
        }

        logger.info("发送错误消息 ======== 2.封装PKIMesage结构体");

        FreeText freeText = new FreeText();
        freeText.setApplyUserType(applyUserType);
        freeText.setRaSignSn(raSignSn);

        PKIMessage errorPKIMessage = null;
        try {
            errorPKIMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, privateKey, caCert, raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, SdkJsonUtils.object2Json(freeText), isUseHsm);
        } catch (Exception e) {
            logger.error("=============== 封装ErrorMsg的PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        logger.info("发送错误消息 ======== 3.发送证书错误消息");
        SdkResult postResult = null;
        try {
            postResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, privateKey, errorPKIMessage.getEncoded(), null,url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setError(postResult.getError());
                return sdkResult;
            }
        } catch (Exception e) {
            logger.error(" ============= 发送Http请求异常:", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        return sdkResult;
    }

    /**
     * 检查CMP响应  检查消息头(1)
     *
     * @param pkiMessage
     * @return
     */
    public static SdkResult checkCmpHeaderAndSign(byte[] pkiMessage, String raName, String caName, byte[] senderNonce, String transId, X509Certificate caCert, boolean isUseHsm) throws NamingException {
        logger.debug("CmpApi.checkCmpHeaderAndSign>>>>> raName:" + raName + " caName:" + caName + " senderNonce:" + Base64.toBase64String(senderNonce) + " transId:" + transId + " caCert:" + SdkCertUtils.certToFullB64(caCert));
        SdkResult sdkResult = new SdkResult();
        PKIMessage respObject = PKIMessage.getInstance(pkiMessage);
        if (respObject == null) {
            logger.error(" =================== No pkiMessage response message.");
            sdkResult.setError(ErrorEnum.NO_PKI_MESSAGE_RESP_MESSAGE);
            return sdkResult;
        }

        // 检测接收到的发送者是不是预期的CA发送的
        PKIHeader header = respObject.getHeader();
        if (header == null) {
            logger.error(" =================== No header in response message.");
            sdkResult.setError(ErrorEnum.NO_HEADER_IN_RESPONSE_MESSAGE);
            return sdkResult;
        }
        logger.info("================="+header.getSender().getName().toString());

        String caDN = DnUtil.getRFC4519X500Name(header.getSender().getName()).toString();
        if (StringUtils.isBlank(caDN) || !caDN.equalsIgnoreCase(DnUtil.getRFC4519X500Name(caName).toString())) {
            logger.error(" =================== received caDn is:" + caDN + " but expect:" + DnUtil.getRFC4519X500Name(caName).toString());
            sdkResult.setError(ErrorEnum.RECEIVED_CA_DN_NOT_EXPECT);
            return sdkResult;
        }

        // 判断CA返回的nonce的长度
        if (header.getSenderNonce().getOctets().length != 16) {
            logger.error(" =================== Wrong length of received sender nonce (made up by server). Is " + header.getSenderNonce().getOctets().length + " byte but should be 16.");
            sdkResult.setError(ErrorEnum.WRONG_LEN_OF_RECEIVED_SENDER_NONCE);
            return sdkResult;
        }

        //
        if (!Arrays.equals(header.getSenderNonce().getOctets(), senderNonce)) {
            logger.error(" =================== recipient nonce not the same as we sent away as the sender nonce. Sent: " + Arrays.toString(senderNonce) + " Received: " + Arrays.toString(header.getRecipNonce().getOctets()));
            sdkResult.setError(ErrorEnum.RECIPIENT_NONCE_NOT_THE_SAME_AS_WE_SENT);
            return sdkResult;
        }

        // 检测CA返回的transId是否和请求的一致
        String s = new String(header.getTransactionID().getOctets());
        if (!transId.equalsIgnoreCase(s)) {
            logger.error(" =================== transid is not the same as the one we sent");
            sdkResult.setError(ErrorEnum.TRANS_ID_IS_NOT_THE_SAME_AS_WE_SENT);
            return sdkResult;
        }

        if (caCert == null) {
            sdkResult.setError(ErrorEnum.CA_CERT_INFO_IS_ERROR);
            return sdkResult;
        }

        // 获取消息体保护算法
        AlgorithmIdentifier algId = header.getProtectionAlg();
        String protectionAlgId = algId.getAlgorithm().getId();
        if (algId == null || algId.getAlgorithm() == null || protectionAlgId == null) {
            logger.error("检查消息头和签名 ======  Not possible to get algorithm.");
            sdkResult.setError(ErrorEnum.NO_PROTECTION_ALG_IN_PKI_HEADER);
            return sdkResult;
        }

        logger.debug(" ============= RA收到CA的返回内容的保护数据算法oid:{}", protectionAlgId);
        // 获取CertRepMessage中的签名原文和签名值
        PublicKey caPublicKey = caCert.getPublicKey();
        byte[] data = CmpMessageHelper.getProtectedBytes(respObject);
        byte[] signData = respObject.getProtection().getBytes();
        boolean verifyResult = false;
        logger.debug("data>>>:[{}] signData>>>:[{}] caPublicKey>>>:[{}]", Base64.toBase64String(data),Base64.toBase64String(signData), Base64.toBase64String(caPublicKey.getEncoded()));
        try {
            if (isUseHsm) {
                verifyResult = SdkHsmUtils.verifyCertByYunHsm(protectionAlgId, caPublicKey, data, signData);
            } else {
                verifyResult = SdkHsmUtils.verifyCertByBC(protectionAlgId, caPublicKey, data, signData);
            }
        } catch (Exception e) {
            logger.error(" =================== 加密机验签异常", e);
            sdkResult.setError(ErrorEnum.GMSSL_VERIFY_SIGN_DATA_IS_EXCEPTION);
            return sdkResult;
        }
        if (!verifyResult) {
            logger.error(" =================== 加密机验签失败");
            sdkResult.setError(ErrorEnum.GMSSL_VERIFY_SIGN_DATA_IS_ERROR);
            return sdkResult;
        }

        return sdkResult;
    }

    /**
     * 解析各种CA响应的消息体
     * @param raSignPriKey  RA签名私钥
     * @param caCert 管理CA证书
     * @param applyUserType 申请用户类型
     * @param applyType 申请类型
     * @param pkiMessageBytes PKIMessage
     * @param transId 事务ID
     * @param userCertDN 用户证书DN
     * @param raSignSn RA签名证书SN
     * @param keyIndex 密钥索引
     * @param pwd 私钥密码
     * @param isUseHsm 是否使用密码机
     * @param caSdkRedisCacheManagerService CASDK缓存管理类
     * @return SdkResult
     */
    public static SdkResult resolveVarietyRepMessage(byte[] raSignPriKey, X509Certificate caCert, int applyUserType, int applyType,
                                                     byte[] pkiMessageBytes, String transId, String userCertDN,String raSignSn,int keyIndex,String pwd, boolean isUseHsm,
                                                     CaSdkRedisCacheManagerService caSdkRedisCacheManagerService) {
        SdkResult sdkResult = new SdkResult();
        PKIMessage pkiMessage = PKIMessage.getInstance(pkiMessageBytes);
        PKIBody body = pkiMessage.getBody();
        // 处理23-ErrorMsgContent 结构体
        if (body.getType() == PKIBody.TYPE_ERROR) {
            if(logger.isDebugEnabled()) {
                logger.debug("处理23-ErrorMsgContent 结构体");
            }
            SdkResult errMsgSdkResult = null;
            try {
                errMsgSdkResult = CmpMessageHelper.resolveErrorMsgContent(body);
            } catch (Exception e) {
                logger.error("解析ErrorMsgContent异常", e);
                sdkResult.setError(ErrorEnum.RESOLVE_ERROR_MSG_CONTENT_EXCEPTION);
                return sdkResult;
            }
            sdkResult.setError(errMsgSdkResult.getError());
            return sdkResult;
        }
        // 处理1,3,8-CertRepMessage结构体
        else if (body.getType() == PKIBody.TYPE_INIT_REP || body.getType() == PKIBody.TYPE_CERT_REP
                || body.getType() == PKIBody.TYPE_KEY_UPDATE_REP) {
            if(logger.isDebugEnabled()) {
                logger.debug("处理1,3,8-CertRepMessage结构体");
            }
            SdkResult certRepSdkResult = null;
            try {
                certRepSdkResult = CmpMessageHelper.resolveCertRepMessage(applyUserType, raSignPriKey, body, userCertDN,
                        transId, caCert,keyIndex,pwd, isUseHsm, caSdkRedisCacheManagerService, pkiMessage);
            } catch (Exception e) {
                logger.error("解析CertRepMessage异常", e);
                sdkResult.setError(ErrorEnum.RESOLVE_CERT_REP_MESSAGE_EXCEPTION);
                return sdkResult;
            }
            if (!certRepSdkResult.isSuccess()) {
                sdkResult.setError(certRepSdkResult.getError());
                return sdkResult;
            }
            sdkResult.setInfo(certRepSdkResult.getInfo());
            return sdkResult;

        }
        // 处理12-RevRepContent结构体
        else if (body.getType() == PKIBody.TYPE_REVOCATION_REP) {
            if(logger.isDebugEnabled()) {
                logger.debug("处理12-RevRepContent结构体");
            }
            SdkResult revRepSdkResult = null;
            try {
                revRepSdkResult = CmpMessageHelper.resolveRevRepContent(body);
            } catch (Exception e) {
                logger.error("解析RevRepContent异常", e);
                sdkResult.setError(ErrorEnum.RESOLVE_REV_REP_CONTENT_EXCEPTION);
                return sdkResult;
            }
            if (!revRepSdkResult.isSuccess()) {
                sdkResult.setError(revRepSdkResult.getError());
                return sdkResult;
            }
            sdkResult.setInfo(revRepSdkResult.getInfo());
        }
        // 处理10-KeyRecRepContent结构体
        else if (body.getType() == PKIBody.TYPE_KEY_RECOVERY_REP) {
            if(logger.isDebugEnabled()) {
                logger.debug("处理10-KeyRecRepContent结构体");
            }
            SdkResult recRepSdkResult = null;
            try {
                recRepSdkResult = CmpMessageHelper.resolveKeyRecRepContent(raSignPriKey,caCert,body,userCertDN,raSignSn,keyIndex,pwd,isUseHsm, pkiMessage);
            } catch (Exception e) {
                logger.error("解析KeyRecRepContent异常", e);
                sdkResult.setError(ErrorEnum.RESOLVE_KEY_REC_REP_CONTENT_EXCEPTION);
                return sdkResult;
            }
            if (!recRepSdkResult.isSuccess()) {
                sdkResult.setError(recRepSdkResult.getError());
                return sdkResult;
            }
            sdkResult.setInfo(recRepSdkResult.getInfo());
        }else {
            logger.info("Cert body tag is:[{}]", body.getType());
            sdkResult.setError(ErrorEnum.RA_NOT_SUPPORT_THIS_CERT_BODY_TAG);
        }
        return sdkResult;
    }

    /**
     * 解析CertRepMessage结构体
     * @param applyUserType 申请用户类型
     * @param raSignPriKey RA签名私钥
     * @param body 消息体
     * @param userCertDN 用户证书DN
     * @param transId 事务ID
     * @param caCert 管理CA证书
     * @param keyIndex 密钥索引
     * @param pwd 私钥密码
     * @param isUseHsm 是否使用密码机
     * @param caSdkRedisCacheManagerService CASDK缓存管理类
     * @param pkiMessage PKIMessage
     * @return SdkResult
     * @throws IOException 异常
     * @throws NamingException 异常
     */
    public static SdkResult resolveCertRepMessage(int applyUserType, byte[] raSignPriKey, PKIBody body, String userCertDN,
                                                  String transId, X509Certificate caCert, int keyIndex, String pwd, boolean isUseHsm,
                                                  CaSdkRedisCacheManagerService caSdkRedisCacheManagerService, PKIMessage pkiMessage) throws IOException, NamingException {
        SdkResult sdkResult = new SdkResult();
        // 获取certRepMessage结构体
        CertRepMessage certRepMessage = (CertRepMessage) body.getContent();
        if (certRepMessage == null) {
            logger.error("============== No CertRepMessage for certificate received.");
            sdkResult.setError(ErrorEnum.NO_CERT_REQ_MESSAGE_RECEIVED);
            return sdkResult;
        }

        CertResponse[] certResponses = certRepMessage.getResponse();

        UserCertInfo userCertInfo = new UserCertInfo();
        ManagerCertInfo managerCertInfo = new ManagerCertInfo();
        X509Certificate signCert = null;   // 保存下签名证书的
        X509Certificate encCert = null;   // 保存下加密证书的
        for (CertResponse resp : certResponses) {
            // 依次获取证书返回CertResponse
            if (resp == null) {
                logger.error("============== No CertResponse for certificate received.");
                sdkResult.setError(ErrorEnum.NO_CERT_RESPONSE_MESSAGE_RECEIVED);
                return sdkResult;
            }

            // 获取返回对象中的PKIStatusInfo结构体
            final PKIStatusInfo info = resp.getStatus();
            if (info == null) {
                logger.error("No PKIStatusInfo for certificate received.");
                sdkResult.setError(ErrorEnum.NO_PKI_STATUS_INFO_FOR_RECEIVE);
                return sdkResult;
            }

            // 获取返回的PkiStatus内容
            int pkiStatus = info.getStatus().intValue();
            if (pkiStatus != 0) {
                //错误原因
                DERUTF8String stringAt = info.getStatusString().getStringAt(0);
                logger.error("Received Status is [{}] but should be 0 because [{}]", pkiStatus, stringAt);
                //错误码
                String failInfo = String.valueOf(info.getFailInfo().intValue());
                ErrorEnum errorEnumByCode = getErrorEnumByCode(failInfo);
                if (null != errorEnumByCode) {
                    sdkResult.setError(errorEnumByCode);
                } else if ("10001".equals(failInfo)) {
                    sdkResult.setError(ErrorEnum.CA_ILLEGAL_REQUEST_PARAMETER);
                } else {
                    sdkResult.setError(ErrorEnum.CA_OPEN_API_INNER_EXCEPTION);
                }
                return sdkResult;
            }

            // 获取CertifiedKeyPair结构体
            final CertifiedKeyPair certifiedKeyPair = resp.getCertifiedKeyPair();
            if (certifiedKeyPair == null) {
                logger.error("No CertifiedKeyPair for certificate received.");
                sdkResult.setError(ErrorEnum.NO_CERTIFIED_KEY_PAIR_FOR_RECEIVED);
                return sdkResult;
            }

            // 获取CertOrEncCert结构体
            final CertOrEncCert certOrEncCert = certifiedKeyPair.getCertOrEncCert();
            if (certOrEncCert == null) {
                logger.error("No CertOrEncCert for certificate received.");
                sdkResult.setError(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
                return sdkResult;
            }
            CMPCertificate cmpCertificate = certOrEncCert.getCertificate();
            // 说明该CertificateKeyPair中包含了明文加密证书和加密私钥信封
            if (applyUserType == SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1) {
                // 与请求体中的requestId对应匹配
                if(logger.isDebugEnabled()) {
                    logger.debug("用户证书申请");
                }
                String cmpInfo = caSdkRedisCacheManagerService.getCaSdkCmpInfo(transId);
                BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(cmpInfo, BaseCMPInfo.class);
                if (baseCMPInfo == null) {
                    logger.error(" ============= No ra send transId.");
                    sdkResult.setError(ErrorEnum.NO_RA_SEND_TRANS_ID);
                    return sdkResult;
                }
                boolean isSignCert = false;
                long requestId = baseCMPInfo.getRequestId();
                long receiveCertReqId = resp.getCertReqId().getValue().longValue();
                if (receiveCertReqId == requestId) {
                    isSignCert = true;
                } else if (receiveCertReqId == -1) {
                    isSignCert = false;
                } else {
                    logger.error("=============== Received CertReqId is [{}] but should be [{}]", receiveCertReqId, requestId);
                    sdkResult.setError(ErrorEnum.RA_RECEIVED_CERT_REQ_ID_IS_ERROR);
                    return sdkResult;
                }

                SdkResult checkResult = CmpMessageHelper.checkCMPCert(caCert, cmpCertificate, userCertDN);
                if (!checkResult.isSuccess()) {
                    sdkResult.setError(checkResult.getError());
                    return sdkResult;
                }

                if (isSignCert) {
                    // 获取签名证书
                    signCert = (X509Certificate) checkResult.getInfo();
                    String base64SignCert = SdkCertUtils.certToFullB64(signCert);
                    userCertInfo.setSignCert(base64SignCert);
                } else {
                    // 获取明文加密证书
                    encCert = (X509Certificate) checkResult.getInfo();
                    String base64EncCert = SdkCertUtils.certToFullB64(encCert);
                    userCertInfo.setEncCert(base64EncCert);
                }

                EncryptedValue privateKey = certifiedKeyPair.getPrivateKey();

                // 获取私钥信封
                if (privateKey != null) {
//                    ASN1OctetString priKey = ASN1OctetString.getInstance(ASN1Sequence.getInstance(resp.getEncoded()).getObjectAt(3));
//                    userCertInfo.setEncPriKey(new String(priKey.getOctets()));
                    if (signCert == null) {
                        CertResponse signCertResp = certResponses[1];
                        try {
                            signCert = SdkCertUtils.convertDerCertToCert(signCertResp.getCertifiedKeyPair().getCertOrEncCert().getCertificate().getEncoded());
                        } catch (Exception e) {
                            logger.error("Not possible to create certificate.", e);
                            sdkResult.setError(ErrorEnum.NOT_POSSIBLE_TO_CREATE_CERT);
                            return sdkResult;
                        }
                    }
                    SignedAndEnvelopedData signedAndEnvelopedData = null;
                    try {
                        signedAndEnvelopedData = CmpMessageHelper.buildSignedAndEnvelopedData(raSignPriKey, isUseHsm, caCert.getSigAlgName(), SdkCertUtils.getIssuerByX509Cert(signCert), privateKey, signCert.getSerialNumber().toString(16), keyIndex, pwd);
                    } catch (Exception e) {
                        logger.error("Build SignedAndEnvelopedData From Encryptedvalue Exception. ", e);
                        sdkResult.setError(ErrorEnum.BUILD_SIGNEDANDENVLOPEDDATA_FROM_ENCRYPTEDVALUE_EXCPTION);
                        return sdkResult;
                    }
                    userCertInfo.setEncPriKey(Base64.toBase64String(signedAndEnvelopedData.getDEREncoded()));
                }
            }

            // 如果是管理类申请
            else if (applyUserType == SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
                // 与请求体中的requestId对应匹配
                if(logger.isDebugEnabled()) {
                    logger.debug("管理类申请");
                }
                String cmpInfo = caSdkRedisCacheManagerService.getCaSdkCmpInfo(transId);
                BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(cmpInfo, BaseCMPInfo.class);
                if (baseCMPInfo == null) {
                    logger.info("================ No ra send transId.");
                    sdkResult.setError(ErrorEnum.NO_RA_SEND_TRANS_ID);
                    return sdkResult;
                }
                boolean isSignCert = false;
                long requestId = baseCMPInfo.getRequestId();
                long receiveCertReqId = resp.getCertReqId().getValue().longValue();
                if (receiveCertReqId == requestId) {
                    isSignCert = true;
                } else if (receiveCertReqId == -1) {
                    isSignCert = false;
                } else {
                    logger.info( "================== Received CertReqId is [{}] but should be [{}]", receiveCertReqId, requestId);
                    sdkResult.setError(ErrorEnum.RA_RECEIVED_CERT_REQ_ID_IS_ERROR);
                    return sdkResult;
                }
                if (cmpCertificate != null) {
                    SdkResult checkResult = CmpMessageHelper.checkCMPCert(caCert, cmpCertificate, userCertDN);
                    if (!checkResult.isSuccess()) {
                        sdkResult.setError(checkResult.getError());
                        return sdkResult;
                    }
                    signCert = (X509Certificate) checkResult.getInfo();
                }
                if (isSignCert) {
                    // 获取签名证书
                    String base64SignCert = SdkCertUtils.certToFullB64(signCert);
                    managerCertInfo.setSignCert(base64SignCert);
                } else {
//                    ASN1OctetString encCert = ASN1OctetString.getInstance(ASN1Sequence.getInstance(resp.getEncoded()).getObjectAt(3));
//                    managerCertInfo.setEncEncCert(new String(encCert.getOctets()));

                    // 获取密文加密证书
                    EncryptedValue encryptedCert = certifiedKeyPair.getCertOrEncCert().getEncryptedCert();
                    if (signCert == null) {
                        CertResponse signCertResp = certResponses[1];
                        try {
                            signCert = SdkCertUtils.convertDerCertToCert(signCertResp.getCertifiedKeyPair().getCertOrEncCert().getCertificate().getEncoded());
                        } catch (Exception e) {
                            logger.error("Not possible to create certificate.", e);
                            sdkResult.setError(ErrorEnum.NOT_POSSIBLE_TO_CREATE_CERT);
                            return sdkResult;
                        }
                    }
                    SM2EnvelopedData envelopedData = null;
                    try {
                        envelopedData = CmpMessageHelper.buildEnvelopedData(caCert.getSigAlgName(), SdkCertUtils.getIssuerByX509Cert(signCert), encryptedCert, signCert.getSerialNumber().toString(16));
                    } catch (Exception e) {
                        logger.error("Build SignedAndEnvelopedData From Encryptedvalue Exception.", e);
                        sdkResult.setError(ErrorEnum.BUILD_SIGNEDANDENVLOPEDDATA_FROM_ENCRYPTEDVALUE_EXCPTION);
                        return sdkResult;
                    }
                    managerCertInfo.setEncEncCert(Base64.toBase64String(envelopedData.getEncoded()));
                }
            }
        }

        if(applyUserType == SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1) {
            // p7b封装
            SdkResult result = CmpMessageHelper.checkCMPCertAndBuildP7bCerts(pkiMessage.getExtraCerts(), signCert, encCert, isUseHsm);
            if(result.isSuccess()) {
                userCertInfo.setExtraCertsP7b((String)result.getInfo());
            } else {
                return result;
            }
        }

        if (applyUserType == SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1) {
            sdkResult.setInfo(userCertInfo);
        } else if (applyUserType == SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.info("managerCertInfo=========== [{}]", SdkJsonUtils.object2Json(managerCertInfo));
            sdkResult.setInfo(managerCertInfo);
        }
        return sdkResult;
    }

    /**
     * 解析错误消息结构体
     *
     * @param body
     * @return
     */
    public static SdkResult resolveErrorMsgContent(PKIBody body) {
        SdkResult sdkResult = new SdkResult();
        ErrorMsgContent errorMsgContent = (ErrorMsgContent) body.getContent();
        PKIStatusInfo info = errorMsgContent.getPKIStatusInfo();
        // TODO 此处可能也需要对CA返回的PKI的ErrorMsg消息体中的错误做特殊处理

        logger.info("========== CA返回的错误消息结构体 =========  errorCode:" + info.getFailInfo().intValue());
        logger.info("========== CA返回的错误消息结构体 =========  errorMsg:" + info.getStatusString().getStringAt(0));
        if (ErrorEnum.RA_CERT_ISSUE_STATUS_REVOKED.code ==  info.getFailInfo().intValue()) {
            sdkResult.setError(ErrorEnum.RA_CERT_ISSUE_STATUS_REVOKED);
        } else {
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_PKI_ERROR_MSG);
        }
        return sdkResult;
    }


    /**
     * 解析撤销响应结构体
     *
     * @param body
     * @return
     */
    public static SdkResult resolveRevRepContent(PKIBody body) {
        SdkResult sdkResult = new SdkResult();
        RevRepContent revRepContent = (RevRepContent) body.getContent();
        PKIStatusInfo[] pkiStatusInfos = revRepContent.getStatus();
        int status = PKIStatus.GRANTED;
        if (status == pkiStatusInfos[0].getStatus().intValue()) {
            logger.info("==========CA返回的撤销/冻结/解冻成功=========");
            return sdkResult;
        }else {
            String code = String.valueOf(pkiStatusInfos[0].getFailInfo().intValue());
            //错误原因
            PKIFreeText pkiFreeTexts = pkiStatusInfos[0].getStatusString();
            String failInfo = pkiFreeTexts.getStringAt(0).toString();
            logger.info("==========CA返回的撤销/冻结/解冻失败=========原因： " + failInfo);
            ErrorEnum errorEnumByCode = getErrorEnumByCode(code);
            if (null != errorEnumByCode){
                sdkResult.setError(errorEnumByCode);
            }else if("10001".equals(failInfo)){
                sdkResult.setError(ErrorEnum.CA_ILLEGAL_REQUEST_PARAMETER);
            }else{
                sdkResult.setError(ErrorEnum.CA_OPEN_API_INNER_EXCEPTION);
            }

        }
        return sdkResult;
    }


    /**
     * 解析密钥恢复响应结构体
     *
     * @param body
     * @return
     */
    private static SdkResult resolveKeyRecRepContent(byte[] raSignPriKey, X509Certificate caCert,PKIBody body, String userCertDN,
                                                     String raSignSn,int keyIndex,String pwd,boolean isUseHsm, PKIMessage pkiMessage) throws NamingException {
        SdkResult sdkResult = new SdkResult();
        KeyRecRepContent keyRecRepContent = (KeyRecRepContent) body.getContent();
        if (keyRecRepContent == null) {
            logger.info("============== No CertRepMessage for certificate received.");
            sdkResult.setError(ErrorEnum.NO_KEY_REC_REP_CONTENT_MESSAGE_RECEIVED);
            return sdkResult;
        }

        // 获取返回的PkiStatus内容
        PKIStatusInfo pkiStatusInfo = keyRecRepContent.getStatus();
        if (pkiStatusInfo == null) {
            logger.info("No PKIStatusInfo for certificate received.");
            sdkResult.setError(ErrorEnum.NO_PKI_STATUS_INFO_FOR_RECEIVE);
            return sdkResult;
        }
        int pkiStatus = pkiStatusInfo.getStatus().intValue();
        if (PKIStatus.GRANTED != pkiStatus) {

            //错误原因
            DERUTF8String stringAt = pkiStatusInfo.getStatusString().getStringAt(0);
            logger.error("Received Status is:{} but should be 0 because:{}", pkiStatus, stringAt);
            //错误码
            String failInfo = String.valueOf(pkiStatusInfo.getFailInfo().intValue());
            ErrorEnum errorEnumByCode = getErrorEnumByCode(failInfo);
            if (null != errorEnumByCode){
                sdkResult.setError(errorEnumByCode);
            }else if("10001".equals(failInfo)){
                sdkResult.setError(ErrorEnum.CA_ILLEGAL_REQUEST_PARAMETER);
            }else{
                sdkResult.setError(ErrorEnum.CA_OPEN_API_INNER_EXCEPTION);
            }
            return sdkResult;
        }

        CMPCertificate newSigCert = keyRecRepContent.getNewSigCert();
        CertifiedKeyPair[] keyPairHist = keyRecRepContent.getKeyPairHist();
        CertOrEncCert oldEncCert = keyPairHist[0].getCertOrEncCert();
        EncryptedValue encPprKey = keyPairHist[0].getPrivateKey();

        UserCertInfo userCertInfo = new UserCertInfo();

        if (null == newSigCert) {
            logger.info("No NewSignCert for certificate received.");
            sdkResult.setError(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
            return sdkResult;
        }
        SdkResult checkResult = CmpMessageHelper.checkCMPCert(caCert,newSigCert, userCertDN);
        if (!checkResult.isSuccess()) {
            sdkResult.setError(checkResult.getError());
            return sdkResult;
        }
        // 获取签名证书
        X509Certificate signCert = (X509Certificate) checkResult.getInfo();
        String base64SignCert = SdkCertUtils.certToFullB64(signCert);
        userCertInfo.setSignCert(base64SignCert);

        if (null == oldEncCert || null == oldEncCert.getCertificate()) {
            logger.info("No OldEncCert for certificate received.");
            sdkResult.setError(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
            return sdkResult;
        }
        checkResult = CmpMessageHelper.checkCMPCert(caCert,oldEncCert.getCertificate(), userCertDN);
        if (!checkResult.isSuccess()) {
            sdkResult.setError(checkResult.getError());
            return sdkResult;
        }
        // 获取明文加密证书
        X509Certificate encCert = (X509Certificate) checkResult.getInfo();
        String base64EncCert = SdkCertUtils.certToFullB64(encCert);
        userCertInfo.setEncCert(base64EncCert);


        // 获取CertifiedKeyPair结构体
        if (encPprKey == null) {
            logger.info("No encPprKey for certificate received.");
            sdkResult.setError(ErrorEnum.NO_CERTIFIED_KEY_PAIR_FOR_RECEIVED);
            return sdkResult;
        }
        // 获取私钥信封
        else {
            SignedAndEnvelopedData signedAndEnvelopedData = null;
            try {
                    signedAndEnvelopedData = CmpMessageHelper.buildSignedAndEnvelopedData(raSignPriKey, isUseHsm, caCert.getSigAlgName(), SdkCertUtils.getIssuerByX509Cert(signCert), encPprKey,signCert.getSerialNumber().toString(16),keyIndex,pwd);
                } catch (Exception e) {
                logger.error("Build SignedAndEnvelopedData From Encryptedvalue Exception.",e);
                sdkResult.setError(ErrorEnum.BUILD_SIGNEDANDENVLOPEDDATA_FROM_ENCRYPTEDVALUE_EXCPTION);
                return sdkResult;
            }
            userCertInfo.setEncPriKey(Base64.toBase64String(signedAndEnvelopedData.getDEREncoded()));
        }

        // p7b封装
        SdkResult result = CmpMessageHelper.checkCMPCertAndBuildP7bCerts(pkiMessage.getExtraCerts(), signCert, encCert, isUseHsm);
        if(result.isSuccess()) {
            userCertInfo.setExtraCertsP7b((String)result.getInfo());
        } else {
            return result;
        }
        sdkResult.setInfo(userCertInfo);
        return sdkResult;
    }

    /**
     * 检查用户证书并封装CA证书链
     * @param extraCerts CA证书链
     * @param signCert 签名证书
     * @param encCert 加密证书
     * @param isUseHsm 是否使用密码机
     * @return SdkResult info=p7b的CA证书链
     */
    private static SdkResult checkCMPCertAndBuildP7bCerts(CMPCertificate[] extraCerts, X509Certificate signCert, X509Certificate encCert, boolean isUseHsm) {
        /*
        1、遍历extraCerts
        2、Certificate转换成X509Certificate
        3、封装成p7b证书链
        4、验证用户证书
        */
        if (extraCerts != null && extraCerts.length > 0) {
            if(logger.isDebugEnabled()) {
                logger.debug("extraCerts.length:[{}]", extraCerts.length);
            }
            List<X509Certificate> extraCertsList = new ArrayList<>();
            boolean verifySuccess = false;
            X509Certificate x509Certificate = null;
            for (CMPCertificate extraCert : extraCerts) {
                try {
                    x509Certificate = GMSSLX509Utils.convertCertificate(extraCert.getX509v3PKCert());
                } catch (Exception e) {
                    logger.error("证书格式转换异常", e);
                    return SdkResult.failure(ErrorEnum.CMP_CERT_ENCODE_EXCEPTION);
                }
                // 验证证书
                if (!verifySuccess) {
                    try {
                        if(isUseHsm) {
                            verifySuccess = SdkHsmUtils.verifyCertByYunHsm(signCert, x509Certificate.getPublicKey());
                        } else {
                            verifySuccess = SdkHsmUtils.verifyCertByBC(signCert, x509Certificate.getPublicKey());
                        }
                        if(encCert != null) {
                            if(isUseHsm) {
                                verifySuccess = verifySuccess && SdkHsmUtils.verifyCertByYunHsm(encCert, x509Certificate.getPublicKey());
                            } else {
                                verifySuccess = verifySuccess && SdkHsmUtils.verifyCertByBC(encCert, x509Certificate.getPublicKey());
                            }
                        }
                    } catch (Exception e) {
                        logger.error("用户证书格式验证失败", e);
                        return SdkResult.failure(ErrorEnum.USER_CERT_VERIFY_FAIL);
                    }
                }
                extraCertsList.add(x509Certificate);
            }

            if (!verifySuccess) {
                logger.error("用户证书格式验证失败");
                return SdkResult.failure(ErrorEnum.USER_CERT_VERIFY_FAIL);
            }
            // p7b封装
            String extraCertsP7b;
            try {
                extraCertsList = SdkCertUtils.sortCerts(extraCertsList);
                extraCertsP7b = Pkcs7Utils.createCertChainByCerts(extraCertsList);
            } catch (Exception e) {
                logger.error("sdk接口-封装证书链异常", e);
                return SdkResult.failure(ErrorEnum.BUILD_TRAIN_CERT_P7b_IS_ERROR);
            }
            return SdkResult.success(extraCertsP7b);
        } else {
            logger.error("缺少CA证书链");
            return SdkResult.failure(ErrorEnum.CA_CERTS_NOT_EXIST);
        }
    }


    /**
     * 生成证书确认PKIMessage
     *
     * @param hash
     * @return
     */
    public static CertConfirmContent genCertConfirmContent(String hash, long certReqId) {
        CertStatus certStatus = new CertStatus(hash.getBytes(), new BigInteger(String.valueOf(certReqId)));
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(certStatus);
        CertConfirmContent certConfirmContent = CertConfirmContent.getInstance(new DERSequence(v));
        return certConfirmContent;
    }

    /**
     * 封装 撤销 RevReqContent
     *
     * @param caDN
     * @param userCertDN
     * @param signSn
     * @return
     */
    public static RevReqContent genRevReqContent(String caDN, String userCertDN, String signSn) throws NamingException {
        CertTemplateBuilder certTemplateBuilder = new CertTemplateBuilder();
        certTemplateBuilder.setIssuer(DnUtil.getRFC4519X500Name(caDN));
//        certTemplateBuilder.setSubject(new X500Name(userCertDN));
        String signSnDecimal = new BigInteger(signSn, 16).toString(10);
        certTemplateBuilder.setSerialNumber(new ASN1Integer(Long.valueOf(signSnDecimal)));

        // 此处可以封装多个需要撤销的证书信息 目前只封装一个
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(certTemplateBuilder.build());

//        ReasonFlags reasonFlags = new ReasonFlags(ReasonFlags.unused);
//        ExtensionsGenerator extGen = new ExtensionsGenerator();
//        extGen.addExtension(Extension.reasonCode, true, reasonFlags);
//        Extensions crlEntryDetails = extGen.generate();
//        v.add(crlEntryDetails);


        ASN1Sequence seq = new DERSequence(v);
        RevDetails revDetails = RevDetails.getInstance(seq);
        RevReqContent revReqContent = new RevReqContent(revDetails);
        return revReqContent;
    }


    /**
     * 生成错误确认消息
     *
     * @param pkiStatus    状态  PKIStatus.rejection
     * @param errorCode    错误码
     * @param errorDetails 错误细节描述
     * @return
     */
    public static ErrorMsgContent genErrorMsgContent(PKIStatus pkiStatus, int errorCode, String errorDetails) {
        PKIFreeText pkiFreeText = new PKIFreeText(errorDetails);
        PKIStatusInfo pkiStatusInfo = new PKIStatusInfo(pkiStatus);
        ErrorMsgContent errorMessage = new ErrorMsgContent(pkiStatusInfo, new ASN1Integer(errorCode), pkiFreeText);
        return errorMessage;
    }


    /**
     * 生成错误确认消息
     *
     * @param pkiStatus    状态  PKIStatus.rejection
     * @param statusString 状态信息
     * @param failInfo     失败信息  PKIFailureInfo.badDataFormat
     * @param errorCode    错误码
     * @param errorDetails 错误细节描述
     * @return
     */
    public static ErrorMsgContent genErrorMsgContent(int pkiStatus, String statusString, int failInfo, int errorCode, String errorDetails) {
        PKIFailureInfo pkiFailureInfo = new PKIFailureInfo(failInfo);
        PKIFreeText pkiFreeText = new PKIFreeText(statusString);
        PKIStatusInfo pkiStatusInfo = new PKIStatusInfo(PKIStatus.getInstance(pkiStatus), pkiFreeText, pkiFailureInfo);
        PKIFreeText errorDetailsPkiFreeText = new PKIFreeText(errorDetails);
        ErrorMsgContent errorMessage = new ErrorMsgContent(pkiStatusInfo, new ASN1Integer(errorCode), errorDetailsPkiFreeText);
        return errorMessage;
    }


    /**
     * 将PKIMessage结构体转为为header和body字节数组
     */
    public static byte[] getProtectedBytes(PKIMessage msg) {
        return getProtectedBytes(msg.getHeader(), msg.getBody());
    }


    /**
     * 获取header和body组合的字节数组
     *
     * @param header
     * @param body
     * @return
     */
    public static byte[] getProtectedBytes(PKIHeader header, PKIBody body) {
        byte[] res = null;
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(header);
        v.add(body);
        ASN1Encodable protectedPart = new DERSequence(v);
        try {
            ByteArrayOutputStream bao = new ByteArrayOutputStream();
            DEROutputStream out = new DEROutputStream(bao);
            out.writeObject(protectedPart);
            res = bao.toByteArray();
        } catch (Exception ex) {
            logger.error(ex.getLocalizedMessage(), ex);
        }
        return res;
    }


    /**
     * 将DER编码的公钥封装成P10证书请求
     *
     * @param pubDer
     * @param dn
     * @param caAlg
     * @return
     */
    public static String createP10FromPubKeyDer(byte[] pubDer, String dn, String caAlg) {
        String p10 = "";
        try {
            ASN1Sequence seq = ASN1Sequence.getInstance(pubDer);
            SubjectPublicKeyInfo spi = new SubjectPublicKeyInfo(seq);
            CertificationRequestInfo cf1 = new CertificationRequestInfo(new X509Name(dn), spi, null);

            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            DEROutputStream derOutStream = new DEROutputStream(outStream);
            derOutStream.writeObject(cf1);
            byte[] tbsdata = outStream.toByteArray();
            DERBitString dbs = new DERBitString(tbsdata);
            CertificationRequest cr = new CertificationRequest(cf1, new AlgorithmIdentifier(new DERObjectIdentifier(
                    "1.2.840.113549.1.1.4")), dbs);
            ByteArrayOutputStream outStream1 = new ByteArrayOutputStream();
            DEROutputStream derOutStream1 = new DEROutputStream(outStream1);
            derOutStream1.writeObject(cr);
            byte[] p10data = outStream1.toByteArray();

            p10 = new String(Base64.encode(p10data)).replace("\n", "").replaceAll("\r", "");
        } catch (Exception e) {
            logger.error("P10异常 ", e);
            p10 = "";
        }
        return p10;
    }

    /**
     * 校验CA返回的证书内容
     *
     * @param cmpCertificate
     * @param userCertDN
     * @return
     */
    private static SdkResult checkCMPCert(X509Certificate caCert, CMPCertificate cmpCertificate, String userCertDN) throws NamingException {
        SdkResult sdkResult = new SdkResult();
        if (cmpCertificate == null) {
            logger.info("No X509CertificateStructure for certificate received.");
            sdkResult.setError(ErrorEnum.NO_X509_CERT_FOR_RECEIVED);
            return sdkResult;
        }

        // 获取返回中的证书结构体
        byte encoded[] = new byte[0];
        try {
            encoded = cmpCertificate.getEncoded();
            if (encoded == null || encoded.length <= 0) {
                logger.info("No encoded certificate received");
                sdkResult.setError(ErrorEnum.NO_ENCODE_CERT_FOR_RECEIVED);
                return sdkResult;
            }
        } catch (IOException e) {
            logger.error("CMPCertificate Encode Exception.", e);
            sdkResult.setError(ErrorEnum.CMP_CERT_ENCODE_EXCEPTION);
            return sdkResult;
        }

        X509Certificate cert = null;
        try {
            cert = SdkCertUtils.convertDerCertToCert(encoded);
        } catch (Exception e) {
            logger.error("Not possible to create certificate.", e);
            sdkResult.setError(ErrorEnum.NOT_POSSIBLE_TO_CREATE_CERT);
            return sdkResult;
        }

        // 检测生成的证书中用户DN，是否和申请时的DN一致
       //X500Name certDN = X500Name.getInstance(cert.getSubjectX500Principal().getEncoded());
        String certDN = SdkCertUtils.getSubjectByX509Cert(cert);
        userCertDN = DnUtil.getRFC4519X500Name(userCertDN).toString();
        if (!certDN.equalsIgnoreCase(userCertDN)) {
            logger.error("Subject is:[{}] but should be:[{}]",certDN ,userCertDN);
            sdkResult.setError(ErrorEnum.RECEIVE_SUBJECT_DN_IS_NOT_SAME_APPLY);
            return sdkResult;
        }

        if (caCert == null) {
            sdkResult.setError(ErrorEnum.CA_CERT_INFO_IS_ERROR);
            return sdkResult;
        }
/* 多根情况下，CA证书为管理体系证书，仅作为验证PKI消息签名，不在签发证书
        // 检测用户证书的颁发者DN，是否和申请时的DN一致
        String caDN = SdkCertUtils.getSubjectByX509Cert(caCert);
        if (!SdkCertUtils.getIssuerByX509Cert(cert).equals(caDN)) {
            logger.error("Issuer is:[{}] but should be:[{}]" , SdkCertUtils.getIssuerByX509Cert(cert), caDN);
            sdkResult.setError(ErrorEnum.RECEIVE_ISSUE_DN_IS_NOT_SAME_APPLY);
            return sdkResult;
        }

        // RA验证是否是上级CA签发的
//        logger.info("CmpApi.checkCMPCert>>>> cert:" + SdkCertUtils.certToFullB64(cert) + " caCert:" + SdkCertUtils.certToFullB64(caCert));

        try {
            // GMSSLSM2SignUtils.verifyCertByYunHsm(cert, caCert.getPublicKey());
            cert.verify(caCert.getPublicKey());
        } catch (Exception e) {
            logger.error("Certificate not verifying.", e);
            sdkResult.setError(ErrorEnum.RA_VERIFY_CA_REP_CERT_ERROR);
            return sdkResult;
        }
*/
        sdkResult.setInfo(cert);
        return sdkResult;
    }

    /**
     * 构建加密私钥信封
     *
     * @return
     */
    private static SignedAndEnvelopedData buildSignedAndEnvelopedData(byte[] raSignPriKey, boolean isUseHsm, String caAlg, String issuerCaCertDn, EncryptedValue encryptedValue, String recipSignSn, int keyIndex, String pwd) throws Exception {
        // 创建SignedAndEnvelopedData一级元素
        ASN1Integer version = new ASN1Integer(1);
        ASN1Set digestAlgorithms = null;
        if ("SHA-1WithRSA".equalsIgnoreCase(caAlg) || "SHA1WithRSA".equalsIgnoreCase(caAlg) || "SHA256WithRSA".equalsIgnoreCase(caAlg)) {
            digestAlgorithms = new DERSet(new AlgorithmIdentifier(DigestObjectIdentifiers.sha1));
        } else {
            digestAlgorithms = new DERSet(new AlgorithmIdentifier(DigestObjectIdentifiers.sm3));
        }
        // 组装
        RecipientIdentifier rid = new RecipientIdentifier(new IssuerAndSerialNumber(DnUtil.getRFC4519X500Name(issuerCaCertDn), new BigInteger(recipSignSn, 16)));
        KeyTransRecipientInfo keyTransRecipientInfo = new KeyTransRecipientInfo(rid, encryptedValue.getKeyAlg(), new DEROctetString(encryptedValue.getEncSymmKey().getOctets()));
        ASN1Set recipientInfos = new DERSet(keyTransRecipientInfo);

        EncryptedContentInfo encryptedContentInfo = new EncryptedContentInfo(SymmetryObjectIdentifiers.sm4, encryptedValue.getSymmAlg(), new DEROctetString(encryptedValue.getEncValue().getOctets()));

        byte[] mergeData = SdkCertUtils.byteMergerAll(version.getEncoded(), recipientInfos.getEncoded(), digestAlgorithms.getEncoded(), encryptedContentInfo.getEncoded());
        ASN1Set signerInfos = Pkcs7Utils.makeSignerInfos(raSignPriKey, isUseHsm, caAlg, issuerCaCertDn, new BigInteger(recipSignSn, 16), keyIndex, pwd, mergeData);
        SignedAndEnvelopedData envelopedData = new SignedAndEnvelopedData(version, recipientInfos, digestAlgorithms, encryptedContentInfo, null, null, signerInfos);
        return envelopedData;
    }


    /**
     * 构建加密私钥信封
     *
     * @return
     */
    private static SM2EnvelopedData buildEnvelopedData(String caAlg, String issuerCaCertDn, EncryptedValue encryptedValue, String recipSignSn) throws Exception {
        logger.info("caAlg:{}, issuerCaCertDn:{},recipSignSn:{},encryptedValue:{}", caAlg, issuerCaCertDn, recipSignSn, Base64.toBase64String(encryptedValue.getEncoded()));
        // 创建SignedAndEnvelopedData一级元素
        ASN1Integer version = new ASN1Integer(1);
        // 组装
        RecipientIdentifier rid = new RecipientIdentifier(new IssuerAndSerialNumber(DnUtil.getRFC4519X500Name(issuerCaCertDn), new BigInteger(recipSignSn, 16)));
        KeyTransRecipientInfo keyTransRecipientInfo = new KeyTransRecipientInfo(rid, encryptedValue.getKeyAlg(), new DEROctetString(encryptedValue.getEncSymmKey().getOctets()));
        ASN1Set recipientInfos = new DERSet(keyTransRecipientInfo);

        EncryptedContentInfo encryptedContentInfo = new EncryptedContentInfo(SymmetryObjectIdentifiers.sm4, encryptedValue.getSymmAlg(), new DEROctetString(encryptedValue.getEncValue().getOctets()));

        SM2EnvelopedData envelopedData = new SM2EnvelopedData(version, recipientInfos, encryptedContentInfo);
        return envelopedData;
    }

    /**
     * 通过密码机生成随机数
     *
     * @return
     */
    public static byte[] genRandomByHsm(int length)  {
        //  String randomByHsm = GMSSLRandomUtils.generateRandomByYunhsm(length);
        byte[] normalSenderNonce = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(normalSenderNonce);
        //  byte[] random = Base64.decode(randomByHsm);
        return normalSenderNonce;
    }
}
