package com.xdja.ca.sdk;

import com.alibaba.fastjson.JSON;
import com.xdja.ca.asn1.NISTObjectIdentifiers;
import com.xdja.ca.asn1.RsaObjectIdentifiers;
import com.xdja.ca.asn1.SM2ObjectIdentifiers;
import com.xdja.ca.bean.BaseCMPInfo;
import com.xdja.ca.cache.RedisClient;
import com.xdja.ca.constant.SdkConstants;
import com.xdja.ca.error.ErrorEnum;
import com.xdja.ca.helper.CmpMessageHelper;
import com.xdja.ca.service.CaSdkRedisCacheManagerService;
import com.xdja.ca.service.impl.CaSdkRedisCacheManagerServiceImpl;
import com.xdja.ca.utils.ClientHttpUtils;
import com.xdja.ca.utils.SdkJsonUtils;
import com.xdja.ca.vo.FreeText;
import com.xdja.ca.vo.RevokeCertInfo;
import com.xdja.ca.vo.TempInfo;
import com.xdja.ca.vo.UpdateCertInfo;
import com.xdja.pki.apache.client.utils.json.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.cmp.CertConfirmContent;
import org.bouncycastle.asn1.cmp.PKIBody;
import org.bouncycastle.asn1.cmp.PKIMessage;
import org.bouncycastle.asn1.cmp.RevReqContent;
import org.bouncycastle.asn1.crmf.CertReqMessages;
import org.bouncycastle.asn1.crmf.CertRequest;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.naming.NamingException;
import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

/**
 * CMP对外保留的api方法
 * @author syg
 */
public class CmpApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private  CaSdkRedisCacheManagerService caSdkRedisCacheManagerService;

    /**
     * RA服务器证书对应的密码机索引
     */
    private int keyIndex;

    /**
     * 私钥访问控制码
     */
    private String pwd;

    /**
     * CA证书
     */
    private X509Certificate caCert;
    /**
     * CA证书  可为空
     */
    private X509Certificate[] caCerts;

    /**
     * RA服务器签名证书sn
     */
    private String raSignSn;

    /**
     * CA请求路径
     */
    private String caBaseUrl;

    /**
     * 是否走安全通道
     */
    private boolean isHttps;

    /**
     * 通道是否使用密码设备
     */
    private boolean isUseHsm;

    /**
     * 算法类型
     */
    private String protectAlgName;

    /**
     * RA服务器签名私钥
     */
    private byte[] raSignPriKey;

    private static final String CA_OPENAPI_SERVER_NAME = "/ca-openapi";

    public CmpApi(int keyIndex, String pwd, X509Certificate[] caCerts, X509Certificate caCert, String caServiceIp, int caServicePort, String raSignSn, String protectAlgName, boolean isHttps, boolean isUseHsm) {
        this.keyIndex = keyIndex;
        this.pwd = pwd;
        this.caCerts = caCerts;
        this.caCert = caCert;
        this.caBaseUrl = caServiceIp + ":" + caServicePort + CA_OPENAPI_SERVER_NAME;   // 例如: http://11.12.110.83:8080/ca-openapi
        this.raSignSn = raSignSn;
        this.protectAlgName = protectAlgName;
        this.isHttps = isHttps;
        this.isUseHsm = isUseHsm;
        this.caSdkRedisCacheManagerService = new CaSdkRedisCacheManagerServiceImpl();
    }
    public CmpApi(int keyIndex, String pwd, X509Certificate[] caCerts, X509Certificate caCert, String caServiceIp,
                  int caServicePort, String raSignSn, String protectAlgName, boolean isHttps, boolean isUseHsm,
                  RedisClient redisClient, int transIdExpireTime) {
        this.keyIndex = keyIndex;
        this.pwd = pwd;
        this.caCerts = caCerts;
        this.caCert = caCert;
        this.caBaseUrl = caServiceIp + ":" + caServicePort + CA_OPENAPI_SERVER_NAME;   // 例如: http://11.12.110.83:8080/ca-openapi
        this.raSignSn = raSignSn;
        this.protectAlgName = protectAlgName;
        this.isHttps = isHttps;
        this.isUseHsm = isUseHsm;
        this.caSdkRedisCacheManagerService = new CaSdkRedisCacheManagerServiceImpl(redisClient, transIdExpireTime);
    }

    public CmpApi(byte[] privateKey, X509Certificate[] caCerts, X509Certificate caCert, String caServiceIp, int caServicePort, String raSignSn, String protectAlgName, boolean isHttps, boolean isUseHsm) {
        this.raSignPriKey = privateKey;
        this.caCerts = caCerts;
        this.caCert = caCert;
        this.caBaseUrl = caServiceIp + ":" + caServicePort + CA_OPENAPI_SERVER_NAME;   // 例如: http://11.12.110.83:8080/ca-openapi
        this.raSignSn = raSignSn;
        this.protectAlgName = protectAlgName;
        this.isHttps = isHttps;
        this.isUseHsm = isUseHsm;
        this.caSdkRedisCacheManagerService = new CaSdkRedisCacheManagerServiceImpl();
    }
    public CmpApi(byte[] privateKey, X509Certificate[] caCerts, X509Certificate caCert, String caServiceIp, int caServicePort,
                  String raSignSn, String protectAlgName, boolean isHttps, boolean isUseHsm,
                  RedisClient redisClient, int transIdExpireTime) {
        this.raSignPriKey = privateKey;
        this.caCerts = caCerts;
        this.caCert = caCert;
        this.caBaseUrl = caServiceIp + ":" + caServicePort + CA_OPENAPI_SERVER_NAME;   // 例如: http://11.12.110.83:8080/ca-openapi
        this.raSignSn = raSignSn;
        this.protectAlgName = protectAlgName;
        this.isHttps = isHttps;
        this.isUseHsm = isUseHsm;
        this.caSdkRedisCacheManagerService = new CaSdkRedisCacheManagerServiceImpl(redisClient, transIdExpireTime);
    }

    /************************************************** CMP 公有接口 *************************************************************/

    /**
     * 用户证书申请 - 发送证书创建请求(1)
     *
     * @param applyUserType 申请用户类型  1：普通用户  2：管理员
     * @param transId       事务id
     * @param raDN          RA的唯一标识DN
     * @param caDN          CA的唯一标识DN
     * @param certValidity  证书有效期  (天)
     * @param tempInfo      模板相关信息结构体
     * @param signPubKey    签名公钥
     * @param encPubKey     加密公钥
     * @param signAlg       签名算法  例如: SM3WithSM2
     * @param certDN        用户自定义的证书DN
     * @return
     */
    public SdkResult sendIssuerCertReqMessages(Boolean isUKey, int applyUserType, String transId, String raDN, String caDN, int certValidity, TempInfo tempInfo, String signPubKey, String encPubKey, String signAlg, String certDN, Integer keyFormat) {
        logger.info("签发申请 ======== 【开始】申请事务Id为:{} ，用户类型为:{},raDN为:{},caDN为:{},申请有效期为:{},模板信息为:{},签名公钥为:{},加密公钥为:{},申请签名算法为:{},申请证书主体为:{}", transId, applyUserType, raDN, caDN, certValidity, SdkJsonUtils.object2Json(tempInfo), signPubKey, encPubKey, signAlg, certDN);
        SdkResult sdkResult = new SdkResult();

        logger.info("签发申请 ======== 0.参与校验");
        // 判断用户类型
        if (applyUserType != SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1 && applyUserType != SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.error("=============== 请求用户类型错误");
            sdkResult.setError(ErrorEnum.APPLY_USER_TYPE_IS_ERROR);
            return sdkResult;
        }

        // 对于申请证书的有效期限制
        if (certValidity <= 0) {
            logger.error("=============== 参数中证书有效期不可以小于等于0");
            sdkResult.setError(ErrorEnum.CERT_VALIDITY_CANNOT_LESS_ZERO);
            return sdkResult;
        }

        // 对入参进行判空处理
        if (tempInfo == null || StringUtils.isAnyBlank(transId, raDN, caDN, signPubKey, signAlg, certDN, tempInfo.tempNo)) {
            logger.error("=============== 参数中tempNo,transId,raDN,caDN,signPubKey,protectionAlg,certDN存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }
        // 获取recipNonce值
        logger.info("签发申请 ======== 1.向CA获取随机数");
        String url = "/v1/cmp";
        byte[] recipNonce = null;
        try {
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transId", transId);
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,paramMap, caBaseUrl + url, "application/pkixcmp",raSignSn,caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            recipNonce = ((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取CMP请求随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_CMP_RANDOM_IS_EXCEPTION);
            return sdkResult;
        }

        if (recipNonce == null) {
            logger.error("============== CA返回数据为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        ASN1ObjectIdentifier signAlgIde = null;
        if ("SM3withSM2".equalsIgnoreCase(signAlg)) {
            signAlgIde = SM2ObjectIdentifiers.sm2SignWithSm3;
        } else if ("sha-1WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha1WithRSA;
        } else if ("sha256WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha256WithRSA;
        } else if ("SHA256WithECDSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = NISTObjectIdentifiers.nistSignAlgorithm;
        } else {
            logger.error("===============  证书签名算法不支持:{}", signAlg);
            sdkResult.setError(ErrorEnum.CERT_SIGN_ALG_IS_NOT_SUPPORT);
            return sdkResult;
        }

        // 请求时间戳随机数
        long certReqId = caSdkRedisCacheManagerService.getIncCaCmpReqId();

        logger.info("签发申请 ======== 2.封装CertRequest结构体");
        // 根据不同的用户类型封装证书请求对象 CertRequset
        CertRequest[] certRequests = null;
        // 普通用户只提供签名P10，所以只有一个CertRequest对象
        if (applyUserType == SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1) {
            certRequests = new CertRequest[1];
            // 封装证书请求对象 CertRequset
            CertRequest certRequest = null;
            try {
                certRequest = CmpMessageHelper.genCertRequest(null,certValidity, certDN, signPubKey == null ? null : Base64.decode(signPubKey), signAlgIde, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
            } catch (Exception e) {
                logger.error(" ===============  封装【签名】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[0] = certRequest;
        }
        //  管理类实体提供签名P10和加密公钥，所以只有两个个CertRequest对象
        else if (applyUserType == SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            certRequests = new CertRequest[2];
            CertRequest signCertRequest;
            try {
                signCertRequest = CmpMessageHelper.genCertRequest(null, certValidity, certDN, signPubKey == null ? null : Base64.decode(signPubKey), signAlgIde, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
            } catch (Exception e) {
                logger.error(" ============= 封装【签名】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[0] = signCertRequest;

            CertRequest encCertRequest = null;
            try {
                encCertRequest = CmpMessageHelper.genCertRequest(isUKey, certValidity, certDN, Base64.decode(encPubKey), signAlgIde, -1, SdkConstants.CERT_TYPE_ENC_3);
            } catch (Exception e) {
                logger.error("============= 封装【加密】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[1] = encCertRequest;
        }

        // 封装证书请求消息对象 CertReqMessages
        logger.info("签发申请 ======== 3.封装CertReqMessages结构体");
        CertReqMessages certReqMessages = null;
        try {
            certReqMessages = CmpMessageHelper.genCertReqMessages(certRequests);
        } catch (Exception e) {
            logger.error(" ============== 封装certRequestMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        // RA产生十六字节随机数 senderNonce
        byte[] senderNonce = new byte[16];
        try {
            senderNonce = CmpMessageHelper.genRandomByHsm(senderNonce.length);
        } catch (Exception e) {
            logger.error("============== 通过密码机获取随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_RANDOM_BY_HSM_EXCEPTION);
            return sdkResult;
        }

        // 封装header中的freeText属性
        FreeText freeText = new FreeText();
        freeText.setApplyUserType(applyUserType);
        freeText.setTempInfo(tempInfo);            // 申请证书对应的模板信息
        freeText.setRaSignSn(raSignSn);// RA服务器签名证书sn
        freeText.setKeyFormat(keyFormat);//秘钥格式 可以为空
        // 缓存数据
        BaseCMPInfo baseCMPInfo = new BaseCMPInfo(senderNonce, recipNonce, transId, certReqId, 0);
        if(logger.isDebugEnabled()) {
            logger.debug("CASDK put transId:[{}] baseCMPInfo:[{}] ", transId, JsonUtils.object2Json(baseCMPInfo));
        }
        caSdkRedisCacheManagerService.cacheCaSdkCmpInfo(transId,JsonUtils.object2Json(baseCMPInfo));

        // 封装PKIMessage消息格式体
        logger.info("签发申请 ======== 4.封装PKIMessage结构体");
        PKIMessage pkiMessage = null;
        try {
            pkiMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, raSignPriKey, caCert, raDN, caDN, PKIBody.TYPE_INIT_REQ, recipNonce, senderNonce, transId, certReqMessages, JSON.toJSONString(freeText), isUseHsm);
        } catch (Exception e) {
            logger.error(" =============== 封装PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }
        // 2. 发送证书申请请求
        logger.info("签发申请 ======== 5.发送证书申请请求");
        url = "/v1/cmp";
        byte[] pkiInfo = null;
        try {
            SdkResult postResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, pkiMessage.getEncoded(), null,caBaseUrl + url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
            pkiInfo = (byte[]) postResult.getInfo();
        } catch (Exception e) {
            logger.error(" ============= 发送Http请求异常", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }
        if (pkiInfo == null) {
            logger.info("============== 接收CA返回的数据内容为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }
        // 检查返回消息的header和签名信息
        logger.info("签发申请 ======== 6.检查CA返回消息的header和签名信息");
        SdkResult checkCRGSdkResult;
        try {
            checkCRGSdkResult = CmpMessageHelper.checkCmpHeaderAndSign(pkiInfo, raDN, caDN, senderNonce, transId, caCert, isUseHsm);
        } catch (NamingException e) {
            logger.error("============= 检查CA返回消息的header和签名信息", e);
            sdkResult.setError(ErrorEnum.CERT_DN_IS_NOT_FORMAT);
            return sdkResult;
        }
        if (!checkCRGSdkResult.isSuccess()) {
            sdkResult.setErrorBean(checkCRGSdkResult.getErrorBean());
            logger.error("签发申请 ======== 6.1 解析CA返回的头和签名错误");
            return sdkResult;
        }
        // 获取返回body中的证书信息
        logger.info("签发申请 ======== 7.获取CA返回的证书信息");
        SdkResult resolveCmpRepSdkResult = CmpMessageHelper.resolveVarietyRepMessage(raSignPriKey, caCert, applyUserType,
                SdkConstants.CERT_APPLY_TYPE_ISSUE_1, pkiInfo, transId, certDN, raSignSn, keyIndex, pwd, isUseHsm, caSdkRedisCacheManagerService);
        if (!resolveCmpRepSdkResult.isSuccess()) {
            logger.error("签发申请 ======== 7.1 解析CA返回的消息体错误:{}", JsonUtils.object2Json(resolveCmpRepSdkResult.getErrorBean()));
            sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
            return sdkResult;
        }
        sdkResult.setInfo(resolveCmpRepSdkResult.getInfo());
        logger.info("签发申请 ======== 【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }


    /**
     * 用户证书申请 - 发送证书更新请求
     *
     * @param applyUserType  申请用户类型  1：普通用户  2：管理员
     * @param transId        事务id
     * @param raDN           RA的唯一标识DN
     * @param caDN           CA的唯一标识DN
     * @param certValidity   证书有效期  (天)
     * @param tempInfo       模板相关信息结构体
     * @param updateCertInfo 更新信息结构体
     * @param signSn         待更新证书(签名)序列号
     * @param certDN         用户自定义的证书DN
     * @return
     */
    public SdkResult sendUpdateCertReqMessages(int applyUserType, String transId, String raDN, String caDN, int certValidity, TempInfo tempInfo, UpdateCertInfo updateCertInfo, String signSn, String certDN, Integer keyFormat) {
        logger.info("更新申请 ======== 【开始】申请事务Id为:{} ，用户类型为:{},raDN为:{},caDN为:{},申请有效期为:{},模板信息为:{},待更新内容为:{},待更新证书签名sn为:{},申请证书主体为:{}", transId, applyUserType, raDN, caDN, certValidity, SdkJsonUtils.object2Json(tempInfo), SdkJsonUtils.object2Json(updateCertInfo), signSn, certDN);
        SdkResult sdkResult = new SdkResult();

        logger.info("更新申请 ======== 0.参与校验");
        if (applyUserType != SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1 && applyUserType != SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.info("请求用户类型错误");
            sdkResult.setError(ErrorEnum.APPLY_USER_TYPE_IS_ERROR);
            return sdkResult;
        }

        // 对于申请证书的有效期限制
        if (certValidity < 0) {
            logger.info("=============== 参数中证书有效期不可以小于0");
            sdkResult.setError(ErrorEnum.CERT_VALIDITY_CANNOT_LESS_ZERO);
            return sdkResult;
        }

        // 对入参进行判空处理
        if (tempInfo == null || updateCertInfo == null || StringUtils.isAnyBlank(transId, raDN, caDN, updateCertInfo.getSignAlg(), certDN, tempInfo.tempNo)) {
            logger.info("=============== 参数中tempNo,transId,raDN,caDN,signAlg,certDN存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }

        String url = "/v1/cmp";
        logger.info("更新申请 ======== 1.向CA获取随机数");
        byte[] recipNonce = null;
        try {
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transId", transId);
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,paramMap, caBaseUrl + url, "application/pkixcmp",raSignSn,caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            recipNonce = ((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取CMP请求随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_CMP_RANDOM_IS_EXCEPTION);
            return sdkResult;
        }
        if (recipNonce == null) {
            logger.info("============== CA返回数据为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        // 获取P10信息中的申请DN和公钥信息
        byte[] signPublicKey = null;
        if (updateCertInfo != null) {
            signPublicKey = updateCertInfo.getSignPublicKey() == null ? null : Base64.decode(updateCertInfo.getSignPublicKey());   // 密钥可更新 可不更新 不更新时 不传入公钥信息
        }
        String signAlg = updateCertInfo.getSignAlg();
        ASN1ObjectIdentifier signAlgIde = null;
        if ("SM3withSM2".equalsIgnoreCase(signAlg)) {
            signAlgIde = SM2ObjectIdentifiers.sm2SignWithSm3;
        } else if ("sha-1WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha1WithRSA;
        } else if ("sha256WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha256WithRSA;
        } else if ("SHA256WithECDSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = NISTObjectIdentifiers.nistSignAlgorithm;
        } else {
            logger.info("===============  证书签名算法不支持：{}", signAlg);
            sdkResult.setError(ErrorEnum.CERT_SIGN_ALG_IS_NOT_SUPPORT);
            return sdkResult;
        }

        // 请求时间戳随机数
        long certReqId = caSdkRedisCacheManagerService.getIncCaCmpReqId();

        // 封装证书请求对象 CertRequset
        logger.info("更新申请 ======== 2.封装CertRequset结构体");
        CertRequest[] certRequests = null;
        if (applyUserType == SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1) {
            certRequests = new CertRequest[1];
            // 封装证书请求对象 CertRequset
            CertRequest certRequest = null;
            try {
                certRequest = CmpMessageHelper.genCertRequest(null, certValidity, certDN, signPublicKey, signAlgIde, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
            } catch (Exception e) {
                logger.error(" ================= 封装【签名】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[0] = certRequest;
        } else if (applyUserType == SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            certRequests = new CertRequest[2];
            CertRequest signCertRequest = null;
            try {
                signCertRequest = CmpMessageHelper.genCertRequest(null, certValidity, certDN, signPublicKey, signAlgIde, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
            } catch (Exception e) {
                logger.error(" ================ 封装【签名】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[0] = signCertRequest;
            CertRequest encCertRequest = null;
            try {
                byte[] encPublicKey = StringUtils.isBlank(updateCertInfo.getEncPublicKey()) ? null : Base64.decode(updateCertInfo.getEncPublicKey());
                encCertRequest = CmpMessageHelper.genCertRequest(true, certValidity, certDN, encPublicKey, signAlgIde, certReqId, SdkConstants.CERT_TYPE_ENC_3);
            } catch (Exception e) {
                logger.error(" ============== 封装【加密】certRequest异常", e);
                sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
                return sdkResult;
            }
            certRequests[1] = encCertRequest;
        }


        // 封装证书请求消息对象 CertReqMessages
        logger.info("更新申请 ======== 3.封装CertReqMessages结构体");
        CertReqMessages certReqMessages = null;
        try {
            certReqMessages = CmpMessageHelper.genCertReqMessages(certRequests);
        } catch (Exception e) {
            logger.error("封装certRequestMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        // RA产生十六字节随机数 senderNonce
        byte[] senderNonce = new byte[16];
        try {
            senderNonce = CmpMessageHelper.genRandomByHsm(senderNonce.length);
        } catch (Exception e) {
            logger.error(" ============== 通过密码机获取随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_RANDOM_BY_HSM_EXCEPTION);
            return sdkResult;
        }

        // 封装header中的freeText属性
        FreeText freeText = new FreeText();
        freeText.setApplyUserType(applyUserType);
        freeText.setTempInfo(tempInfo);
        freeText.setSignSn(signSn);
        freeText.setRaSignSn(raSignSn);
        freeText.setKeyFormat(keyFormat);

        BaseCMPInfo baseCMPInfo = new BaseCMPInfo(senderNonce, recipNonce, transId, certReqId, 0);
        if(logger.isDebugEnabled()) {
            logger.debug("CASDK put transId:[{}] baseCMPInfo:[{}]", transId, JsonUtils.object2Json(baseCMPInfo));
        }
        caSdkRedisCacheManagerService.cacheCaSdkCmpInfo(transId,JsonUtils.object2Json(baseCMPInfo));

        // 封装PKIMessage消息格式体
        logger.info("更新申请 ======== 4.封装PKIMessage结构体");
        PKIMessage pkiMessage = null;
        try {
            pkiMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, raSignPriKey, caCert, raDN, caDN, PKIBody.TYPE_KEY_UPDATE_REQ, recipNonce, senderNonce, transId, certReqMessages, JSON.toJSONString(freeText), isUseHsm);
        } catch (Exception e) {
            logger.error(" =============== 封装PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }
        // 2. 发送证书申请请求
        logger.info("更新申请 ======== 5.发送证书更新请求");
        byte[] pkiInfo = null;
        try {
            SdkResult postResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, pkiMessage.getEncoded(), null,caBaseUrl + url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
            pkiInfo = (byte[]) postResult.getInfo();
        } catch (Exception e) {
            logger.error("============= 发送Http请求异常", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        if (pkiInfo == null) {
            logger.info("============== 接收CA返回的数据内容为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        // 检查返回消息的header和签名信息
        logger.info("更新申请 ======== 6.检查CA返回消息的header和签名信息");
        SdkResult checkCRGSdkResult;
        try {
            checkCRGSdkResult = CmpMessageHelper.checkCmpHeaderAndSign(pkiInfo, raDN, caDN, senderNonce, transId, caCert, isUseHsm);
        } catch (NamingException e) {
            logger.error("============= 检查CA返回消息的header和签名信息", e);
            sdkResult.setError(ErrorEnum.CERT_DN_IS_NOT_FORMAT);
            return sdkResult;
        }
        if (!checkCRGSdkResult.isSuccess()) {
            logger.info("更新申请 ======== 6.1 解析CA返回的头和签名错误");
            sdkResult.setErrorBean(checkCRGSdkResult.getErrorBean());
            return sdkResult;
        }

        // 获取返回body中的证书信息
        logger.info("更新申请 ======== 7.获取CA返回body中的更新后的证书信息");
        SdkResult resolveCmpRepSdkResult = CmpMessageHelper.resolveVarietyRepMessage(raSignPriKey, caCert, applyUserType,
                SdkConstants.CERT_APPLY_TYPE_UPDATE_2, pkiInfo, transId, certDN, raSignSn, keyIndex, pwd, isUseHsm, caSdkRedisCacheManagerService);
        if (!resolveCmpRepSdkResult.isSuccess()) {
            logger.info("更新申请 ======== 7.1 解析CA返回的消息体错误");
            sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
            return sdkResult;
        }
        sdkResult.setInfo(resolveCmpRepSdkResult.getInfo());
        logger.info("更新申请 ======== 【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }

    /**
     * 用户证书申请 - 发送证书撤销请求
     *
     * @param applyUserType 申请用户类型  1：普通用户  2：管理员
     * @param transId       事务id
     * @param raDN          RA的唯一标识DN
     * @param caDN          CA的唯一标识DN
     * @param signSn        需要撤销的签名证书序列号
     * @param userCertDN    用户自定义的证书DN
     * @param revokeType    撤销类型
     * @param revokeReason  用户输入的撤销原因  0 未指明原因  1 私钥泄露  3 身份信息变更  5 不在使用  9 权限回收
     * @return
     */
    public SdkResult sendRevokeCertReqMessages(int applyUserType, String transId, String raDN, String caDN, String signSn, String userCertDN, int revokeType, String revokeReason) {
        logger.info("撤销/冻结/解冻申请 ======== 【开始】申请事务Id为:{},用户类型为:{},raDN为:{},caDN为:{},待撤销签名证书sn为:{},待撤销证书主体为:{},撤销类型为:{},撤销原因为:{}", transId, applyUserType, raDN, caDN, signSn, userCertDN, revokeType, revokeReason);
        SdkResult sdkResult = new SdkResult();

        logger.info("撤销/冻结/解冻申请 ======== 0.参与校验");
        // 判断用户类型
        if (applyUserType != SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1 && applyUserType != SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.info("=============== 请求用户类型错误");
            sdkResult.setError(ErrorEnum.APPLY_USER_TYPE_IS_ERROR);
            return sdkResult;
        }

        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(transId, raDN, caDN, signSn)) {
            logger.info("=============== 参数中transId,raDN,caDN,signSn存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }

        String url = "/v1/cmp";
        logger.info("撤销/冻结/解冻申请 ======== 1.向CA获取随机数");
        byte[] recipNonce = null;
        try {
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transId", transId);
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,paramMap, caBaseUrl + url, "application/pkixcmp",raSignSn,caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            recipNonce = ((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取CMP请求随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_CMP_RANDOM_IS_EXCEPTION);
            return sdkResult;
        }

        // 生成 RevReqContent结构体
        logger.info("撤销/冻结/解冻申请 ======== 2.封装RevReqContent结构体");
        RevReqContent revReqContent = null;
        try {
            revReqContent = CmpMessageHelper.genRevReqContent(caDN, userCertDN, signSn);
        } catch (Exception e) {
            logger.error("封装RevReqContent异常", e);
            sdkResult.setError(ErrorEnum.MAKE_REV_REQ_CONTENT_EXCEPTION);
            return sdkResult;
        }

        // RA产生十六字节随机数 senderNonce
        byte[] senderNonce = new byte[16];
        try {
            senderNonce = CmpMessageHelper.genRandomByHsm(senderNonce.length);
        } catch (Exception e) {
            logger.error(" ============== 通过密码机获取随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_RANDOM_BY_HSM_EXCEPTION);
            return sdkResult;
        }

        FreeText freeText = new FreeText();
        freeText.setApplyUserType(applyUserType);
        freeText.setRevokeCertInfo(new RevokeCertInfo(revokeType, revokeReason));
        freeText.setRaSignSn(raSignSn);

        // 封装PKIMessage消息格式体
        logger.info("撤销/冻结/解冻申请 ======== 3.封装PkiMessage结构体");
        PKIMessage pkiMessage = null;
        try {
            pkiMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, raSignPriKey, caCert, raDN, caDN, PKIBody.TYPE_REVOCATION_REQ, recipNonce, senderNonce, transId, revReqContent, JSON.toJSONString(freeText), isUseHsm);
        } catch (Exception e) {
            logger.error("封装PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        logger.info("撤销/冻结/解冻申请 ======== 4.发送证书撤销请求");
        byte[] pkiInfo = null;
        try {
            SdkResult postResult =  ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, pkiMessage.getEncoded(),null, caBaseUrl + url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
            pkiInfo = (byte[]) postResult.getInfo();
        } catch (Exception e) {
            logger.error("============= 发送Http请求异常", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        if (pkiInfo == null) {
            logger.info("============== 接收CA返回的数据内容为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        // 3. 检查返回消息的header和签名信息
        logger.info("撤销/冻结/解冻申请 ======== 5.检查CA返回消息的header和签名信息");
        SdkResult checkCRGSdkResult;
        try {
            checkCRGSdkResult = CmpMessageHelper.checkCmpHeaderAndSign(pkiInfo, raDN, caDN, senderNonce, transId, caCert, isUseHsm);
        } catch (NamingException e) {
            logger.error(" ============= 检查CA返回消息的header和签名信息", e);
            sdkResult.setError(ErrorEnum.CERT_DN_IS_NOT_FORMAT);
            return sdkResult;
        }

        if (!checkCRGSdkResult.isSuccess()) {
            logger.info("撤销/冻结/解冻申请 ======== 5.1 解析CA返回的头和签名错误");
            sdkResult.setErrorBean(checkCRGSdkResult.getErrorBean());
            return sdkResult;
        }

        // 解析撤销响应结构体
        logger.info("撤销/冻结/解冻申请 ======== 6.获取CA返回body中的撤销后的信息");
        SdkResult resolveCmpRepSdkResult = CmpMessageHelper.resolveVarietyRepMessage(raSignPriKey, caCert, applyUserType,
                SdkConstants.CERT_APPLY_TYPE_REVOKE_3, pkiInfo, transId, userCertDN, raSignSn, keyIndex, pwd, isUseHsm, caSdkRedisCacheManagerService);
        if (!resolveCmpRepSdkResult.isSuccess()) {
            // 针对RA系统而言  CA返回该错误码  RA系统按照正常的业务流程处理  (异常补救方案)
            if(revokeType == SdkConstants.REV_REASON_CERTIFICATE_HOLD_6 ){
                if(ErrorEnum.RA_CERT_ISSUE_STATUS_FROZEND.code == resolveCmpRepSdkResult.getErrorBean().getErrCode()){
                    logger.info("冻结申请 ======== RA申请撤销的证书已被冻结");
                }else {
                    logger.info("冻结申请 ======== 6.1 解析CA返回的消息体错误");
                    sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
                }
            }else if(revokeType == SdkConstants.REV_REASON_REMOVE_FROM_CRL_8 ){
                if(ErrorEnum.RA_CERT_ISSUE_STATUE_NORMAL.code == resolveCmpRepSdkResult.getErrorBean().getErrCode()){
                    logger.info("解冻申请 ======== RA申请撤销的证书已被解冻");
                }else {
                    logger.info("解冻申请 ======== 6.1 解析CA返回的消息体错误");
                    sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
                }
            }else {
                if (ErrorEnum.RA_CERT_ISSUE_STATUS_REVOKED.code == resolveCmpRepSdkResult.getErrorBean().getErrCode()) {
                    logger.info("撤销申请 ======== RA申请撤销的证书已被撤销");
                }else {
                    logger.info("撤销申请 ======== 6.1 解析CA返回的消息体错误");
                    sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
                }
            }

            sdkResult.setInfo(resolveCmpRepSdkResult.getInfo());
            logger.info("撤销/冻结/解冻申请 ======== 【结束】申请事务Id为：{}", transId);
            return sdkResult;
        }

        sdkResult.setInfo(resolveCmpRepSdkResult.getInfo());
        logger.info("撤销/冻结/解冻申请 ======== 【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }

    /**
     * 用户证书申请 - 发送证书恢复请求
     *
     * @param transId    事务id
     * @param raDN       RA的唯一标识DN
     * @param caDN       CA的唯一标识DN
     * @param signPubKey 签名公钥
     * @param signAlg    生成证书使用的签名算法  例如: SM3WithSM2
     * @param certDN     用户自定义的证书DN
     * @return
     */
    public SdkResult sendRecoveryCertReqMessages(String transId, String raDN, String caDN, String signPubKey, String signAlg, String certDN, String recoverySignCertSn, Integer keyFormat) {
        logger.info("恢复申请 ======== 【开始】申请事务Id为:{},raDN为:{},caDN为:{},签名公钥为:{},签名算法为:{},申请证书主体为:{},待恢复的签名证书sn:{}", transId, raDN, caDN, signPubKey, signAlg, certDN, recoverySignCertSn);
        SdkResult sdkResult = new SdkResult();

        logger.info("恢复申请 ======== 0.参与校验");

        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(transId, raDN, caDN, signPubKey, signAlg, certDN)) {
            logger.info("=============== 参数中transId,raDN,caDN,signPubKey,signAlg,certDN存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }

        // 获取recipNonce值
        logger.info("恢复申请 ======== 1.向CA获取随机数");
        String url = "/v1/cmp";
        byte[] recipNonce = null;
        try {
            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transId", transId);
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,paramMap, caBaseUrl + url, "application/pkixcmp",raSignSn,caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            recipNonce = (byte[])getResult.getInfo();
        } catch (Exception e) {
            logger.error("===============  获取CMP请求随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_CMP_RANDOM_IS_EXCEPTION);
            return sdkResult;
        }
        if (recipNonce == null) {
            logger.info("============== CA返回数据为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        ASN1ObjectIdentifier signAlgIde = null;
        if ("SM3withSM2".equalsIgnoreCase(signAlg)) {
            signAlgIde = SM2ObjectIdentifiers.sm2SignWithSm3;
        } else if ("sha-1WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha1WithRSA;
        } else if ("sha256WithRSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = RsaObjectIdentifiers.sha256WithRSA;
        } else if ("SHA256WithECDSA".equalsIgnoreCase(signAlg)) {
            signAlgIde = NISTObjectIdentifiers.nistSignAlgorithm;
        } else {
            logger.error("===============  证书签名算法不支持:{}", signAlg);
            sdkResult.setError(ErrorEnum.CERT_SIGN_ALG_IS_NOT_SUPPORT);
            return sdkResult;
        }

        // 请求时间戳随机数
        long certReqId = caSdkRedisCacheManagerService.getIncCaCmpReqId();

        logger.info("恢复申请 ======== 2.封装CertRequest结构体");
        // 根据不同的用户类型封装证书请求对象 CertRequset
        // 普通用户只提供签名P10，所以只有一个CertRequest对象
        CertRequest[] certRequests = new CertRequest[1];
        // 封装证书请求对象 CertRequset
        CertRequest certRequest = null;
        try {
            certRequest = CmpMessageHelper.genCertRequest(null, 0, certDN, signPubKey == null ? null : Base64.decode(signPubKey), signAlgIde, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
        } catch (Exception e) {
            logger.error(" ===============  封装【签名】certRequest异常", e);
            sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
            return sdkResult;
        }
        certRequests[0] = certRequest;

        // 封装证书请求消息对象 CertReqMessages
        logger.info("恢复申请 ======== 3.封装CertReqMessages结构体");
        CertReqMessages certReqMessages = null;
        try {
            certReqMessages = CmpMessageHelper.genCertReqMessages(certRequests);
        } catch (Exception e) {
            logger.error(" ============== 封装certRequestMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_CERT_REQUEST_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        // RA产生十六字节随机数 senderNonce
        byte[] senderNonce = new byte[16];
        try {
            senderNonce = CmpMessageHelper.genRandomByHsm(senderNonce.length);
        } catch (Exception e) {
            logger.error(" ============== 通过密码机获取随机数异常", e);
            sdkResult.setError(ErrorEnum.GET_RANDOM_BY_HSM_EXCEPTION);
            return sdkResult;
        }

        // 封装header中的freeText属性
        FreeText freeText = new FreeText();
        freeText.setRaSignSn(raSignSn);            // RA服务器签名证书sn
        freeText.setSignSn(recoverySignCertSn);    // 待恢复密钥的签名证书sn
        freeText.setKeyFormat(keyFormat);
        // 缓存数据
        BaseCMPInfo baseCMPInfo = new BaseCMPInfo(senderNonce, recipNonce, transId, certReqId, 0);
        if(logger.isDebugEnabled()) {
            logger.debug("CASDK put transId:[{}] baseCMPInfo:[{}]", transId, JsonUtils.object2Json(baseCMPInfo));
        }
        caSdkRedisCacheManagerService.cacheCaSdkCmpInfo(transId,JsonUtils.object2Json(baseCMPInfo));

        // 封装PKIMessage消息格式体
        logger.info("恢复申请 ======== 4.封装PKIMessage结构体");
        PKIMessage pkiMessage = null;
        try {
            pkiMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, raSignPriKey, caCert, raDN, caDN, PKIBody.TYPE_KEY_RECOVERY_REQ, recipNonce, senderNonce, transId, certReqMessages, JSON.toJSONString(freeText), isUseHsm);
        } catch (Exception e) {
            logger.error(" =============== 封装PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        // 2. 发送证书申请请求
        logger.info("恢复申请 ======== 5.发送证书恢复申请请求");
        url = "/v1/cmp";
        byte[] pkiInfo = null;
        try {
            SdkResult postResult =ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, pkiMessage.getEncoded(), null,caBaseUrl + url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
            pkiInfo = (byte[]) postResult.getInfo();
        } catch (Exception e) {
            logger.error(" ============= 发送Http请求异常", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        if (pkiInfo == null) {
            logger.info("============== 接收CA返回的数据内容为空");
            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_INFO_IS_EMPTY);
            return sdkResult;
        }

        // 检查返回消息的header和签名信息
        logger.info("恢复申请 ======== 6.检查CA返回消息的header和签名信息");
        SdkResult checkCRGSdkResult;
        try {
            checkCRGSdkResult = CmpMessageHelper.checkCmpHeaderAndSign(pkiInfo, raDN, caDN, senderNonce, transId, caCert, isUseHsm);
        } catch (NamingException e) {
            logger.error(" ============= 检查CA返回消息的header和签名信息", e);
            sdkResult.setError(ErrorEnum.CERT_DN_IS_NOT_FORMAT);
            return sdkResult;
        }
        if (!checkCRGSdkResult.isSuccess()) {
            logger.info("恢复申请 ======== 6.1 解析CA返回的头和签名错误");
            sdkResult.setErrorBean(checkCRGSdkResult.getErrorBean());
            return sdkResult;
        }

        // 获取返回body中的证书信息
        logger.info("恢复申请 ======== 7.获取CA返回的证书信息");
        SdkResult resolveCmpRepSdkResult = CmpMessageHelper.resolveVarietyRepMessage(raSignPriKey, caCert,
                SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1, SdkConstants.CERT_APPLY_TYPE_RECOVERY_4, pkiInfo, transId, certDN, raSignSn, keyIndex, pwd, isUseHsm, caSdkRedisCacheManagerService);
        if (!resolveCmpRepSdkResult.isSuccess()) {
            logger.info("恢复申请 ======== 7.1 解析CA返回的消息体错误");
            sdkResult.setErrorBean(resolveCmpRepSdkResult.getErrorBean());
            return sdkResult;
        }

        sdkResult.setInfo(resolveCmpRepSdkResult.getInfo());
        logger.info("恢复申请 ======== 【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }


    /**
     * 发送证书申请正确确认消息到CA
     *
     * @param applyUserType 申请用户类型  1：普通用户  2：管理员
     * @param transId       事务id
     * @param raDN          RA的唯一标识DN
     * @param caDN          CA的唯一标识DN
     */
    public SdkResult sendCertConfirmContent(int applyUserType, String transId, String raDN, String caDN, X509Certificate caCert) {
        logger.info("发送证书 签发和更新的 确认消息 ======== 【开始】申请事务Id为：{} ", transId);
        // 封装PKIConfirmContent返回给CA  
        SdkResult sdkResult = new SdkResult();

        if (applyUserType != SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1 && applyUserType != SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.info(" ================ 请求用户类型错误");
            sdkResult.setError(ErrorEnum.APPLY_USER_TYPE_IS_ERROR);
            return sdkResult;
        }

        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(transId, raDN, caDN, caDN)) {
            logger.info("=============== 参数中transId,raDN,caDN,caDN存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }

        String url = "/v1/cmp";
        String cmpInfo = caSdkRedisCacheManagerService.removeCaSdkCmpInfo(transId);
        if (cmpInfo == null) {
            sdkResult.setError(ErrorEnum.LOCAL_CMP_CACHE_IS_EMPTY);
            return sdkResult;
        }
        BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(cmpInfo, BaseCMPInfo.class);
        if (baseCMPInfo == null) {
            logger.error(" ================== 未找到RA发从的该transId:[{}]" ,transId);
            if(logger.isDebugEnabled()) {
                logger.debug(" =================== baseCMPInfo:[{}]", SdkJsonUtils.object2Json(baseCMPInfo));
            }
            sdkResult.setError(ErrorEnum.CANNOT_GET_TRANS_ID_FORM_LOCAL_CACHE);
            return sdkResult;
        }

        byte[] recipNonce = baseCMPInfo.getRecipientNonce();
        byte[] senderNonce = baseCMPInfo.getSenderNonce();
        long certReqId = baseCMPInfo.getRequestId();

        logger.info("发送证书 签发和更新的 确认消息 ======== 1.封装CertConfirmContent结构体");
        CertConfirmContent certConfirmContent = null;
        try {
            certConfirmContent = CmpMessageHelper.genCertConfirmContent(transId, certReqId);
        } catch (Exception e) {
            logger.info("封装CertConfirmContent异常", e);
            sdkResult.setError(ErrorEnum.MAKE_CERT_CONFIRM_CONTENT_EXCEPTION);
            return sdkResult;
        }

        FreeText freeText = new FreeText();
        freeText.setApplyUserType(applyUserType);
        freeText.setRaSignSn(raSignSn);

        logger.info("发送证书 签发和更新的 确认消息 ======== 2.封装PkiMessage结构体");
        PKIMessage certConfirmPKIMessage = null;
        try {
            certConfirmPKIMessage = CmpMessageHelper.genPKIMessage(keyIndex, pwd, raSignPriKey, caCert, raDN, caDN, PKIBody.TYPE_CERT_CONFIRM, recipNonce, senderNonce, transId, certConfirmContent, SdkJsonUtils.object2Json(freeText), isUseHsm);
        } catch (Exception e) {
            logger.info("封装PKIMessage异常", e);
            sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
            return sdkResult;
        }
        logger.info("发送证书 签发和更新的 确认消息 ======== 3.发送证书证书确认消息");
        try {
            SdkResult postResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, certConfirmPKIMessage.getEncoded(), null,caBaseUrl + url, "application/pkixcmp", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info(" ============= 发送Http请求异常", e);
            sdkResult.setError(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
            return sdkResult;
        }

        logger.info("sendCertConfirmContent.sendCmpHttpPost.result>>>>[{}]", SdkJsonUtils.object2Json(sdkResult));
        caSdkRedisCacheManagerService.removeCaSdkCmpInfo(transId);
        logger.info("发送证书 签发和更新的 确认消息 ========  【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }


    /**
     * 发送错误确认消息到CA
     *
     * @param applyUserType 申请用户类型  1：普通用户  2：管理员
     * @param transId       事务id
     * @param raDN          RA的唯一标识DN
     * @param caDN          CA的唯一标识DN
     * @param errorCode     错误码
     * @param errorMsg      错误消息
     */
    public SdkResult sendErrorMsgContent(int applyUserType, String transId, String raDN, String caDN, int errorCode, String errorMsg) {
        logger.info("发送错误消息 ======== 【开始】申请事务Id为：{}", transId);
        SdkResult sdkResult = new SdkResult();

        logger.info("发送错误消息 ======== 参与校验");
        if (applyUserType != SdkConstants.APPLY_USER_TYPE_NORMAL_USER_1 && applyUserType != SdkConstants.APPLY_USER_TYPE_ADMIN_2) {
            logger.info("请求用户类型错误");
            sdkResult.setError(ErrorEnum.APPLY_USER_TYPE_IS_ERROR);
            return sdkResult;
        }

        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(transId, raDN, caDN)) {
            logger.info("=============== 参数中transId,raDN,caDN存在空值");
            sdkResult.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            return sdkResult;
        }

        String cmpInfo = caSdkRedisCacheManagerService.removeCaSdkCmpInfo(transId);
        if (cmpInfo == null) {
            logger.info("本地缓存数据为空");
            sdkResult.setError(ErrorEnum.LOCAL_CMP_CACHE_IS_EMPTY);
            return sdkResult;
        }
        BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(cmpInfo, BaseCMPInfo.class);
        if (baseCMPInfo == null) {
            logger.info(" ================== 未找到RA发从的该transId:{}", transId);
            sdkResult.setError(ErrorEnum.CANNOT_GET_TRANS_ID_FORM_LOCAL_CACHE);
            return sdkResult;
        }
        byte[] recipNonce = baseCMPInfo.getRecipientNonce();
        byte[] senderNonce = baseCMPInfo.getSenderNonce();

        SdkResult errorResult = CmpMessageHelper.genErrorPKIMsg(keyIndex, pwd, raSignPriKey, caCert, caCerts, raSignSn, applyUserType, errorMsg, errorCode, raDN, caDN, recipNonce, senderNonce, transId, caBaseUrl + "/v1/cmp", protectAlgName, isHttps, isUseHsm);
        if (!errorResult.isSuccess()) {
            sdkResult.setErrorBean(errorResult.getErrorBean());
        }
        caSdkRedisCacheManagerService.removeCaSdkCmpInfo(transId);
        logger.info("发送错误消息 ======== 【结束】申请事务Id为:{}", transId);
        return sdkResult;
    }

    /************************************************* 非CMP协议公有接口 ********************************************************/
    /**
     * 获取用户CA证书失效时间
     * @return
     */
    public SdkResult getUserCACertFileTime(Long caId){
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/userCAFileTime/" + caId;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url, null,raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            sdkResult.setInfo(getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取用户CA失效时间异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }

        return sdkResult;
    }

    /**
     * 获取所有用户CA
     * @return
     */
    public SdkResult getUserCA(){
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/userCA/list";
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url, null,raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            sdkResult.setInfo(getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取所有用户CA异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }

        return sdkResult;
    }

    /**
     * 获取当前RA系统的BaseDN
     *
     * @return
     */
    public SdkResult getRaBaseDN() {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/ra/baseDn";
        String raBaseDN = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url, null,raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            raBaseDN = new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取RA系统BaseDN异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(raBaseDN);
        return sdkResult;
    }

    /**
     * 获取某个证书的状态
     *
     * @param signSn 获取证书状态的签名证书sn
     * @return
     */
    public SdkResult getCertStatus(String signSn) {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/cert/status/" + signSn;
        String certStatus = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url,null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            certStatus = new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取证书状态异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(certStatus);
        return sdkResult;
    }

    /**
     * 下载用户证书数据流信息
     *
     * @param signSn 签名证书序列号
     * @return
     */
    public SdkResult downloadCertByteInfo(String signSn) {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/cert/download/" + signSn;
        String certByte = null;
        try {
            SdkResult getResult =  ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,null, caBaseUrl + url,null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            certByte = new String((byte[]) getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  下载用户证书异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(certByte);
        return sdkResult;
    }

    /**
     * 获取证书详细信息(基本域和扩展域)
     *
     * @param signSn 签名证书序列号
     * @return
     */
    public SdkResult getCertDetailInfo(String signSn) {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/cert/detail/" + signSn;
        String detail = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url,null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            detail = new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取证书详情异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(detail);
        return sdkResult;
    }

    /**
     * 同步CA授权给当前RA系统的模板信息
     *
     * @return
     */
    public SdkResult getCertTemplateList() {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/template/list";
        String list = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url,null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            list = new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取证书模板列表异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(list);
        return sdkResult;
    }

    /**
     * 获取某个证书模板的详细信息
     *
     * @param tempNo 模板编号
     * @return
     */
    public SdkResult getCertTemplateDetailInfo(String tempNo) {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/template/detail/" + tempNo;
        String tempDetail = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null,null, caBaseUrl + url, null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            tempDetail =new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取证书模板详情异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(tempDetail);
        return sdkResult;
    }

    /**
     * 获取RA管理员证书模板信息
     *
     * @return
     * @throws IOException
     */
    public SdkResult getRaOperatorCertTemp() {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/template/manage";
        String tempDetail = null;
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, null, null,caBaseUrl + url, null, raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"get");
            if (!getResult.isSuccess()) {
                sdkResult.setErrorBean(getResult.getErrorBean());
                return sdkResult;
            }
            tempDetail = new String((byte[])getResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取管理员证书模板详情异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(tempDetail);
        return sdkResult;
    }

    /**
     * RA管理员登录认证
     *
     * @return
     * @throws IOException
     */
    public SdkResult raAdminLoginAuthen(String signSn) {
        SdkResult sdkResult = new SdkResult();
        String url = "/v1/api/ra/login";
        Map<String, String> map = new HashMap<>();
        map.put("sn", signSn);
        String authenInfo = null;
        try {
            SdkResult postResult = ClientHttpUtils.sendApacheClientRequest(keyIndex, pwd, raSignPriKey, SdkJsonUtils.object2Json(map).getBytes(),null, caBaseUrl + url, "application/json", raSignSn, caCerts, protectAlgName, isHttps, isUseHsm,"post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorBean(postResult.getErrorBean());
                return sdkResult;
            }
            authenInfo = new String((byte[])postResult.getInfo());
        } catch (Exception e) {
            logger.error("===============  管理员登录认证异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }
        sdkResult.setInfo(authenInfo);
        return sdkResult;
    }

    /**
     * 测试CA服务连通性
     *
     * @param caServerIp
     * @param caServerPort
     * @return
     */
    public SdkResult sendCaServerMessages(String caServerIp, int caServerPort, String p7bHashCode, X509Certificate[] caCerts) {
        logger.info("测试CA服务连通性================");
        SdkResult sdkResult = new SdkResult();

        String url = caServerIp + ":" + caServerPort + "/ca-openapi/v1/api/interface/test";
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put("certChainHash", p7bHashCode);
        try {
            SdkResult getResult = ClientHttpUtils.sendApacheClientRequestInit(null,paramMap, url,null, protectAlgName , isHttps,"get",isUseHsm);
            if (!getResult.isSuccess()) {
                if(getResult.getErrorBean().errCode != 11451){
                    sdkResult.setError(ErrorEnum.TEST_CA_SERVER_CONNECT_ERROR);
                    return sdkResult;
                } else {
                    return getResult;
                }
            }

        } catch (Exception e) {
            logger.error("===============  获取CA系统运行状态异常", e);
            sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
            return sdkResult;
        }

        return sdkResult;
    }
}
