package com.xdja.ca.sdk;

import com.xdja.ca.error.ErrorBean;
import com.xdja.ca.error.ErrorEnum;
import java.io.Serializable;

/**
 * 通用返回结果Bean
 * @author syg
 *
 */
public class SdkResult implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * 执行结果 0-成功
	 */
	public final static int SUCCESS = 0;
	/**
	 * 业务异常结果(兼容Service层直接返回业务异常枚举情况)
	 */
	public final static int FAILURE = -1;

	/**
	 * 执行结果
	 */
	private int code;
	/**
	 * 业务数据
	 */
	private Object info;
	/**
	 * 异常对象
	 */
	private ErrorEnum error;

	private ErrorBean errorBean;

	/**
	 * 无参构造函数
	 */
	public SdkResult() {}

	/**
	 * 构造函数
	 *
	 * @param code 错误码
	 * @param info 返回业务数据
	 */
	public SdkResult(int code, Object info) {
		this(code, info, null);
	}

	/**
	 * 构造函数
	 *
	 * @param code 执行结果代码
	 * @param info 返回业务数据
	 * @param error 业务异常枚举
	 */
	public SdkResult(int code, Object info, ErrorEnum error) {
		this.code = code;
		this.info = info;
		this.error = error;
	}

	public int getCode() {
		return code;
	}

	public void setCode(int code) {
		this.code = code;
	}

	public Object getInfo() {
		return info;
	}

	public void setInfo(Object info) {
		this.info = info;
	}

	public ErrorBean getErrorBean() {
		return errorBean;
	}

	public void setErrorBean(ErrorBean errorBean) {
		this.code = errorBean.getErrCode();
		this.errorBean = errorBean;
	}

	public ErrorEnum getError() {
		return error;
	}

	public void setError(ErrorEnum error) {
		this.code = error.code;
		this.error = error;
		ErrorBean errorBean = new ErrorBean(error.code, error.errMsg);
		this.errorBean = errorBean;
	}

	/**
	 * 是否执行成功
	 *
	 * @return
	 */
	public boolean isSuccess() {
		return code == SUCCESS;
	}

	/**
	 * 构造成功返回结果
	 *
	 * @param info 业务数据
	 * @return 结果对象
	 */
	public static SdkResult success(Object info) {
		return new SdkResult(SUCCESS, info);
	}


	/**
	 * 构造函数 失败
	 *
	 * @param errorEnum 执行结果代码 枚举
	 */
	public static SdkResult failure(ErrorEnum errorEnum) {
		return failure(errorEnum.code, errorEnum.errMsg);
	}

	/**
	 * 构造函数 失败
	 *
	 * @param errorCode 业务异常枚举
	 * @param errorMsg  错误描述英文
	 */
	public static SdkResult failure(Integer errorCode, String errorMsg) {
		ErrorBean errorBean = new ErrorBean(errorCode, errorMsg);
		SdkResult result = new SdkResult();
		result.setErrorBean(errorBean);
		return result;
	}

	public static void main(String[] args) {
		System.out.println(SdkResult.failure(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION));
		SdkResult sdkResult = new SdkResult();
		sdkResult.setError(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
		System.out.println(sdkResult);
		SdkResult sdkResult1 = new SdkResult();
		sdkResult.setError(ErrorEnum.CONNECT_CA_OPEN_API_REFUSED);
		sdkResult1.setErrorBean(sdkResult.getErrorBean());
		System.out.println(sdkResult1);
		System.out.println(SdkResult.success());
		System.out.println(SdkResult.success("1111"));
	}
	/**
	 * 构造成功无业务数据返回结果
	 *
	 * @return 结果对象
	 */
	public static SdkResult success() {
		return new SdkResult(SUCCESS, null);
	}

	/**
	 * 构造失败返回结果
	 *
	 * @param code 错误码
	 * @return 结果对象
	 */
	public static SdkResult failure(int code) {
		return new SdkResult(code, null);
	}


	/**
	 * 构造失败返回结果(兼容Service层返回业务异常枚举)
	 *
	 * @param info 业务数据
	 * @param error 异常枚举对象
	 * @return
	 */
	public static SdkResult failure(Object info, ErrorEnum error) {
		return new SdkResult(FAILURE, info, error);
	}



	@Override
	public String toString() {
		return "SdkResult{" +
				"code=" + code +
				", info=" + info +
				", errorBean=" + errorBean +
				'}';
	}
}
