package com.xdja.ca.utils;

import com.xdja.ca.error.ErrorBean;
import com.xdja.ca.error.ErrorEnum;
import com.xdja.ca.sdk.SdkResult;
import com.xdja.pki.apache.client.core.ApacheHttpException;
import com.xdja.pki.apache.client.core.ClientErrorBean;
import com.xdja.pki.apache.client.core.ClientErrorEnum;
import com.xdja.pki.apache.client.result.AdaptClientResult;
import com.xdja.pki.apache.client.result.ClientResult;
import com.xdja.pki.apache.client.result.RAClientResult;
import com.xdja.pki.apache.client.utils.ApacheClientHttpUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.util.EntityUtils;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

import static com.xdja.ca.error.ErrorEnum.getErrorEnumByCode;

public class ClientHttpUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(ClientHttpUtils.class);

    private static ErrorEnum getErrorEnum(ErrorBean errorBean) {
        String errCode = String.valueOf(errorBean.getErrCode());
        ErrorEnum errorEnumByCode = getErrorEnumByCode(errCode);
        if (null != errorEnumByCode) {
            return errorEnumByCode;
        } else if ("10001".equals(errCode)) {
            return ErrorEnum.CA_ILLEGAL_REQUEST_PARAMETER;
        } else if ("10002".equals(errCode)) {
            return ErrorEnum.CA_MISSING_REQUIRED_PARAMETERS;
        } else {
            logger.info("=================CA的open-api内部异常==================");
            return ErrorEnum.CA_OPEN_API_INNER_EXCEPTION;
        }
    }

    /**
     * 通过POST的方式发送
     *
     * @param reqBody
     * @param url
     * @param contentType
     * @param raSignSn
     * @return
     * @throws IOException
     */
    public static SdkResult sendApacheClientRequest(int keyIndex, String pwd, byte[] privateKey, byte[] reqBody, Map<String, String> paramsMap, String url, String contentType, String raSignSn, X509Certificate[] caCerts, String protectionAlgName, boolean isHttps, boolean isUseHsm, String requestMethod) throws ApacheHttpException {
        SdkResult sdkResult = new SdkResult();
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
        //  isUseHsm = protectionAlgName.equalsIgnoreCase("SM3withSM2");
        logger.debug("===============================通过密码机配置和算法确定通道是否使用密码机:{}", isUseHsm);
        if (isUseHsm) {
            if (keyIndex >= 32 || keyIndex <= 0) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_INDEX_IS_ERROR);
                return sdkResult;
            }

            if (StringUtils.isBlank(pwd)) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_PWD_IS_EMPTY);
                return sdkResult;
            }
        } else {
            if (privateKey == null) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_PRIVATE_KEY_IS_EMPTY);
                return sdkResult;
            }
        }

        if (StringUtils.isBlank(raSignSn)) {
            sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_SIGN_SN_IS_EMPTY);
            return sdkResult;
        }

        String time = String.valueOf(System.currentTimeMillis());
        String signData = null;
        String base64Data = null;
        try {
            if ("post".equalsIgnoreCase(requestMethod)) {
                base64Data = OpenApiReqSignDataUtil.buildPostReqSignData(raSignSn, time, reqBody);
            } else if ("get".equalsIgnoreCase(requestMethod)) {
                String[] split = url.split("/");
                StringBuffer sb = new StringBuffer();
                for (int i = 1; i < split.length; i++) {
                    sb.append("/").append(split[i]);
                }
                String uri = sb.toString();
                base64Data = OpenApiReqSignDataUtil.buildGetReqSignData(raSignSn, time, uri);
            }else {
                logger.info("暂不支持该请求方式 ========== :{}", requestMethod);
                sdkResult.setError(ErrorEnum.REQUEST_METHOD_IS_NOT_SUPPORT);
                return sdkResult;
            }
            logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：{}", protectionAlgName);
            if (isUseHsm) {
                signData = SdkHsmUtils.signByYunHsm(protectionAlgName, keyIndex, pwd, base64Data);
            } else {
                signData = SdkHsmUtils.signByBC(protectionAlgName, privateKey, base64Data);
            }
            if (StringUtils.isBlank(signData)) {
                sdkResult.setError(ErrorEnum.SIGN_BY_HSM_IS_ERROR);
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info("请求 ========== 获取签名值异常", e);
            sdkResult.setError(ErrorEnum.GET_SIGN_BY_YUN_HSM_EXCEPTION);
            return sdkResult;
        }
        Map<String,String> headerMap = new HashMap<>();
        headerMap.put("sn", raSignSn);
        headerMap.put("time", time);
        headerMap.put("sign", signData);
        CloseableHttpResponse closeableHttpResponse = ApacheClientHttpUtils.sendApacheClientRequest(reqBody, paramsMap, headerMap, url, contentType, protectionAlgName, isHttps, requestMethod, isUseHsm,null);
        ClientResult clientResultFromClientResponse = null;
        try {
            clientResultFromClientResponse = AdaptClientResult.getResultFromClientResponse(closeableHttpResponse);
        } catch (IOException e) {
            logger.error("请求返回数据解析异常",e);
            sdkResult.setErrorBean(new ErrorBean(ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.code,ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.desc));
            return sdkResult;
        }
        if (!clientResultFromClientResponse.isSuccess()) {
            sdkResult.setErrorBean(new ErrorBean(clientResultFromClientResponse.getClientErrorBean().getErrCode(), clientResultFromClientResponse.getClientErrorBean().getErrMsg()));
        }
        sdkResult.setInfo(clientResultFromClientResponse.getInfo());
//       sdkResult = ApacheClientHttpUtils.exeHttpsRequest(url, reqBody, paramsMap,contentType, raSignSn, time, signData,protectionAlgName,isHttps,requestMethod);
        return sdkResult;
    }

    /**
     * 该接口只针对CA测试初始化流程中测试连通性时，没有RA的签名sn使用
     * CA测对测试连通性接口不进行验签拦截
     *
     * @param paramsMap
     * @param url
     * @param isHttps
     * @param protectionAlgName
     * @return
     */
    public static SdkResult sendApacheClientRequestInit( byte[] reqBody,Map<String, String> paramsMap, String url, String contentType, String protectionAlgName, boolean isHttps,String requestMethod,boolean isUseHsm) throws ApacheHttpException {
        SdkResult sdkResult = new SdkResult();
//        SdkResult sdkResult = ClientHttpUtils.exeHttpsRequest(url, reqBody, paramsMap,contentType, null, null, null,protectionAlgName,isHttps,requestMethod);
        CloseableHttpResponse closeableHttpResponse = ApacheClientHttpUtils.sendApacheClientRequest(reqBody, paramsMap, null, url, contentType, protectionAlgName, isHttps, requestMethod, isUseHsm,null);
        ClientResult resultFromClientResponse = null;
        try {
            resultFromClientResponse = AdaptClientResult.getResultFromClientResponse(closeableHttpResponse);
           if (resultFromClientResponse.isSuccess()){
               sdkResult.setInfo(resultFromClientResponse.getInfo());
               return sdkResult;
           }
        } catch (IOException e) {
            logger.error("请求返回数据解析异常",e);
            sdkResult.setErrorBean(new ErrorBean(ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.code,ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.desc));
            return sdkResult;
        }
        ClientErrorBean clientErrorBean = resultFromClientResponse.getClientErrorBean();
        sdkResult.setError(getErrorEnum(new ErrorBean(clientErrorBean.getErrCode(),clientErrorBean.getErrMsg())));
        return sdkResult;
    }

    /**
     * 将Apache的client返回的Respond转换为对应的result对象
     * @param response
     * @return
     * @throws IOException
     */
    private static SdkResult getResultFromClientResponse(CloseableHttpResponse response) throws IOException {
        SdkResult sdkResult = new SdkResult();
        byte[] body = null;
        // 获取响应状态
        int statusCode = response.getStatusLine().getStatusCode();
        logger.info("========== 执行调用远程状态码【" + statusCode + "】");

        // 获取结果实体
        org.apache.http.HttpEntity entity = response.getEntity();
        // 按指定编码转换结果实体为String类型
        if (entity != null) {
            body = EntityUtils.toByteArray(entity);
        }
        if (statusCode != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(body), ErrorBean.class);
                logger.info("POST请求========= CA的外部接口返回非200状态信息:" + statusCode + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                sdkResult.setError(getErrorEnum(errorBean));
                return sdkResult;
            } catch (Exception e) {
                logger.error(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        if (null != body && body.length != 0) {
            try {
                byte[] resp = Base64.decode(body);
                sdkResult.setInfo(resp);
            } catch (Exception e) {
                sdkResult.setInfo(body);
            }
        } else {
            sdkResult.setInfo(null);
        }

        return sdkResult;
    }


//    /**
//     * 使用apache发送https请求
//     *
//     * @param url
//     * @param message
//     * @param contentType
//     * @param raSignSn
//     * @param time
//     * @param signData
//     * @return
//     */
//    private static SdkResult exeHttpsRequest(String url, byte[] message,Map<String, String> paramsMap,  String contentType, String raSignSn, String time, String signData,String signAlgName,boolean isHttps,String requestMethod) {
//        SdkResult sdkResult = new SdkResult();
//        byte[] body = null;
//        // 获取client实例
//        ClientHttpUtils.getClient(isHttps,signAlgName);
//        try {
//            // 指定报文头Content-type、User-Agent
//            URIBuilder uri = null;
//            if (isHttps){
//                uri = new URIBuilder("https://" + url);
//            }else {
//                uri = new URIBuilder("http://" + url);
//            }
//            CloseableHttpResponse response = null;
//            HttpUriRequest httpUriRequest = null;
//            ByteArrayEntity bodyEntity = null;
//
//            if ("post".equalsIgnoreCase(requestMethod)){
//                httpUriRequest = new HttpPost(uri.build());
//                bodyEntity = new ByteArrayEntity(Base64.encode(message));
//                ((HttpPost) httpUriRequest).setEntity(bodyEntity);
//            }
//            //get请求带参数
//            else if ("get".equalsIgnoreCase(requestMethod)){
//                if (null != paramsMap && paramsMap.size() != 0) {
//                    List<NameValuePair> list = new LinkedList<NameValuePair>();
//                    for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
//                        BasicNameValuePair param = new BasicNameValuePair(entry.getKey(), entry.getValue());
//                        list.add(param);
//                    }
//                    uri.setParameters(list);
//                }
//                httpUriRequest = new HttpGet(uri.build());
//                ((HttpGet) httpUriRequest).setConfig(requestConfig);
//            }
//            if (StringUtils.isNotBlank(contentType)){
//                httpUriRequest.setHeader("Content-type", contentType);
//            }
//            if (StringUtils.isNotBlank(raSignSn) && StringUtils.isNotBlank(time) && StringUtils.isNotBlank(signData)) {
//                httpUriRequest.setHeader("sn", raSignSn);
//                httpUriRequest.setHeader("time", time);
//                httpUriRequest.setHeader("sign", signData);
//            }
//            logger.info("uri ============  " + uri);
//            // 执行请求操作，并拿到结果（同步阻塞）
//            try {
//                response = client.execute(httpUriRequest,context);
//            } catch (Exception e) {
//                logger.error("请求 ========== Apache安全通道工具类异常{}", e);
//                sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
//                return sdkResult;
//            }
//            // 获取响应状态
//            int statusCode = response.getStatusLine().getStatusCode();
//            logger.info("========== 执行调用远程状态码【" + statusCode + "】");
//
//            // 获取结果实体
//            org.apache.http.HttpEntity entity = response.getEntity();
//            // 按指定编码转换结果实体为String类型
//            if (entity != null) {
//                body = EntityUtils.toByteArray(entity);
//            }
//            if (statusCode != 200) {
//                try {
//                    ErrorBean errorBean = SdkJsonUtils.json2Object(new String(body), ErrorBean.class);
//                    logger.info("POST请求========= CA的外部接口返回非200状态信息:" + statusCode + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
//                    sdkResult.setError(getErrorEnum(errorBean));
//                    return sdkResult;
//                } catch (Exception e) {
//                    logger.error(" ========== 异常信息:{}", e);
//                    sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
//                    return sdkResult;
//                }
//            }
//            if (body.length!=0){
//                try {
//                    byte[] resp = Base64.decode(body);
//                    sdkResult.setInfo(resp);
//                } catch (Exception e) {
//                    sdkResult.setInfo(body);
//                }
//            }else {
//                sdkResult.setInfo(null);
//            }
//            // 释放链接
//            EntityUtils.consume(entity);
//            response.close();
//        } catch (Exception ex) {
//            logger.error(" ========== 异常信息:{}", ex);
//            sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
//            return sdkResult;
//        }
//        return sdkResult;
//    }
//
//
//    private static SSLContext createIgnoreVerifySSL(String signAlgName) throws KeyManagementException, GMSSLContext.GMSSLException {
//        SSLContext sc = null;
//        logger.info("=============================signAlgName===================="+signAlgName);
//        if (StringUtils.isBlank(signAlgName)){
//            return sc;
//        }
//
//        if("SM3withSM2".equalsIgnoreCase(signAlgName)){
//            sc = GMSSLContext.getClientInstance("GMSSLv1.1").getSslContext();
//        }else{
//            sc = GMSSLContext.getClientInstance("TLSv1.2").getSslContext();
//        }
//
//        // 实现一个X509TrustManager接口，用于绕过验证，不用修改里面的方法
//        X509TrustManager trustManager = new X509TrustManager() {
//            @Override
//            public void checkClientTrusted(X509Certificate[] paramArrayOfX509Certificate,
//                                           String paramString) throws CertificateException {
//            }
//
//            @Override
//            public void checkServerTrusted(X509Certificate[] paramArrayOfX509Certificate,
//                                           String paramString) throws CertificateException {
//            }
//
//            @Override
//            public X509Certificate[] getAcceptedIssuers() {
//                return null;
//            }
//        };
//
//        sc.init(null, new TrustManager[]{trustManager}, null);
//        return sc;
//    }
//
//    /**
//     * 获取client
//     *
//     * @return
//     */
//    private static CloseableHttpClient getClient(boolean isHttps, String signAlgName) {
//        Registry<ConnectionSocketFactory> socketFactoryRegistry = null;
//        if (client == null) {
//            if (isHttps) {
//                SSLContext sslcontext = null;
//                // 采用绕过验证的方式处理https请求
//                try {
//                    sslcontext = createIgnoreVerifySSL(signAlgName);
//                } catch (Exception e) {
//                    logger.error("=====================", e);
//                }
//                // 设置协议http和https对应的处理socket链接工厂的对象
//                socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
//                        .register("https", new SSLConnectionSocketFactory(sslcontext,
//                                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER)).build();
//            } else {
//                // 设置协议http和https对应的处理socket链接工厂的对象
//                socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
//                        .register("http", PlainConnectionSocketFactory.INSTANCE).build();
//            }
//            PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
//            connManager.setMaxTotal(500);
//            connManager.setDefaultMaxPerRoute(500);
////            // 检查连接的状态，由客户端通过配置去主动关闭其认为是失效的链接
////            IdleConnectionMonitorThread idleConnectionMonitor = new IdleConnectionMonitorThread(connManager);
////            idleConnectionMonitor.start();
//
//            // 指定链接 keep alive策略 告诉httpclient 链接大概什么时候过期 可以关闭
//            ConnectionKeepAliveStrategy myStrategy = new DefaultConnectionKeepAliveStrategy() {
//                public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
//                    long keepAlive = super.getKeepAliveDuration(response, context);
//                    if (keepAlive == -1) {
//                        keepAlive = 60000;
//                    }
//                    return keepAlive;
//                }
//            };
//
//            // 创建自定义的httpclient对象
//            client = HttpClients.custom().setKeepAliveStrategy(myStrategy).setConnectionManager(connManager).build();
//            // ConnectionRequestTimeout 使用连接池时，从连接池获取连接的超时时间
//            // ConnectTimeout 建立连接时间，三次握手完成时间
//            // SocketTimeout 数据传输过程中数据包之间间隔的最大时间
//            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(6000).setSocketTimeout(20000).setConnectTimeout(6000).build();
//            context = HttpClientContext.create();
//        }
//        return client;
//    }

}
