package com.xdja.ca.utils;

import com.xdja.ca.error.ErrorBean;
import com.xdja.ca.error.ErrorEnum;
import com.xdja.ca.sdk.SdkResult;
import com.xdja.pki.gmssl.http.GMSSLHttpClient;
import com.xdja.pki.gmssl.http.GMSSLHttpsClient;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpRequest;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpResponse;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpException;
import org.apache.commons.lang3.StringUtils;

import org.apache.http.impl.client.CloseableHttpClient;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import java.io.IOException;

import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

import static com.xdja.ca.error.ErrorEnum.getErrorEnumByCode;

public class GMSSLHttpReqUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(GMSSLHttpReqUtils.class);

    private static CloseableHttpClient client = null;

    /**
     * 通过POST的方式发送
     *
     * @param reqBody
     * @param url
     * @param contentType
     * @param raSignSn
     * @return
     * @throws IOException
     */
    public static SdkResult sendGMSSLHttpPostReturnByte(int keyIndex, String pwd, byte[] privateKey, byte[] reqBody, String url, String contentType, String raSignSn, X509Certificate[] caCerts, String protectionAlgName, boolean isHttps, boolean isUseHsm) {
        SdkResult sdkResult = new SdkResult();
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
      //  isUseHsm = protectionAlgName.equalsIgnoreCase("SM3withSM2");
        if (isUseHsm) {
            if (keyIndex >= 32 || keyIndex <= 0) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_INDEX_IS_ERROR);
                return sdkResult;
            }

            if (StringUtils.isBlank(pwd)) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_PWD_IS_EMPTY);
                return sdkResult;
            }
        } else {
            if (privateKey == null) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_PRIVATE_KEY_IS_EMPTY);
                return sdkResult;
            }
        }

        if (StringUtils.isBlank(raSignSn)) {
            sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_SIGN_SN_IS_EMPTY);
            return sdkResult;
        }

        GMSSLHttpClient httpClient = null;
        String time = String.valueOf(System.currentTimeMillis());
        String signData;
        try {
            String base64Data = OpenApiReqSignDataUtil.buildPostReqSignData(raSignSn, time, Base64.encode(reqBody));
            logger.debug("待签名原文 ========== " + base64Data);

            logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：" + protectionAlgName);
            if (isUseHsm) {
                signData = SdkHsmUtils.signByYunHsm(protectionAlgName, keyIndex, pwd, base64Data);
            } else {
                signData = SdkHsmUtils.signByBC(protectionAlgName, privateKey, base64Data);
            }
            if (StringUtils.isBlank(signData)) {
                sdkResult.setError(ErrorEnum.SIGN_BY_HSM_IS_ERROR);
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info("POST请求 ========== 获取签名值异常", e);
            sdkResult.setError(ErrorEnum.GET_SIGN_BY_YUN_HSM_EXCEPTION);
            return sdkResult;
        }

        if (isHttps) {
            if (caCerts != null && caCerts.length > 0) {
                try {
                    httpClient = new GMSSLHttpsClient(caCerts);
                } catch (GMSSLHttpException e) {
                    logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
                    sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
                    return sdkResult;
                }
            } else {
                httpClient = new GMSSLHttpsClient();
            }
            url = "https://" + url;
        } else {
            httpClient = new GMSSLHttpClient();
            url = "http://" + url;
        }

        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("Content-type", contentType);
        headerMap.put("sn", raSignSn);
        headerMap.put("time", time);
        headerMap.put("sign", signData);
        request.setHeaders(headerMap);

        request.setBody(Base64.encode(reqBody));

        GMSSLHttpResponse response = null;
        try {
            response = httpClient.post(request);
        } catch (Exception e) {
            logger.error("POST请求 ========== 国密安全通道工具类异常{}", e);
            sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
            return sdkResult;
        }

        logger.debug("========== sendGMSSLHttpPost.result>>>>" + response.getStatusCode() + " " + new String(response.getBody()));
        if (response.getStatusCode() != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(response.getBody()), ErrorBean.class);
                logger.info("POST请求========= CA的外部接口返回非200状态信息:" + response.getStatusCode() + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                /**
                 * 签发/撤销证书：非法请求参数 缺少必要参数 内部异常  （业务异常由CMP解析）
                 * 确认/错误消息：非法请求参数 缺少必要参数 内部异常  +  业务异常
                 *
                 */
                sdkResult.setError(getErrorEnum(errorBean));
                return sdkResult;
            } catch (Exception e) {
                logger.info("POST请求========= CA的外部接口返回非200状态码:" + response.getStatusCode());
                logger.info(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        byte[] resp = Base64.decode(response.getBody());
        sdkResult.setInfo(resp);
        return sdkResult;
    }


    /**
     * 通过GET的方式请求
     *
     * @param paramsMap
     * @param url
     * @param raSignSn
     * @param isHttps
     * @return
     * @throws IOException
     */
    public static SdkResult sendGMSSLHttpGetReturnByte(int keyIndex, String pwd, byte[] privateKey, Map<String, String> paramsMap, String url, String raSignSn, X509Certificate[] caCerts, String protectionAlgName, boolean isHttps, boolean isUseHsm) {
        SdkResult sdkResult = new SdkResult();
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
     //   isUseHsm = protectionAlgName.equalsIgnoreCase("SM3withSM2");
        if (isUseHsm) {
            if (keyIndex >= 32 || keyIndex <= 0) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_INDEX_IS_ERROR);
                return sdkResult;
            }

            if (StringUtils.isBlank(pwd)) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_PWD_IS_EMPTY);
                return sdkResult;
            }
        } else {
            if (privateKey == null) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_PRIVATE_KEY_IS_EMPTY);
                return sdkResult;
            }
        }

        if (StringUtils.isBlank(raSignSn)) {
            sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_SIGN_SN_IS_EMPTY);
            return sdkResult;
        }

        GMSSLHttpClient httpClient = null;
        String time = String.valueOf(System.currentTimeMillis());
        String signData;
        try {
            String[] split = url.split("/");
            StringBuffer sb = new StringBuffer();
            for (int i = 1; i < split.length; i++) {
                sb.append("/").append(split[i]);
            }
            String uri = sb.toString();
            logger.debug("sendGMSSLHttpGetReturnByte.uri=========" + uri);

            String base64Data = OpenApiReqSignDataUtil.buildGetReqSignData(raSignSn, time, uri);
            logger.debug("待签名原文 ========== " + base64Data);

            logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：" + protectionAlgName);
            if (isUseHsm) {
                signData = SdkHsmUtils.signByYunHsm(protectionAlgName, keyIndex, pwd, base64Data);
            } else {
                signData = SdkHsmUtils.signByBC(protectionAlgName, privateKey, base64Data);
            }
            if (StringUtils.isBlank(signData)) {
                sdkResult.setError(ErrorEnum.SIGN_BY_HSM_IS_ERROR);
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info("GET请求 ========== 获取签名值异常", e);
            sdkResult.setError(ErrorEnum.GET_SIGN_BY_YUN_HSM_EXCEPTION);
            return sdkResult;
        }

        if (isHttps) {
            if (caCerts != null && caCerts.length > 0) {
                try {
                    httpClient = new GMSSLHttpsClient(caCerts);
                } catch (GMSSLHttpException e) {
                    logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
                    sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
                    return sdkResult;
                }
            } else {
                httpClient = new GMSSLHttpsClient();
            }
            url = "https://" + url;
        } else {
            httpClient = new GMSSLHttpClient();
            url = "http://" + url;
        }

        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        request.setParams(paramsMap);
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("sn", raSignSn);
        headerMap.put("time", time);
        headerMap.put("sign", signData);
        request.setHeaders(headerMap);
        GMSSLHttpResponse response = null;
        try {
            response = httpClient.get(request);
        } catch (Exception e) {
            logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
            sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
            return sdkResult;
        }

        logger.debug("=============== sendGMSSLHttpGet.result>>>>" + response.getStatusCode() + " " + new String(response.getBody()));
        if (response.getStatusCode() != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(response.getBody()), ErrorBean.class);
                logger.info("GET请求=========CA的外部接口返回非200状态信息:" + response.getStatusCode() + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                sdkResult.setError(getErrorEnum(errorBean));
                return sdkResult;
            } catch (Exception e) {
                logger.info("GET请求=========CA的外部接口返回非200状态码:" + response.getStatusCode());
                logger.info(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        byte[] resp = Base64.decode(response.getBody());
        sdkResult.setInfo(resp);
        return sdkResult;
    }

    private static ErrorEnum getErrorEnum(ErrorBean errorBean) {
        String errCode = String.valueOf(errorBean.getErrCode());
        ErrorEnum errorEnumByCode = getErrorEnumByCode(errCode);
        if (null != errorEnumByCode) {
            return errorEnumByCode;
        } else if ("10001".equals(errCode)) {
            return ErrorEnum.CA_ILLEGAL_REQUEST_PARAMETER;
        } else if ("10002".equals(errCode)) {
            return ErrorEnum.CA_MISSING_REQUIRED_PARAMETERS;
        } else {
            logger.info("=================CA的open-api内部异常==================");
            return ErrorEnum.CA_OPEN_API_INNER_EXCEPTION;
        }
    }

    /**
     * 通过POST的方式发送
     *
     * @param reqBody
     * @param url
     * @param contentType
     * @param raSignSn
     * @return
     * @throws IOException
     */
    public static SdkResult sendGMSSLHttpPostReturnString(int keyIndex, String pwd, byte[] privateKey, byte[] reqBody, String url, String contentType, String raSignSn, X509Certificate[] caCerts, String protectionAlgName, boolean isHttps, boolean isUseHsm) {
        SdkResult sdkResult = new SdkResult();
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
      //  isUseHsm = protectionAlgName.equalsIgnoreCase("SM3withSM2");
        if (isUseHsm) {
            if (keyIndex >= 32 || keyIndex <= 0) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_INDEX_IS_ERROR);
                return sdkResult;
            }

            if (StringUtils.isBlank(pwd)) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_PWD_IS_EMPTY);
                return sdkResult;
            }
        } else {
            if (privateKey == null) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_PRIVATE_KEY_IS_EMPTY);
                return sdkResult;
            }
        }

        if (StringUtils.isBlank(raSignSn)) {
            sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_SIGN_SN_IS_EMPTY);
            return sdkResult;
        }

        GMSSLHttpClient httpClient = null;
        String time = String.valueOf(System.currentTimeMillis());
        String signData;

        try {
            String base64Data = OpenApiReqSignDataUtil.buildPostReqSignData(raSignSn, time, Base64.encode(reqBody));
            logger.debug("待签名原文 ========== " + base64Data);

            logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：" + protectionAlgName);
            if (isUseHsm) {
                signData = SdkHsmUtils.signByYunHsm(protectionAlgName, keyIndex, pwd, base64Data);
            } else {
                signData = SdkHsmUtils.signByBC(protectionAlgName, privateKey, base64Data);
            }
            if (StringUtils.isBlank(signData)) {
                sdkResult.setError(ErrorEnum.SIGN_BY_HSM_IS_ERROR);
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info("POST请求 ========== 获取签名值异常", e);
            sdkResult.setError(ErrorEnum.GET_SIGN_BY_YUN_HSM_EXCEPTION);
            return sdkResult;
        }

        if (isHttps) {
            if (caCerts != null && caCerts.length > 0) {
                try {
                    httpClient = new GMSSLHttpsClient(caCerts);
                } catch (GMSSLHttpException e) {
                    logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
                    sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
                    return sdkResult;
                }
            } else {
                httpClient = new GMSSLHttpsClient();
            }
            url = "https://" + url;
        } else {
            httpClient = new GMSSLHttpClient();
            url = "http://" + url;
        }

        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("Content-type", contentType);
        headerMap.put("sn", raSignSn);
        headerMap.put("time", time);
        headerMap.put("sign", signData);
        request.setHeaders(headerMap);

        request.setBody(Base64.encode(reqBody));

        GMSSLHttpResponse response = null;
        logger.info("请求地址 ========== " + url);
        try {
            response = httpClient.post(request);

        } catch (Exception e) {
            logger.error("POST请求 ========== 国密安全通道工具类异常{}", e);
            sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
            return sdkResult;
        }

        logger.debug("========== sendGMSSLHttpPost.result>>>>" + response.getStatusCode() + " " + new String(response.getBody()));
        if (response.getStatusCode() != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(response.getBody()), ErrorBean.class);
                logger.info("POST请求========= CA的外部接口返回非200状态信息:" + response.getStatusCode() + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                sdkResult.setError(getErrorEnum(errorBean));
                return sdkResult;
            } catch (Exception e) {
                logger.info("POST请求========= CA的外部接口返回非200状态码:" + response.getStatusCode());
                logger.error(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        String resp = new String(response.getBody());
        sdkResult.setInfo(resp);
        return sdkResult;
    }

    /**
     * 通过GET的方式请求
     *
     * @param paramsMap
     * @param url
     * @param raSignSn
     * @param isHttps
     * @return
     * @throws IOException
     */
    public static SdkResult sendGMSSLHttpGetReturnString(int keyIndex, String pwd, byte[] privateKey, Map<String, String> paramsMap, String url, String raSignSn, X509Certificate[] caCerts, String protectionAlgName, boolean isHttps, boolean isUseHsm) {
        SdkResult sdkResult = new SdkResult();
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
      //  isUseHsm = protectionAlgName.equalsIgnoreCase("SM3withSM2");
        if (isUseHsm) {
            if (keyIndex >= 32 || keyIndex <= 0) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_INDEX_IS_ERROR);
                return sdkResult;
            }

            if (StringUtils.isBlank(pwd)) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_KEY_PWD_IS_EMPTY);
                return sdkResult;
            }
        } else {
            if (privateKey == null) {
                sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_PRIVATE_KEY_IS_EMPTY);
                return sdkResult;
            }
        }

        if (StringUtils.isBlank(raSignSn)) {
            sdkResult.setError(ErrorEnum.CMP_API_PARAMS_RA_SIGN_SN_IS_EMPTY);
            return sdkResult;
        }

        GMSSLHttpClient httpClient = null;
        String signData;
        String time = String.valueOf(System.currentTimeMillis());
        try {
            String[] split = url.split("/");
            StringBuffer sb = new StringBuffer();
            for (int i = 1; i < split.length; i++) {
                sb.append("/").append(split[i]);
            }
            String uri = sb.toString();

            logger.debug("sendGMSSLHttpGetReturnByte.uri=========" + uri);

            String base64Data = OpenApiReqSignDataUtil.buildGetReqSignData(raSignSn, time, uri);
            logger.debug("待签名原文 ========== " + base64Data);

            logger.debug(" ================ RA封装消息使用的服务器证书的签名算法为：" + protectionAlgName);
            if (isUseHsm) {
                signData = SdkHsmUtils.signByYunHsm(protectionAlgName, keyIndex, pwd, base64Data);
            } else {
                signData = SdkHsmUtils.signByBC(protectionAlgName, privateKey, base64Data);
            }
            if (StringUtils.isBlank(signData)) {
                sdkResult.setError(ErrorEnum.SIGN_BY_HSM_IS_ERROR);
                return sdkResult;
            }
        } catch (Exception e) {
            logger.info("GET请求 ========== 获取签名值异常", e);
            sdkResult.setError(ErrorEnum.GET_SIGN_BY_YUN_HSM_EXCEPTION);
            return sdkResult;
        }

        if (isHttps) {
            if (caCerts != null && caCerts.length > 0) {
                try {
                    httpClient = new GMSSLHttpsClient(caCerts);
                } catch (GMSSLHttpException e) {
                    logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
                    sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
                    return sdkResult;
                }
            } else {
                httpClient = new GMSSLHttpsClient();
            }
            url = "https://" + url;
        } else {
            httpClient = new GMSSLHttpClient();
            url = "http://" + url;
        }

        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        request.setParams(paramsMap);
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("sn", raSignSn);
        headerMap.put("time", time);
        headerMap.put("sign", signData);
        request.setHeaders(headerMap);
        GMSSLHttpResponse response = null;
        try {
            response = httpClient.get(request);
        } catch (Exception e) {
            logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
            sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
            return sdkResult;
        }

        logger.debug("=============== sendGMSSLHttpGet.result>>>>" + response.getStatusCode() + " " + new String(response.getBody()));
        if (response.getStatusCode() != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(response.getBody()), ErrorBean.class);
                logger.info("GET请求=========CA的外部接口返回非200状态信息:" + response.getStatusCode() + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                sdkResult.setError(getErrorEnum(errorBean));
                return sdkResult;
            } catch (Exception e) {
                logger.info("GET请求=========CA的外部接口返回非200状态码:" + response.getStatusCode());
//                logger.info(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        String resp = new String(response.getBody());
        sdkResult.setInfo(resp);
        return sdkResult;
    }

    /**
     * 该接口只针对CA测试初始化流程中测试连通性时，没有RA的签名sn使用
     * CA测对测试连通性接口不进行验签拦截
     *
     * @param paramsMap
     * @param url
     * @param isHttps
     * @param caCerts
     * @return
     */
    public static SdkResult sendGMSSLHttpGetReturnByteInit(Map<String, String> paramsMap, String url, X509Certificate[] caCerts, boolean isHttps) {
        SdkResult sdkResult = new SdkResult();

        GMSSLHttpClient httpClient = null;
        String time = String.valueOf(System.currentTimeMillis());
        if (isHttps) {
            if (caCerts != null && caCerts.length > 0) {
                try {
                    httpClient = new GMSSLHttpsClient(caCerts);
                } catch (GMSSLHttpException e) {
                    logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
                    sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
                    return sdkResult;
                }
            } else {
                httpClient = new GMSSLHttpsClient();
            }
            url = "https://" + url;
        } else {
            httpClient = new GMSSLHttpClient();
            url = "http://" + url;
        }

        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        request.setParams(paramsMap);
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("sn", null);
        headerMap.put("time", time);
        headerMap.put("sign", null);
        request.setHeaders(headerMap);
        GMSSLHttpResponse response = null;
        try {
            response = httpClient.get(request);
        } catch (Exception e) {
            logger.error("GET请求 ========== 国密安全通道工具类异常{}", e);
            sdkResult.setError(ErrorEnum.GMSSL_HTTP_CLIENT_INTER_EXCEPTION);
            return sdkResult;
        }

        logger.debug("=============== sendGMSSLHttpGet.result>>>>" + response.getStatusCode() + " " + new String(response.getBody()));
        if (response.getStatusCode() != 200) {
            try {
                ErrorBean errorBean = SdkJsonUtils.json2Object(new String(response.getBody()), ErrorBean.class);
                logger.info("GET请求=========CA的外部接口返回非200状态信息:" + response.getStatusCode() + " ====== errorCode:" + errorBean.getErrCode() + " ====== errorMsg:" + errorBean.getErrMsg());
                sdkResult.setErrorBean(errorBean);
                return sdkResult;
            } catch (Exception e) {
                logger.info("GET请求=========CA的外部接口返回非200状态码:" + response.getStatusCode());
                logger.info(" ========== 异常信息:{}", e);
                sdkResult.setError(ErrorEnum.CA_OPEN_API_RETURN_EXCEPTION);
                return sdkResult;
            }
        }
        byte[] resp = Base64.decode(response.getBody());
        sdkResult.setInfo(resp);
        return sdkResult;
    }

}
