package com.xdja.pki.gmssl.https;

import com.xdja.pki.gmssl.GMSSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import java.io.IOException;
import java.net.URL;
import java.security.KeyStore;

public class GMHttpsClient {

    private SSLSocketFactory socketFactory;
    private static Logger logger = LoggerFactory.getLogger(GMHttpsClient.class.getName());

    public GMHttpsClient(String protocol) throws GMHttpsException {
        try {
            socketFactory = GMSSLContext.getClientInstance(protocol).getSocketFactory();
        } catch (GMSSLContext.GMSSLException e) {
            throw new GMHttpsException("get client instance exception", e);
        }
    }

    public GMHttpsClient(KeyStore keyStore, char[] password, KeyStore trustStore) throws GMHttpsException {
        try {
            socketFactory = GMSSLContext.getClientInstance(keyStore, password, trustStore, Constants.GMSSLv11).getSocketFactory();
        } catch (GMSSLContext.GMSSLException e) {
            throw new GMHttpsException("get client instance exception", e);
        }
    }

    public interface Listener {
        void onResponse(GMHttpsResponse response);

        void onResponseError(String message);
    }

    /**
     * 通过 url 创建 HttpsURLConnection 对象
     *
     * @param url 要访问的地址
     * @return HttpsURLConnection 连接对象
     * @throws GMHttpsException 国密 https 异常
     */
    private HttpsURLConnection getHttpsURLConnection(String url, GMHttpsMethod method, String body) throws GMHttpsException {
        HttpsURLConnection connection = null;
        try {
            connection = (HttpsURLConnection) new URL(url).openConnection();
            connection.setRequestProperty("Content-type", "text/plain");
            connection.setSSLSocketFactory(socketFactory);
            connection.setDoInput(true);
            connection.setConnectTimeout(10000);
            connection.setReadTimeout(10000);
            if (GMHttpsMethod.GET.equals(method)) {
                connection.connect();
            }
            if (GMHttpsMethod.POST.equals(method)) {
                connection.setDoOutput(true);
                GMHttpsUtils.write(connection.getOutputStream(), body);
            }
        } catch (IOException e) {
            throw new GMHttpsException("get client instance exception", e);
        }

        return connection;
    }

    public GMHttpsResponse request(GMHttpsRequest request) throws GMHttpsException {
        logger.info(request.toString());
        try {
            HttpsURLConnection connection = getHttpsURLConnection(request.getUri(), request.getMethod(), request.getBody());
            String response = "";
            if (connection.getResponseCode() == 200) {
                response = GMHttpsUtils.read(connection.getInputStream());
            }
            connection.disconnect();
            return new GMHttpsResponse(connection.getResponseCode(), response);
        } catch (IOException e) {
            throw new GMHttpsException("GMHttpsClient request exception", e);
        }
    }

    public GMHttpsResponse get(String uri) throws GMHttpsException {
        return request(new GMHttpsRequest(null, uri, GMHttpsMethod.GET, null));
    }

    public GMHttpsResponse post(String uri, String body) throws GMHttpsException {
        return request(new GMHttpsRequest(null, uri, GMHttpsMethod.POST, body));
    }

    public void asyncRequest(GMHttpsRequest request, Listener listener) throws GMHttpsException {
        HttpsURLConnection connection = getHttpsURLConnection(request.getUri(), request.getMethod(), request.getBody());
        new Thread(
                () -> {
                    try {
                        if (connection.getResponseCode() == 200) {
                            String response = GMHttpsUtils.read(connection.getInputStream());
                            listener.onResponse(new GMHttpsResponse(connection.getResponseCode(), response));
                        } else {
                            listener.onResponseError("response code: " + connection.getResponseCode());
                        }
                    } catch (IOException e) {
                        listener.onResponseError("read response error: " + e.toString());
                    }
                }
        ).start();
    }

    public void asyncGet(String url, Listener listener) throws GMHttpsException {
        asyncRequest(new GMHttpsRequest(null, url, GMHttpsMethod.GET, null), listener);
    }

    public void asyncPost(String url, String body, Listener listener) throws GMHttpsException {
        asyncRequest(new GMHttpsRequest(null, url, GMHttpsMethod.POST, body), listener);
    }

}
