package com.xdja.pki.gmssl.main.tomcat.tools;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.utils.*;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.sdf.yunhsm.utils.GMSSLYunHsmUtils;
import com.xdja.pki.gmssl.tomcat.utils.GMSSLTomcatUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLP10Utils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.apache.commons.io.FileUtils;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.cms.EncryptedContentInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.util.Store;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

public class ResolveCertUtils {
    private static Logger logger = LoggerFactory.getLogger(ResolveCertUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static void addHttpsPort(String confPath) throws Exception {

        TomcatPropertiesEntry tomcatProperties = null;
        try {
            tomcatProperties = getTomcatProperties(confPath);
        } catch (IOException e) {
            logger.error("获取配置信息失败", e);
            return;
        }
        System.out.println(tomcatProperties.toString());
        if (tomcatProperties.getKeyType().equalsIgnoreCase("RSA")) {
            addRSAHttpsPort(tomcatProperties, confPath);
        } else if (tomcatProperties.getKeyType().equalsIgnoreCase("EC")) {
            addECHttpsPort(tomcatProperties);
        } else if (tomcatProperties.getKeyType().equalsIgnoreCase("BCEC")) {
            addBCECHttpsPort(tomcatProperties);
        } else {
            throw new Exception("暂不支持" + tomcatProperties.getKeyType() + "类型密钥");
        }

    }

    public static void getSignCert(String signPath, String writePath) throws Exception {
        String signCert = GMSSLFileUtils.fileToString(new File(signPath), "utf-8");
        List<X509Certificate> certs = getCertsByCertChain(signCert);
        for (X509Certificate certificate : certs) {
            if (!GMSSLX509Utils.isCACertificate(certificate)) {
                GMSSLX509Utils.writeCertificateToCer(writePath + "/", "sign", certificate);
                GMSSLX509Utils.writeCertificateToPem(writePath + "/", "sign", certificate);
                System.out.println("已将证书写入至" + writePath + "路径下");
            }
        }
    }

    public static void addRSAHttpsPort(TomcatPropertiesEntry tomcatProperties, String confPath) throws Exception {
        ToolsCertBean certBean = getCommonCert(tomcatProperties.getEncCertPath(),
                tomcatProperties.getEncPrivateKey(),
                tomcatProperties.getSignCertPath(), false);
        GMSSLTomcatUtils.openHttpsPortByJKSWithRSA(certBean.getRootCerts(),
                certBean.getSignCert(),
                certBean.getEncCert(),
                tomcatProperties.getSignPrivateKey(),
                tomcatProperties.getEncPrivateKey(),
                tomcatProperties.getTomcatPath(), tomcatProperties.getHttpsPort());
        System.out.println("已配置" + tomcatProperties.getTomcatPath() + "路径下Tomcat的" + tomcatProperties.getHttpsPort()
                + "端口为TLSv1.2类型的Https通道");
    }

    public static void addBCECHttpsPort(TomcatPropertiesEntry tomcatProperties) throws Exception {
        ToolsCertBean certBean = getCommonCert(tomcatProperties.getEncCertPath(),
                tomcatProperties.getEncPrivateKey(),
                tomcatProperties.getSignCertPath(), true);
        GMSSLTomcatUtils.openHttpsPortByBC(certBean.getRootCerts(),
                certBean.getSignCert(),
                certBean.getEncCert(),
                tomcatProperties.getSignPrivateKey(),
                tomcatProperties.getEncPrivateKey(),
                tomcatProperties.getTomcatPath(), tomcatProperties.getHttpsPort());
        System.out.println("已配置" + tomcatProperties.getTomcatPath() + "路径下Tomcat的" + tomcatProperties.getHttpsPort()
                + "端口为GMSSLV1.1类型的Https通道");
    }


    public static void addECHttpsPort(TomcatPropertiesEntry tomcatProperties) throws Exception {
        X509Certificate encCert;
        try {
            byte[] encData = GMSSLFileUtils.readFileToByte(tomcatProperties.getEncCertPath());
            String cert = getEncCertByEnvelopData(tomcatProperties.getEncKeyIndex(),
                    tomcatProperties.getEncPriKey(),
                    encData,
                    true,
                    null
            );
            encCert = GMSSLX509Utils.readCertificateFromCerByte(GMSSLByteArrayUtils.base64Decode(cert));
        } catch (Exception e) {
            logger.error("解析加密证书链失败", e);
            return;
        }
        X509Certificate sign = null;
        List<X509Certificate> rootCerts = new ArrayList<>();
        try {
            String signCert = GMSSLFileUtils.fileToString(new File(tomcatProperties.getSignCertPath()), "utf-8");
            List<X509Certificate> certs = getCertsByCertChain(signCert);
            for (X509Certificate cert : certs) {
                if (cert.getSubjectX500Principal().equals(encCert.getSubjectX500Principal())) {
                    sign = cert;
                } else {
                    rootCerts.add(cert);
                }
            }
        } catch (Exception e) {
            logger.error("解析签名证书链失败", e);
            return;
        }
        GMSSLTomcatUtils.openHttpsPortByYunHsm(rootCerts, sign, encCert, tomcatProperties.getEncKeyIndex(),
                tomcatProperties.getEncPriKey(), tomcatProperties.getTomcatPath(), tomcatProperties.getHttpsPort());
        System.out.println("已配置" + tomcatProperties.getTomcatPath() + "路径下Tomcat" + tomcatProperties.getHttpsPort()
                + "GMSSLSDFYUNHSMV1.1类型的Https通道");
    }

    private static ToolsCertBean getCommonCert(String encCertPath, PrivateKey encPrivate, String signCertPath, boolean isEc) throws Exception {
        ToolsCertBean toolsCertBean = new ToolsCertBean();
        X509Certificate encCert;
        try {
            if (encPrivate instanceof BCECPrivateKey) {
                isEc = true;
            } else {
                isEc = false;
            }
            byte[] encData = GMSSLFileUtils.readFileToByte(encCertPath);
            String cert = getEncCertByEnvelopData(null,
                    null,
                    encData,
                    isEc,
                    encPrivate);
            encCert = GMSSLX509Utils.readCertificateFromCerByte(GMSSLByteArrayUtils.base64Decode(cert));
        } catch (Exception e) {
            logger.error("解析加密证书链失败", e);
            throw new Exception("解析加密证书链失败", e);
        }
        X509Certificate sign = null;
        List<X509Certificate> rootCerts = new ArrayList<>();
        try {
            String signCert = GMSSLFileUtils.fileToString(new File(signCertPath), "utf-8");
            List<X509Certificate> certs = getCertsByCertChain(signCert);
            for (X509Certificate cert : certs) {
                if (cert.getSubjectX500Principal().equals(encCert.getSubjectX500Principal())) {
                    sign = cert;
                } else {
                    rootCerts.add(cert);
                }
            }
        } catch (Exception e) {
            logger.error("解析签名证书链失败", e);
            throw new Exception("解析签名证书链失败", e);
        }
        toolsCertBean.setEncCert(encCert);
        toolsCertBean.setSignCert(sign);
        toolsCertBean.setRootCerts(rootCerts);
        return toolsCertBean;
    }

    public X509Certificate getEncCert(int keyIndex, String priKeyPwd, String path) throws Exception {
        byte[] data = GMSSLFileUtils.readFileToByte(path);
        String cert = getEncCertByEnvelopData(keyIndex, priKeyPwd, data, true, null);
        return GMSSLX509Utils.readCertificateFromCerByte(GMSSLByteArrayUtils.base64Decode(cert));
    }

    public static List<X509Certificate> getCertsByCertChain(String p7b) throws CMSException, CertificateException {
        p7b = p7b.replaceFirst("-----BEGIN PKCS7-----", "");
        p7b = p7b.replaceFirst("-----END PKCS7-----", "");
        List<X509Certificate> certificateList = new ArrayList<>();
        CMSSignedData cmsSignedData = new CMSSignedData(Base64.decode(p7b));
        Store<X509CertificateHolder> store = cmsSignedData.getCertificates();
        Collection collection = store.getMatches(null);
        Iterator<X509CertificateHolder> it = collection.iterator();
        while (it.hasNext()) {
            X509CertificateHolder x509CertificateHolder = it.next();
            X509Certificate x509Certificate = new JcaX509CertificateConverter().setProvider("BC").getCertificate(x509CertificateHolder);
            certificateList.add(x509Certificate);
        }
        return certificateList;
    }


    /**
     * 根据tomcatConf位置获取配置信息
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static TomcatPropertiesEntry getTomcatProperties(String confPath) throws IOException {
        TomcatPropertiesEntry propertiesEntry = new TomcatPropertiesEntry();
        File file = new File(confPath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //tomcatProperties目录下
        JSONObject tomcatProperties = dataJson.getJSONObject("Tomcat");

        propertiesEntry.setTomcatPath(tomcatProperties.getString("tomcatPath"));
        propertiesEntry.setRootCertPath(tomcatProperties.getString("rootCertPath"));
        propertiesEntry.setSignCertPath(tomcatProperties.getString("signCertPath"));
        propertiesEntry.setEncCertPath(tomcatProperties.getString("encCertPath"));
        propertiesEntry.setEncKeyIndex(tomcatProperties.getInteger("encKeyIndex"));
        propertiesEntry.setEncPriKey(tomcatProperties.getString("encPriKey"));
        propertiesEntry.setHttpsPort(tomcatProperties.getInteger("httpsPort"));
        propertiesEntry.setCryptoType(tomcatProperties.getString("cryptoType"));
        propertiesEntry.setKeyType(tomcatProperties.getString("keyType"));
        propertiesEntry.setEncPriKeyPath(tomcatProperties.getString("encPriKeyPath"));
        propertiesEntry.setSignPriKeyPath(tomcatProperties.getString("signPriKeyPath"));
        return propertiesEntry;
    }


    /**
     * 从数字信封中获取加密证书
     *
     * @param
     * @return B64加密证书
     */
    public static String getEncCertByEnvelopData(Integer keyIndex, String priKeyPwd, byte[] data, boolean isEc, PrivateKey privateKey) throws Exception {

        SM2EnvelopedData envelopedData = SM2EnvelopedData.getInstance(Base64.decode(data));
        ASN1Set recipientInfos = envelopedData.getRecipientInfos();
        ASN1Encodable objectAt = recipientInfos.getObjectAt(0);
        ASN1Sequence derSequence = (ASN1Sequence) recipientInfos.getObjectAt(0);
//        DLSequence derSequence = (DLSequence) recipientInfos.getObjectAt(0);
        DEROctetString pwdSymmetricKey = null;//密文的对称密钥
        for (int i = 0; i < derSequence.size(); i++) {
            Object object = null;
            if ((object = derSequence.getObjectAt(i)) instanceof DEROctetString) {
                pwdSymmetricKey = (DEROctetString) object;
            }
        }
        //获取密文的加密证书
        EncryptedContentInfo encryptedContentInfo = envelopedData.getEncryptedContentInfo();
        ASN1OctetString encryptedContent = encryptedContentInfo.getEncryptedContent();
        //获取对称密钥
        String base64Enc = Base64.toBase64String(pwdSymmetricKey.getOctets());
        String symmetricKey;
        if (isEc) {
            if (privateKey != null) {
                symmetricKey = GMSSLSM2EncryptUtils.decryptASN1ByBC(privateKey, base64Enc);
            } else {
                symmetricKey = GMSSLSM2EncryptUtils.decryptASN1ByYunhsm(keyIndex, priKeyPwd, base64Enc);
            }
        } else {
            symmetricKey = GMSSLRSAEncryptUtils.decryptDataPKCS1ByBC(privateKey, base64Enc);
        }

        //得到明文加密证书
        return GMSSLSM4ECBEncryptUtils.decryptByBCWithPKCS7Padding(symmetricKey, Base64.toBase64String(encryptedContent.getOctets()));
    }

    public static void writeKeyJson(String path, String fileName, String name, List<X509Certificate> certs) {
        List<String> content = new ArrayList<>();
        for (int i = 0; i < certs.size(); i++) {
            try {
                content.add(getPemObjectString(certs.get(i)));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        JSONObject result = new JSONObject();
        // 返回一个JSONArray对象
        JSONArray jsonArray = new JSONArray();
        jsonArray.addAll(content);
        result.put(name, jsonArray);
        System.out.println(result.toJSONString());
        createJsonFile(result.toString(), path, fileName);
    }

    public static String getPemObjectString(Object obj) throws Exception {
        StringWriter stringWriter = new StringWriter();
        GMSSLX509Utils.writePEM(obj, stringWriter);
        return stringWriter.toString();
    }

    public static boolean createJsonFile(String jsonString, String filePath, String fileName) {
        // 标记文件生成是否成功
        boolean flag = true;

        // 拼接文件完整路径
        String fullPath = filePath + File.separator + fileName;

        // 生成json格式文件
        try {
            // 保证创建一个新文件
            File file = new File(fullPath);
            if (!file.getParentFile().exists()) { // 如果父目录不存在，创建父目录
                file.getParentFile().mkdirs();
            }
            if (file.exists()) { // 如果已存在,删除旧文件
                file.delete();
            }
            file.createNewFile();

            if (jsonString.indexOf("'") != -1) {
                //将单引号转义一下，因为JSON串中的字符串类型可以单引号引起来的
                jsonString = jsonString.replaceAll("'", "\\'");
            }
            if (jsonString.indexOf("\"") != -1) {
                //将双引号转义一下，因为JSON串中的字符串类型可以单引号引起来的
                jsonString = jsonString.replaceAll("\"", "\\\"");
            }

            if (jsonString.indexOf("\r\n") != -1) {
                //将回车换行转换一下，因为JSON串中字符串不能出现显式的回车换行
                jsonString = jsonString.replaceAll("\r\n", "\\u000d\\u000a");
            }
            if (jsonString.indexOf("\n") != -1) {
                //将换行转换一下，因为JSON串中字符串不能出现显式的换行
                jsonString = jsonString.replaceAll("\n", "\\u000a");
            }

            //  格式化json字符串
            jsonString = GMSSLYunHsmUtils.formatJson(jsonString);

            // 将格式化后的字符串写入文件
            Writer write = new OutputStreamWriter(new FileOutputStream(file), "UTF-8");
            write.write(jsonString);
            write.flush();
            write.close();
        } catch (Exception e) {
            flag = false;
            e.printStackTrace();
        }
        // 返回是否成功的标记
        return flag;
    }

    public static void generateP10(int keySize, String dn, String signAlgName, String path) throws Exception {
        System.out.println("将随机生成" + keySize + "bits的RSA公钥");
        System.out.println("生成的P10 DN为" + dn);
        System.out.println("生成的P10 签名算法为" + signAlgName.toUpperCase());
        KeyPair signKeyPair = GMSSLRSAKeyUtils.generateKeyPairByBC(keySize);
        KeyPair encKeyPair = GMSSLRSAKeyUtils.generateKeyPairByBC(keySize);
        PKCS10CertificationRequest signP10 = GMSSLP10Utils.generateP10SignByBC(dn, signKeyPair.getPublic(), signKeyPair.getPrivate(), signAlgName);
        GMSSLX509Utils.writePublicKeyToDat(path, "server_enc.dat", encKeyPair.getPublic());
        GMSSLP10Utils.writeP10ToFile(path, "server_sign", signP10);
        GMSSLX509Utils.writePrivateKeyToPem(path, "sign_key", signKeyPair.getPrivate());
        GMSSLX509Utils.writePrivateKeyToPem(path, "enc_key", encKeyPair.getPrivate());
        System.out.println("已生成相关文件在" + path + "路径下");
    }

    public static void generateSm2P10(String dn, String path) throws Exception {
        System.out.println("将随机生成SM2公钥");
        System.out.println("生成的P10 DN为" + dn);
        //System.out.println("生成的P10 签名算法为" + signAlgName.toUpperCase());
        KeyPair signKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        KeyPair encKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        PKCS10CertificationRequest signP10 = GMSSLP10Utils.generateP10SignByBC(dn, signKeyPair.getPublic(), signKeyPair.getPrivate(), GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        GMSSLX509Utils.writePublicKeyToDat(path, "server_enc.dat", encKeyPair.getPublic());
        GMSSLP10Utils.writeP10ToFile(path, "server_sign", signP10);
        GMSSLX509Utils.writePrivateKeyToPem(path, "sign_key", signKeyPair.getPrivate());
        GMSSLX509Utils.writePrivateKeyToPem(path, "enc_key", encKeyPair.getPrivate());
        System.out.println("已生成相关文件在" + path + "路径下");
    }


    public static void getPriKeyStore(String confPath) throws Exception {
        KeyStorePropertiesEntry keyStorePropertiesEntry = null;
        try {
            keyStorePropertiesEntry = getKeyStoreEntry(confPath);
        } catch (IOException e) {
            logger.error("获取配置信息失败", e);
            return;
        }
        System.out.println(keyStorePropertiesEntry.toString());
        ToolsCertBean certBean = getCommonCert(
                keyStorePropertiesEntry.getEncCertPath(),
                keyStorePropertiesEntry.getEncPrivateKey(),
                keyStorePropertiesEntry.getSignCertPath(), false
        );
        KeyStore keyStore;
        if (keyStorePropertiesEntry.getEncPrivateKey() instanceof BCECPrivateKey) {
            keyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStoreWithBKS("password",
                    certBean.getRootCerts().get(0),
                    "sign",
                    keyStorePropertiesEntry.getSignPrivateKey(),
                    certBean.getSignCert(),
                    "enc",
                    keyStorePropertiesEntry.getEncPrivateKey(),
                    certBean.getEncCert());
        } else {
            keyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStoreWithJKS("password",
                    certBean.getRootCerts().get(0),
                    "sign",
                    keyStorePropertiesEntry.getSignPrivateKey(),
                    certBean.getSignCert(),
                    "enc",
                    keyStorePropertiesEntry.getEncPrivateKey(),
                    certBean.getEncCert());
        }

        GMSSLKeyStoreUtils.saveGMSSLKeyStore(keyStore, "password", keyStorePropertiesEntry.getWritePath(), "privateKey");
        System.out.println("已生成相关文件privateKey.keystore在" + keyStorePropertiesEntry.getWritePath() + "路径下");
    }

    /**
     * 根据keyStoreConf位置获取配置信息
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static KeyStorePropertiesEntry getKeyStoreEntry(String confPath) throws IOException {
        KeyStorePropertiesEntry propertiesEntry = new KeyStorePropertiesEntry();
        File file = new File(confPath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //tomcatProperties目录下
        JSONObject tomcatProperties = dataJson.getJSONObject("keystore");
        propertiesEntry.setWritePath(tomcatProperties.getString("writePath"));
        propertiesEntry.setSignCertPath(tomcatProperties.getString("signCertPath"));
        propertiesEntry.setEncCertPath(tomcatProperties.getString("encCertPath"));
        propertiesEntry.setEncPriKeyPath(tomcatProperties.getString("encPriKeyPath"));
        propertiesEntry.setSignPriKeyPath(tomcatProperties.getString("signPriKeyPath"));
        return propertiesEntry;
    }


}
