package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifierProvider;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentSigner;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentVerifierProvider;
import com.xdja.pki.gmssl.operator.utils.*;
import com.xdja.pki.gmssl.x509.utils.bean.CRLEntry;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.CRLNumber;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.CertException;
import org.bouncycastle.cert.X509CRLHolder;
import org.bouncycastle.cert.X509v2CRLBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CRLConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

import java.io.IOException;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CRLException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.List;

public class GMSSLCRLUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 签发CRL接口
     *
     * @param issueCert   签发者证书
     * @param issuerKey   签发者私钥信息
     * @param algorithm   签发算法类型
     * @param crlNumber   CRL 序列号
     * @param thisUpdate  这次更新时间
     * @param nextUpdate  下次更新时间
     * @param extensions  扩展项列表
     * @param revoked     CRL实体 列表
     * @param isWithParam 是否需要签名扩展参数 默认填写false，仅支持卡斯柯项目SM2WithSM3签名填写true
     * @return X509CRL
     * @throws IOException
     * @throws OperatorCreationException
     * @throws CRLException
     */
    public static X509CRL generateCRL(
            X509Certificate issueCert, PrivateKey issuerKey, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, OperatorCreationException, CRLException {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivateKey = (SdfPrivateKey) issuerKey;
                return generateCRLByPCIE(issueCert, pciePrivateKey.getIndex(), pciePrivateKey.getStringPassword(), algorithm, crlNumber, thisUpdate, thisUpdate, extensions, revoked, isWithParam);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivateKey = (SdfPrivateKey) issuerKey;
                return generateCRLByYunhsm(issueCert, hsmPrivateKey.getIndex(), hsmPrivateKey.getStringPassword(), algorithm, crlNumber, thisUpdate, thisUpdate, extensions, revoked, isWithParam);
            case MINI_PCI_E:
                return generateCRLByMiniPcie(issueCert, issuerKey, algorithm, crlNumber, thisUpdate, nextUpdate, extensions, revoked, isWithParam);
            case SANC_HSM:
                return generateCRLBySanc(issueCert, issuerKey, algorithm, crlNumber, thisUpdate, nextUpdate, extensions, revoked, isWithParam);
            case BC:
            default:
                ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, issuerKey);
                return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
        }
    }

    /**
     * 验签CRL
     *
     * @param publicKey 验签公钥
     * @param crl       CRL
     * @return 验签结果
     * @throws IOException
     * @throws CRLException
     * @throws CertException
     * @throws OperatorCreationException
     */
    public static boolean verifyCRL(PublicKey publicKey, X509CRL crl) throws IOException, CRLException, CertException, OperatorCreationException {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                ContentVerifierProvider pcieContentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByPcie(publicKey);
                return verifyCRL(pcieContentVerifierProvider, crl.getEncoded());
            case XDJA_HSM:
                ContentVerifierProvider hsmContentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByYunHsm(publicKey);
                return verifyCRL(hsmContentVerifierProvider, crl.getEncoded());
            case MINI_PCI_E:
                return verifyCRLByMiniPcie(publicKey, crl);
            case SANC_HSM:
                return verifyCRLBySanc(publicKey, crl);
            case BC:
            default:
                ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByBC(publicKey);
                return verifyCRL(contentVerifierProvider, crl.getEncoded());
        }
    }


    public static X509CRL generateCRLBySanc(
            X509Certificate issueCert, PrivateKey issuerKey, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, CRLException {
        GMSSLSancContentSigner contentSigner = new GMSSLSancContentSigner(algorithm, issuerKey, isWithParam);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    public static boolean verifyCRLBySanc(PublicKey publicKey, X509CRL crl) throws IOException, CRLException, CertException {
        GMSSLSancContentVerifierProvider contentVerifierProvider = new GMSSLSancContentVerifierProvider(publicKey);
        return verifyCRL(contentVerifierProvider, crl.getEncoded());
    }

    public static X509CRL generateCRLByMiniPcie(
            X509Certificate issueCert, PrivateKey issuerKey, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, CRLException {
        GMSSLXkfContentSigner contentSigner = new GMSSLXkfContentSigner(algorithm, issuerKey, isWithParam);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    public static boolean verifyCRLByMiniPcie(PublicKey publicKey, X509CRL crl) throws IOException, CRLException, CertException {
        GMSSLXkfContentVerifierProvider contentVerifierProvider = new GMSSLXkfContentVerifierProvider(publicKey);
        return verifyCRL(contentVerifierProvider, crl.getEncoded());
    }
    /************************************************************************************
     *                                 CRL 生成                                         *
     ************************************************************************************/

    /**
     * 生成 CRL 使用 BC 软实现签名
     *
     * @param issueCert  签发者证书
     * @param issuerKey  签发者私钥信息
     * @param algorithm  签发算法类型
     * @param crlNumber  CRL 序列号
     * @param thisUpdate 这次更新时间
     * @param nextUpdate 下次更新时间
     * @param extensions 扩展项列表
     * @param revoked    CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByBC(
            X509Certificate issueCert, PrivateKey issuerKey, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, issuerKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    public static X509CRL generateCRLByBC(
            X509Certificate issueCert, PrivateKey issuerKey, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, OperatorCreationException, CRLException {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, issuerKey, isWithParam);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }


    /**
     * 生成 CRL 使用 密码机 签名
     *
     * @param issueCert         签发者证书
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm         签名算法类型
     * @param crlNumber         CRL 序列号
     * @param thisUpdate        这次更新时间
     * @param nextUpdate        下次更新时间
     * @param extensions        扩展项列表
     * @param revoked           CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByYunhsm(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateCRL(issueCert, sdfPrivateKey, algorithm, crlNumber, thisUpdate, nextUpdate, extensions, revoked, false);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }


    public static X509CRL generateCRLByYunhsm(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateCRL(issueCert, sdfPrivateKey, algorithm, crlNumber, thisUpdate, nextUpdate, extensions, revoked, isWithParam);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey, isWithParam);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    /**
     * 生成 CRL 默认使用 PCIE 签名
     *
     * @param issueCert         签发者证书
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm         签名算法类型
     * @param crlNumber         CRL 序列号
     * @param thisUpdate        这次更新时间
     * @param nextUpdate        下次更新时间
     * @param extensions        扩展项列表
     * @param revoked           CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常userCertificateSerial
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByPCIE(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    public static X509CRL generateCRLByPCIE(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword, String algorithm,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked, boolean isWithParam
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey, isWithParam);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    /**
     * 生成 CRL
     *
     * @param issueCert     签发者证书
     * @param contentSigner 内容签名者
     * @param crlNumber     CRL 序列号
     * @param thisUpdate    这次更新时间
     * @param nextUpdate    下次更新时间
     * @param extensions    扩展项列表
     * @param revoked       CRL实体 列表
     * @return X509CRL
     * @throws IOException  添加扩展项异常
     * @throws CRLException CRL生成异常
     */
    public static X509CRL generateCRL(
            X509Certificate issueCert, ContentSigner contentSigner,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions, List<CRLEntry> revoked
    ) throws IOException, CRLException {
        X509v2CRLBuilder builder = new X509v2CRLBuilder(new X500Name(issueCert.getSubjectDN().getName()), thisUpdate);

        builder.addExtension(Extension.cRLNumber, false, new CRLNumber(crlNumber));
        builder.setNextUpdate(nextUpdate);

        //extensions
        if (extensions != null) {
            for (Extension extension : extensions) {
                builder.addExtension(extension);
            }
        }

        if (revoked != null) {
            for (CRLEntry entry : revoked) {
                builder.addCRLEntry(new BigInteger(entry.getUserCertificateSerial(), 16), entry.getRevocationDate(), entry.getReason());
            }
        }

        X509CRLHolder crlHolder = builder.build(contentSigner);
        return new JcaX509CRLConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCRL(crlHolder);
    }

    public static X509CRL decodeCRLByBC(PublicKey publicKey, byte[] crlEncode) throws OperatorCreationException, CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByBC(publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRLByYunhsm(PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        return decodeCRLBySdf(SdfCryptoType.YUNHSM, publicKey, crlEncode);
    }

    public static X509CRL decodeCRLByPcie(PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        return decodeCRLBySdf(SdfCryptoType.PCIE, publicKey, crlEncode);
    }

    public static X509CRL decodeCRLBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierBySdf(sdfCryptoType, publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRL(ContentVerifierProvider contentVerifierProvider, byte[] crlEncode) throws IOException, CertException, CRLException {
        X509CRLHolder holder = new X509CRLHolder(crlEncode);
        holder.isSignatureValid(contentVerifierProvider);
        return new JcaX509CRLConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCRL(holder);
    }

    private static boolean verifyCRL(ContentVerifierProvider contentVerifierProvider, byte[] crlEncode) throws IOException, CertException {
        X509CRLHolder holder = new X509CRLHolder(crlEncode);
        return holder.isSignatureValid(contentVerifierProvider);
    }
}
