package com.xdja.pki.gmssl.x509.utils;

import com.xdja.SafeKey.bean.MiniPcieIndexEnum;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentSigner;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentVerifierProvider;
import com.xdja.pki.gmssl.operator.utils.GMSSLContentSignerUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLContentVerifierProviderUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLXkfContentSigner;
import com.xdja.pki.gmssl.operator.utils.GMSSLXkfContentVerifierProvider;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.*;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;

import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.*;
import java.util.Date;
import java.util.List;

public class GMSSLCertUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 签发证书接口
     *
     * @param issuer      签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject     证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial      证书序列号
     * @param notBefore   证书有效期开始时间
     * @param notAfter    证书有效期结束时间
     * @param subjectKey  证书主体 公钥信息
     * @param issuerKey   证书签发者 私钥信息
     * @param algorithm   证书签名算法类型 可通过 enum com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm
     * @param extensions  证书扩展项列表
     * @param isWithParam 是否需要签名扩展参数 默认填写false，仅支持卡斯柯项目SM2签名填写true
     * @return
     * @throws Exception
     */
    public static X509Certificate generateCert(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter, issuerKey, subjectKey, algorithm, extensions, isWithParam);
    }

    public static X509Certificate generateCert(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivateKey = (SdfPrivateKey) issuerKey;
                return generateCertByPcie(issuer, subject, serial, notBefore, notAfter, pciePrivateKey.getIndex(), pciePrivateKey.getStringPassword(), subjectKey, algorithm, extensions, isWithParam);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivateKey = (SdfPrivateKey) issuerKey;
                ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, hsmPrivateKey, isWithParam);
                return generateCert(issuer, subject, serial, notBefore, notAfter, subjectKey, contentSigner, extensions);
            case MINI_PCI_E:
                return generateCertByMiniPcie(issuer, subject, serial, notBefore, notAfter, issuerKey, subjectKey, algorithm, extensions, isWithParam);
            case SANC_HSM:
                return generateCertBySanc(issuer, subject, serial, notBefore, notAfter, issuerKey, subjectKey, algorithm, extensions, isWithParam);
            case BC:
            default:
                return generateCertByBC(
                        issuer, subject,
                        serial, notBefore, notAfter,
                        issuerKey, subjectKey, algorithm,
                        extensions, isWithParam
                );
        }
    }

    /**
     * 验签证书
     *
     * @param verifyKey   验签公钥
     * @param certificate 证书
     * @return
     * @throws Exception
     */
    public static boolean verifyCert(PublicKey verifyKey, X509Certificate certificate) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return verifyCertByPCIE(verifyKey, certificate);
            case XDJA_HSM:
                return verifyCertByYunHsm(verifyKey, certificate);
            case MINI_PCI_E:
                return verifyCertByMiniPcie(verifyKey, certificate);
            case SANC_HSM:
                return verifyCertBySanc(verifyKey, certificate);
            case BC:
            default:
                return verifyCertByBC(verifyKey, certificate);
        }
    }

    /**
     * 写入证书  仅支持 MINI_PCI_E
     *
     * @param index         秘钥索引
     * @param password      私钥访问控制码
     * @param certificate   证书
     * @param isEncryptCert 是否为加密证书
     * @return
     * @throws Exception
     */
    public static boolean writeCertToHardWare(int index, String password, X509Certificate certificate, boolean isEncryptCert) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case MINI_PCI_E:
                if (isEncryptCert) {
                    return MiniPcieXKFUtils.writeCert(certificate.getEncoded(),
                            MiniPcieIndexEnum.getInfoFromIndex(index).getEncCertIndex(),
                            password);
                } else {
                    return MiniPcieXKFUtils.writeCert(certificate.getEncoded(),
                            MiniPcieIndexEnum.getInfoFromIndex(index).getSignCertIndex(),
                            password);
                }
            default:
                throw new Exception("un support writeCertToHardWare  with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * 读取证书  仅支持 MINI_PCI_E
     *
     * @param index         秘钥索引
     * @param isEncryptCert 是否为加密证书
     * @return
     * @throws Exception
     */
    public static X509Certificate readCertFromHardWare(int index, boolean isEncryptCert) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case MINI_PCI_E:
                if (isEncryptCert) {
                    byte[] certEncode = MiniPcieXKFUtils.readCert(MiniPcieIndexEnum.getInfoFromIndex(index).getEncCertIndex());
                    return GMSSLX509Utils.readCertificateFromCerByte(certEncode);
                } else {
                    byte[] certEncode = MiniPcieXKFUtils.readCert(MiniPcieIndexEnum.getInfoFromIndex(index).getSignCertIndex());
                    return GMSSLX509Utils.readCertificateFromCerByte(certEncode);
                }
            default:
                throw new Exception("un support writeCertToHardWare  with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    public static X509Certificate generateCertBySanc(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws CertIOException, CertificateException {
        return generateCertBySanc(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter, issuerKey, subjectKey, algorithm, extensions, isWithParam
        );
    }

    public static X509Certificate generateCertBySanc(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws CertIOException, CertificateException {
        GMSSLSancContentSigner contentSigner = new GMSSLSancContentSigner(algorithm, issuerKey, isWithParam);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    public static boolean verifyCertBySanc(PublicKey publicKey, X509Certificate certificate) throws Exception {
        GMSSLSancContentVerifierProvider contentVerifierProvider = new GMSSLSancContentVerifierProvider(publicKey);
        return verifyCert(contentVerifierProvider, certificate);
    }

    public static X509Certificate generateCertByMiniPcie(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws CertIOException, CertificateException {
        return generateCertByMiniPcie(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter, issuerKey, subjectKey, algorithm, extensions, isWithParam
        );
    }

    public static X509Certificate generateCertByMiniPcie(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws CertIOException, CertificateException {
        GMSSLXkfContentSigner contentSigner = new GMSSLXkfContentSigner(algorithm, issuerKey, isWithParam);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    public static boolean verifyCertByMiniPcie(PublicKey publicKey, X509Certificate certificate) throws Exception {
        GMSSLXkfContentVerifierProvider contentVerifierProvider = new GMSSLXkfContentVerifierProvider(publicKey);
        return verifyCert(contentVerifierProvider, certificate);
    }

    /************************************************************************************
     *                                  证书生成                                        *
     ************************************************************************************/
    /**
     * 生成证书  BC 软实现签名
     *
     * @param issuer     签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject    证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial     证书序列号
     * @param notBefore  证书有效期开始时间
     * @param notAfter   证书有效期结束时间
     * @param subjectKey 证书主体 公钥信息
     * @param issuerKey  证书签发者 私钥信息
     * @param algorithm  证书签名算法类型 可通过 enum com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm
     * @param extensions 证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByBC(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions
    ) throws Exception {
        return generateCertByBC(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                issuerKey, subjectKey, algorithm,
                extensions, false
        );
    }

    public static X509Certificate generateCertByBC(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        return generateCertByBC(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                issuerKey, subjectKey, algorithm,
                extensions, isWithParam
        );
    }

    public static X509Certificate generateCertByBC(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions
    ) throws Exception {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, issuerKey);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    public static X509Certificate generateCertByBC(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, issuerKey, isWithParam);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    /**
     * 生成证书 使用密码机签名
     *
     * @param issuer            签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject           证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial            证书序列号
     * @param notBefore         证书有效期开始时间
     * @param notAfter          证书有效期结束时间
     * @param subjectKey        证书主体 公钥信息
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm         签名算法类型
     * @param extensions        证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey, String algorithm,
            List<Extension> extensions
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        return generateCertByYunhsm(
                issuer, subject,
                serial, notBefore, notAfter,
                sdfPrivateKey, subjectKey, algorithm,
                extensions
        );
    }


    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        return generateCertByYunhsm(
                issuer, subject,
                serial, notBefore, notAfter,
                sdfPrivateKey, subjectKey, algorithm,
                extensions, isWithParam
        );
    }

    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey sdfPrivateKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions
    ) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateCert(issuer, subject, serial, notBefore, notAfter, sdfPrivateKey, subjectKey, algorithm, extensions, false);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey);
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey sdfPrivateKey, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateCert(issuer, subject, serial, notBefore, notAfter, sdfPrivateKey, subjectKey, algorithm, extensions, isWithParam);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey, isWithParam);
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    /**
     * 生成证书 使用密码机签名
     *
     * @param issuer            签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject           证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial            证书序列号
     * @param notBefore         证书有效期开始时间
     * @param notAfter          证书有效期结束时间
     * @param subjectKey        证书主体 公钥信息
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm         签名算法类型
     * @param extensions        证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByPcie(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey, String algorithm,
            List<Extension> extensions
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey);
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, contentSigner, extensions
        );
    }

    public static X509Certificate generateCertByPcie(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        return generateCertByPcie(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                issuerKeyIndex, issuerKeyPassword, subjectKey, algorithm, extensions, isWithParam);
    }

    public static X509Certificate generateCertByPcie(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey, String algorithm,
            List<Extension> extensions, boolean isWithParam
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey, isWithParam);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner, extensions
        );
    }

    /**
     * 生成证书 X500Name DN
     *
     * @param issuer        签发者 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject       证书主体 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject       证书主体 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param contentSigner 内容签名者
     * @param serial        证书序列号
     * @param notBefore     证书有效期开始时间
     * @param notAfter      证书有效期结束时间
     * @param subjectKey    证书主体 公钥信息
     * @param extensions    证书扩展项列表
     * @return X509证书
     * @throws CertIOException      添加扩展项异常
     * @throws CertificateException 证书生成异常
     */
    public static X509Certificate generateCert(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PublicKey subjectKey, ContentSigner contentSigner,
            List<Extension> extensions
    ) throws CertIOException, CertificateException {

        //subject public key info
        SubjectPublicKeyInfo subjectPublicKeyInfo = GMSSLX509Utils.convertSubjectPublicKeyInfo(subjectKey);

        //builder
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(issuer, serial, notBefore, notAfter, subject, subjectPublicKeyInfo);

        if (extensions != null) {
            //extensions
            for (Extension extension : extensions) {
                //add extension error throw CertIOException
                builder.addExtension(extension);
            }
        }

        //signer
        X509CertificateHolder holder = builder.build(contentSigner);

        //gen certificate getCertificate throw CertificateException
        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
    }

    public static X509Certificate decodeCertByBC(PublicKey publicKey, byte[] certEncode) throws OperatorCreationException, CertificateException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByBC(publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCertByYunhsm(PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {
        return decodeCertBySdf(SdfCryptoType.YUNHSM, publicKey, certEncode);
    }

    public static X509Certificate decodeCertByPcie(PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {
        return decodeCertBySdf(SdfCryptoType.PCIE, publicKey, certEncode);
    }

    public static X509Certificate decodeCertBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {

        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierBySdf(sdfCryptoType, publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCert(ContentVerifierProvider contentVerifierProvider, byte[] certEncode) throws IOException, CertException, CertificateException {
        X509CertificateHolder holder = new X509CertificateHolder(certEncode);
        holder.isSignatureValid(contentVerifierProvider);
        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
    }

    public static boolean verifyCertByBC(PublicKey publicKey, X509Certificate cert) throws Exception {
        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByBC(publicKey);
        return verifyCert(contentVerifierProvider, cert);
    }

    public static boolean verifyCertByYunHsm(PublicKey publicKey, X509Certificate cert) throws Exception {
        return verifyCertBySdf(SdfCryptoType.YUNHSM, publicKey, cert);
    }

    public static boolean verifyCertByPCIE(PublicKey publicKey, X509Certificate cert) throws Exception {
        return verifyCertBySdf(SdfCryptoType.PCIE, publicKey, cert);
    }

    public static boolean verifyCertBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, X509Certificate cert) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return verifyCert(publicKey, cert);
        }
        ContentVerifierProvider contentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierBySdf(sdfCryptoType, publicKey);
        return verifyCert(contentVerifierProvider, cert);
    }

    public static boolean verifyCert(ContentVerifierProvider verifierProvider, X509Certificate cert) throws Exception {
        return verifyCert(verifierProvider, cert.getEncoded());
    }

    public static boolean verifyCert(ContentVerifierProvider verifierProvider, byte[] cert) throws Exception {
        X509CertificateHolder holder = new X509CertificateHolder(cert);
        return holder.isSignatureValid(verifierProvider);
    }
}
