package com.xdja.pki.gmssl.sdf;

import com.xdja.pki.gmssl.sdf.bean.*;

import java.io.InputStream;
import java.io.OutputStream;
import java.security.KeyPair;

public interface SdfSDK {

    void init() throws SdfSDKException;

    void release() throws SdfSDKException;

    byte[] generateRandom(int uiLength) throws SdfSDKException;

    SdfECCKeyPair generateKeyPairEcc() throws SdfSDKException;

    SdfECCKeyPair generateKeyPairEcc(int algId) throws SdfSDKException;

    boolean checkPrivateKeyAccessRight(int index, byte[] password) throws SdfSDKException;

    SdfECCSignature internalSignECC(int index, byte[] password, byte[] data) throws SdfSDKException;

    SdfECCPublicKey exportSignPublicKeyEcc(int index) throws SdfSDKException;

    void externalVerifyECC(SdfECCPublicKey key, byte[] data, SdfECCSignature pucSignature) throws SdfSDKException;

    void externalVerifyECC(SdfECCPublicKey publicKey, byte[] data, SdfECCSignature sdfECCSignature, String signAlgName) throws SdfSDKException;

    SdfECCPublicKey exportEncPublicKeyEcc(int index) throws SdfSDKException;

    SdfECCCipher externalEncryptECC(SdfECCPublicKey publicKey, byte[] data) throws SdfSDKException;

    SdfECCCipher externalEncryptECC(SdfECCPublicKey publicKey, byte[] data, String stdName) throws SdfSDKException;

    byte[] internalDecryptECC(int index, byte[] password, int len, SdfECCCipher pucEncData) throws SdfSDKException;

    byte[] internalDecryptECC(int index, byte[] password, int len, SdfECCCipher sdfECCCipher, String stdName) throws SdfSDKException;

    /**
     * 生成ECC公钥加密的 会话密钥，返回加密会话密钥
     *
     * @param sdfECCPublicKey 加密的ECC公钥
     * @return ECC加密的会话密钥
     * @throws SdfSDKException 异常
     */
    SdfECCCipher generateKeyWithEpkEcc(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException;

    /**
     * 生成ECC公钥加密的 会话密钥，返回handle
     *
     * @param sdfECCPublicKey 加密的ECC公钥
     * @return 加密的会话密钥
     * @throws SdfSDKException 异常
     */
    long[] generateKeyWithEpkEccHandle(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException;

    /**
     * 生成ECC公钥加密的 会话密钥，返回加密会话密钥及handle
     *
     * @param sdfECCPublicKey 加密ECC公钥
     * @return 加密的会话密钥 和 会话密钥所在handle
     * @throws SdfSDKException
     */
    SdfSymmetricKeyHandle generateKeyWithEpkEccKeyHandle(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException;

    /**
     * 导入ECC公钥加密的 会话密钥
     *
     * @param uiIskIndex   加密ECC公钥对应的私钥 索引
     * @param password     加密ECC公钥对应的私钥 访问控制码
     * @param sdfECCCipher ECC加密结构体
     * @return 会话密钥所在 handle
     * @throws SdfSDKException 异常
     */
    long[] importKeyWithIskEcc(int uiIskIndex, byte[] password, SdfECCCipher sdfECCCipher) throws SdfSDKException;

    long[] importKey(byte[] pucKey) throws SdfSDKException;

    byte[] encrypt(long[] phKeyHandle, SdfAlgIdSymmetric sdfAlgIdBlockCipher, byte[] pucIv, byte[] pucData) throws SdfSDKException;

    byte[] decrypt(long[] phKeyHandle, SdfAlgIdSymmetric sdfAlgIdBlockCipher, byte[] pucIv, byte[] pucEncData) throws SdfSDKException;

    void destroyKey(long[] phKeyHandle) throws SdfSDKException;

    void hashInit(SdfAlgIdHash sdfAlgIdHash) throws SdfSDKException;

    void hashInit(SdfAlgIdHash sdfAlgIdHash, byte[] pucID, SdfECCPublicKey publicKey) throws SdfSDKException;

    void hashUpdate(byte[] data) throws SdfSDKException;

    byte[] hashFinal(int digestLen) throws SdfSDKException;

    void calculateMac(int uiKeyIndex) throws SdfSDKException;

    byte[] sm3Hmac(byte[] dataIn, byte[] key) throws SdfSDKException;

    SdfRSAPublicKey exportSignPublicKeyRsa(int index) throws SdfSDKException;

    SdfRSAPublicKey exportEncPublicKeyRsa(int index) throws SdfSDKException;

    SdfRsaKeyPair generateKeyPairRsa(int bits) throws SdfSDKException;

    byte[] externalPublicKeyOperationRsa(SdfRSAPublicKey sdfRSAPublicKey, byte[] data) throws SdfSDKException;

    byte[] internalPrivateKeyOperationRsa(int index, byte[] password, byte[] data) throws SdfSDKException;

    /**
     * AEAD 加密 对称消息验证码认证加密
     *
     * @param phKeyHandle key 所在的密钥句柄
     * @param uiAlgId     算法标识
     * @param tagLen      消息验证码长度
     * @param nonce       nonce 特定数据
     * @param aad         aad 附加数据
     * @param data        明文数据 待加密数据
     * @return 输出 密文数据 + 校验码 ： cipher || tag
     */
    byte[] encryptAead(long[] phKeyHandle, SdfAlgIdAead uiAlgId, int tagLen, byte[] nonce, byte[] aad, byte[] data) throws SdfSDKException;

    /**
     * AEAD 解密 对称消息验证码认证解密
     *
     * @param phKeyHandle key 所在的密钥句柄
     * @param uiAlgId     算法标识
     * @param tagLen      消息验证码长度
     * @param nonce       nonce 特定数据
     * @param aad         aad 附加数据
     * @param enc         密文数据 待解密数据
     * @return 输出 明文数据 + 校验码 ： plaintext || tag
     */
    byte[] decryptAead(long[] phKeyHandle, SdfAlgIdAead uiAlgId, int tagLen, byte[] nonce, byte[] aad, byte[] enc) throws SdfSDKException;

    /**
     * 导入 会话密钥 并用 密钥加密密钥 解密
     *
     * @param uiAlgId    算法标识 加密模式使用ECB模式
     * @param uiKekIndex 密钥加密密钥索引值
     * @param pucKey     会话密钥密文
     * @return 会话密钥句柄
     */
    long[] importKeyWithKek(SdfAlgIdSymmetric uiAlgId, int uiKekIndex, byte[] pucKey) throws SdfSDKException;


    /**
     * 生成会话密钥并用 RSA公钥 加密输出，不返回handle
     *
     * @param sdfRSAPublicKey 加密的RSA公钥
     * @return RSA加密的会话密钥
     * @throws SdfSDKException 异常
     */
    byte[] generateKeyWithEpkRsa(SdfRSAPublicKey sdfRSAPublicKey) throws SdfSDKException;

    /**
     * 生成会话密钥并用 RSA公钥 加密输出，返回handle
     *
     * @param sdfRSAPublicKey 加密的RSA公钥
     * @return 加密的会话密钥
     * @throws SdfSDKException 异常
     */
    long[] generateKeyWithEpkRsaHandle(SdfRSAPublicKey sdfRSAPublicKey) throws SdfSDKException;

    /**
     * 生成RSA公钥加密的 会话密钥，返回加密会话密钥及handle
     *
     * @param sdfRSAPublicKey 加密RSA公钥
     * @return 加密的会话密钥 和 会话密钥所在handle
     * @throws SdfSDKException
     */
    SdfSymmetricKeyHandle generateKeyWithEpkRsaKeyHandle(SdfRSAPublicKey sdfRSAPublicKey) throws SdfSDKException;

    /**
     * 导入RSA公钥加密的 会话密钥
     *
     * @param uiIskIndex 加密RSA公钥对应的私钥 索引
     * @param password   加密RSA公钥对应的私钥 访问控制码
     * @param pucKey     加密的会话密钥
     * @return 会话密钥所在 handle
     * @throws SdfSDKException 异常
     */
    long[] importKeyWithIskRsa(int uiIskIndex, byte[] password, byte[] pucKey) throws SdfSDKException;

    /**
     * 生成Kek
     *
     * @param uiAlgId    kek加密方式
     * @param uiKekIndex kek对应的密钥索引
     * @return kek密文
     */
    byte[] generateKeyWithKek(int uiAlgId, int uiKekIndex) throws SdfSDKException;

    /*
     * 硬件文件处理接口
     */

    /**
     * 创建文件
     *
     * @param fileName 文件名
     * @param fileSize 文件大小
     * @return 读取文件的内容
     */
    void createFile(String fileName, int fileSize) throws SdfSDKException;

    /**
     * 读文件
     *
     * @param fileName 文件名
     * @param offset   起始位置
     * @param length   读取文件的长度
     * @return 读取文件的内容
     */
    byte[] readFile(String fileName, int offset, int length) throws SdfSDKException;

    /**
     * 写文件
     *
     * @param fileName 文件名
     * @param offset   起始位置
     * @param context  要写入的文件内容
     */
    void writeFile(String fileName, int offset, byte[] context) throws SdfSDKException;

    /**
     * 删除文件
     *
     * @param fileName 文件名
     */
    void deleteFile(String fileName) throws SdfSDKException;


    /**
     * 导出ECC签名公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的签名公钥。
     *
     * @param index 密钥索引
     */
    SdfECCPublicKey exportSignPublicKeyEccEx(int index) throws SdfSDKException;

    /**
     * 导出ECC加密公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的加密公钥。
     *
     * @param index 密钥索引
     */
    SdfECCPublicKey exportEncPublicKeyEccEx(int index) throws SdfSDKException;

    /**
     * 产生ECC密钥对并输出
     * <p/>
     * 描述：请求密码设备产生指定类型和模长的ECC密钥对。
     *
     * @param algId 算法ID
     */
    SdfECCKeyPair generateKeyPairEccEx(int algId, int bits) throws SdfSDKException;

    /**
     * 外部公钥验签
     *
     * @param key             公钥
     * @param data            数据原文
     * @param pucSignature    签名值
     * @param algIdAsymmetric 非对称算法
     * @throws SdfSDKException
     */
    void externalVerifyECCEx(SdfECCPublicKey key, byte[] data, SdfECCSignature pucSignature, SdfAlgIdAsymmetric algIdAsymmetric, int bits) throws SdfSDKException;

    /**
     * 内部私钥签名
     *
     * @param index    密钥索引
     * @param password 密码
     * @param data     数据
     * @return 签名值
     * @throws SdfSDKException
     */
    SdfECCSignature internalSignECCEx(int index, byte[] password, byte[] data, int bits) throws SdfSDKException;

    /**
     * 外部公钥加密
     *
     * @param publicKey       外部公钥
     * @param data            数据
     * @param bits            密钥长度
     * @param algIdAsymmetric 算法标识
     * @return 密文数据
     * @throws SdfSDKException
     */
    SdfECCCipher externalEncryptECCEx(SdfECCPublicKey publicKey, byte[] data, int bits, SdfAlgIdAsymmetric algIdAsymmetric) throws SdfSDKException;

    /**
     * 内部私钥解密
     *
     * @param index           密钥索引
     * @param password        访问控制码
     * @param len             长度
     * @param pucEncData      加密数据
     * @param bits            密钥长度
     * @param algIdAsymmetric 算法标识
     * @return 加密数据
     * @throws SdfSDKException
     */
    byte[] internalDecryptECCEx(int index, byte[] password, int len, SdfECCCipher pucEncData, int bits, SdfAlgIdAsymmetric algIdAsymmetric) throws SdfSDKException;


}
