package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.hsm.api.alg.AlgId;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCKeyPair;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPrivateKey;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.Key;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;

public class SdfECKey {

    private SdfSDK sdfSDK;
    private String stdName;
    private Logger logger = LoggerFactory.getLogger(SdfECKey.class);
    private SdfCryptoType sdfCryptoType;
    private static final int LENGTH_32 = 32;
    private static final int LENGTH_48 = 48;
    private static final int LENGTH_66 = 66;

    public SdfECKey() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
        this.stdName = GMSSLECUtils.SM2p256;
        this.sdfCryptoType = SdfCryptoType.YUNHSM;
    }

    public SdfECKey(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this.sdfSDK = sdfCryptoType.getSdfSDK();
        this.stdName = GMSSLECUtils.SM2p256;
        sdfSDK.init();
        this.sdfCryptoType = sdfCryptoType;
    }


    public SdfECKey(SdfCryptoType sdfCryptoType, String stdName) throws SdfSDKException {
        this.sdfSDK = sdfCryptoType.getSdfSDK();
        this.stdName = stdName;
        sdfSDK.init();
        this.sdfCryptoType = sdfCryptoType;
    }

    public PublicKey exportSignPublicKey(int index) throws SdfSDKException {
        // TODO: 2021/6/30 根据曲线名称判断导出
        return exportPublicKey(index, false);
    }

    public PublicKey exportEncryptPublicKey(int index) throws SdfSDKException {
        return exportPublicKey(index, true);
    }


    public PublicKey exportPublicKey(int index, boolean isEncrypt) throws SdfSDKException {
        try {
            SdfECCPublicKey sdfECCPublicKey;
            switch (stdName) {
                case Constants.NIST_P_256:
                case Constants.SM2_P_256:
                    if (isEncrypt) {
                        sdfECCPublicKey = this.sdfSDK.exportEncPublicKeyEcc(index);
                    } else {
                        sdfECCPublicKey = this.sdfSDK.exportSignPublicKeyEcc(index);
                    }
                    return GMSSLECUtils.convertECPublicKey(sdfECCPublicKey.getX(), sdfECCPublicKey.getY(), stdName);
                case Constants.NIST_P_384:
                case Constants.NIST_P_521:
                    if (isEncrypt) {
                        sdfECCPublicKey = this.sdfSDK.exportEncPublicKeyEccEx(index);
                    } else {
                        sdfECCPublicKey = this.sdfSDK.exportSignPublicKeyEccEx(index);
                    }
                    return GMSSLECUtils.convertPublicKey(sdfECCPublicKey.getX(), sdfECCPublicKey.getY(), stdName);
                default:
                    logger.error("不支持的曲线，名称为{}", stdName);
                    throw new SdfSDKException("不支持的曲线");
            }
        } catch (Exception e) {
            logger.error("导出公钥失败", e);
            throw new SdfSDKException(e);
        }
    }


    /**
     * 生成密钥对
     *
     * @return 返回密钥对
     * @throws SdfSDKException
     */
    public KeyPair generateKeyPair() throws SdfSDKException {
        try {
            SdfECCKeyPair sdfECCKeyPair;
            switch (stdName) {
                case Constants.NIST_P_256:
                    sdfECCKeyPair = sdfSDK.generateKeyPairEcc(AlgId.SGD_ECC_NISTP256);
                    break;
                case Constants.SM2_P_256:
                    sdfECCKeyPair = sdfSDK.generateKeyPairEcc(AlgId.SGD_SM2);
                    break;
                // TODO: 2021/6/30 算法
                case Constants.NIST_P_384:
                    sdfECCKeyPair = sdfSDK.generateKeyPairEccEx(AlgId.SGD_ECC_NISTP384,Constants.KEY_BITS_384);
                    break;
                case Constants.NIST_P_521:
                    sdfECCKeyPair = sdfSDK.generateKeyPairEccEx(AlgId.SGD_ECC_NISTP521,Constants.KEY_BITS_521);
                    break;
                default:
                    logger.error("不支持的曲线，名称为{}", stdName);
                    throw new SdfSDKException("不支持的曲线");
            }
            PrivateKey privateKey = generateEcPrivateKey(sdfECCKeyPair.getSdfECCPrivateKey());
            PublicKey publicKey = generateEcPublicKey(sdfECCKeyPair.getSdfECCPublicKey());
            sdfSDK.release();
            return new KeyPair(publicKey, privateKey);
        } catch (Exception e) {
            logger.error("生成密钥对失败，曲线名称为{}", stdName, e);
            throw new SdfSDKException(e);
        }
    }

    /**
     * 转换为EC私钥
     *
     * @param sdfECCPrivateKey sdf格式的私钥
     * @return EC的私钥
     * @throws SdfSDKException
     */
    private PrivateKey generateEcPrivateKey(SdfECCPrivateKey sdfECCPrivateKey) throws SdfSDKException {
        try {
            switch (stdName) {
                case Constants.NIST_P_256:
                case Constants.SM2_P_256:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(sdfECCPrivateKey.getK(), LENGTH_32), stdName);
                    } else {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(sdfECCPrivateKey.getK(), LENGTH_32), stdName);
                    }
                case Constants.NIST_P_521:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(sdfECCPrivateKey.getK(), LENGTH_66), stdName);
                    } else {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(sdfECCPrivateKey.getK(), LENGTH_66), stdName);
                    }
                case Constants.NIST_P_384:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(sdfECCPrivateKey.getK(), LENGTH_48), stdName);
                    } else {
                        return GMSSLECUtils.convertECPrivateKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(sdfECCPrivateKey.getK(), LENGTH_48), stdName);
                    }
                default:
                    logger.error("不支持的曲线，名称为{}", stdName);
                    throw new SdfSDKException("不支持的曲线");
            }
        } catch (Exception e) {
            logger.error("转换私钥失败，曲线名称为{}", stdName, e);
            throw new SdfSDKException("转换私钥失败", e);
        }
    }

    /**
     * 转换公钥
     *
     * @param publicKey sdf格式公钥
     * @return 公钥
     * @throws SdfSDKException
     */
    private PublicKey generateEcPublicKey(SdfECCPublicKey publicKey) throws SdfSDKException {
        try {
            switch (stdName) {
                case Constants.SM2_P_256:
                case Constants.NIST_P_256:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getX(), LENGTH_32),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_32),
                                stdName);
                    } else {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(publicKey.getX(), LENGTH_32),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_32),
                                stdName);
                    }
                case Constants.NIST_P_384:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getX(), LENGTH_48),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_48),
                                stdName);
                    } else {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(publicKey.getX(), LENGTH_48),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_48),
                                stdName);
                    }
                case Constants.NIST_P_521:
                    if (sdfCryptoType == SdfCryptoType.YUNHSM) {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getX(), LENGTH_66),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_66),
                                stdName);
                    } else {
                        return GMSSLECUtils.convertPublicKey(GMSSLByteArrayUtils.filterByteArrayZeroInFoot(publicKey.getX(), LENGTH_66),
                                GMSSLByteArrayUtils.filterByteArrayZeroInHead(publicKey.getY(), LENGTH_66),
                                stdName);
                    }
                default:
                    logger.error("不支持的曲线，名称为{}", stdName);
                    throw new SdfSDKException("不支持的曲线");
            }
        } catch (Exception e) {
            logger.error("转换公钥钥失败，曲线名称为{}", stdName, e);
            throw new SdfSDKException("转换私钥失败", e);
        }
    }

    public void release() throws SdfSDKException {
        this.sdfSDK.release();
    }
}
