package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.hsm.api.alg.AlgId;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdAead;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.params.AEADParameters;
import org.bouncycastle.crypto.params.ParametersWithIV;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author sxy
 * @description
 * @date 2020年03月06日 10:02
 **/
public class SdfAeadCipher {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final int BLOCK_SIZE = 16;

    private SdfSDK sdfSDK;

    private long[] phKeyHandle;
    private byte[] key;
    private boolean forEncryption;

    private AEADParameters aeadParameters;

    public SdfAeadCipher() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfAeadCipher(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfAeadCipher(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
    }

    public byte[] getKey() {
        return key;
    }

    public void init(boolean forEncryption, CipherParameters params) throws IllegalArgumentException {
        try {
            this.sdfSDK.init();
        } catch (SdfSDKException e) {
            throw new IllegalArgumentException("sdf SDK init error", e);
        }
        if (params instanceof AEADParameters) {
            this.aeadParameters = (AEADParameters) params;
        } else {
            throw new IllegalArgumentException("invalid parameter passed to SdfAeadCipher init - " + params.getClass().getName());
        }

        this.forEncryption = forEncryption;

        this.key = this.aeadParameters.getKey().getKey();

        try {
            this.phKeyHandle = sdfSDK.importKey(key);
        } catch (SdfSDKException e) {
            logger.error("SDF AEAD init import key", e);
            throw new IllegalArgumentException("SDF AEAD init - import key error");
        }
    }

    public String getAlgorithmName() {
        return "AEAD";
    }

    public int getBlockSize() {
        return BLOCK_SIZE;
    }

    public byte[] doFinal(byte[] data) throws DataLengthException, IllegalStateException {
        int tagLen = this.aeadParameters.getMacSize();
        byte[] nonce = this.aeadParameters.getNonce();
        byte[] aad = this.aeadParameters.getAssociatedText();
        try {
            if (this.forEncryption) {
                return sdfSDK.encryptAead(phKeyHandle, SdfAlgIdAead.SGD_AES_CCM_128, tagLen, nonce, aad, data);
            } else {
                return sdfSDK.decryptAead(phKeyHandle, SdfAlgIdAead.SGD_AES_CCM_128, tagLen, nonce, aad, data);
            }
        } catch (SdfSDKException e) {
            logger.error("doFinal forEncryption={} tagLen={}", forEncryption, tagLen);
            GMSSLByteArrayUtils.printHexBinary(logger, "doFinal key", this.key, true);
            GMSSLByteArrayUtils.printHexBinary(logger, "doFinal nonce", nonce, true);
            GMSSLByteArrayUtils.printHexBinary(logger, "doFinal aad", aad, true);
            GMSSLByteArrayUtils.printHexBinary(logger, "doFinal in", data, true);
            logger.error("doFinal error", e);
            throw new IllegalStateException(e.getMessage());
        }
    }

    public void reset() {

    }

    public void release() throws SdfSDKException {
        if (sdfSDK != null && phKeyHandle != null) {
            sdfSDK.destroyKey(phKeyHandle);
            sdfSDK.release();
        }
    }
}
