package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdHash;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import org.bouncycastle.crypto.*;
import org.bouncycastle.crypto.digests.*;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.BigIntegers;

import java.math.BigInteger;

public class SdfEcSignerWithSoftDigest extends SdfECBaseSigner {

    public SdfEcSignerWithSoftDigest() throws SdfSDKException {
        super(SdfCryptoType.YUNHSM);
    }

    public SdfEcSignerWithSoftDigest(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        super(sdfCryptoType.getSdfSDK());
    }

    public SdfEcSignerWithSoftDigest(SdfSDK sdfSDK) throws SdfSDKException {
        super(sdfSDK);
    }

    public SdfEcSignerWithSoftDigest(SdfSDK sdfSDK, String signAlgName) throws SdfSDKException {
        super(sdfSDK, signAlgName);
    }

    public SdfEcSignerWithSoftDigest(SdfCryptoType sdfCryptoType, String signAlgName) throws SdfSDKException {
        super(sdfCryptoType.getSdfSDK(), signAlgName);
    }

    public SdfEcSignerWithSoftDigest(SdfCryptoType sdfCryptoType, String signAlgName, int bits) throws SdfSDKException {
        super(sdfCryptoType.getSdfSDK(), signAlgName);
        this.bits = bits;
    }

    @Override
    public void initDigest(byte[] userID, SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException {
        digest = new SM3Digest();
        byte[] z = getZ(userID, sdfECCPublicKey.getX(), sdfECCPublicKey.getY());
//            GMSSLByteArrayUtils.printHexBinary(logger, "z", z);
        digest.update(z, 0, z.length);
    }

    @Override
    public void initDigest(String signAlgName) throws SdfSDKException {
        switch (signAlgName.toUpperCase()) {
            case Constants.SHA1_WITH_ECDSA:
                digest = new SHA1Digest();
                break;
            case Constants.SHA256_WITH_ECDSA:
                digest = new SHA256Digest();
                break;
            case Constants.SHA384_WITH_ECDSA:
                // TODO: 2021/6/30 增加384和512
                digest = new SHA384Digest();
                break;
            case Constants.SHA512_WITH_ECDSA:
                digest = new SHA512Digest();
                break;
            default:
                throw new SdfSDKException("unSupport signAlg " + signAlgName);
        }
    }


    private byte[] getZ(byte[] userID, byte[] x, byte[] y) {
        digest.reset();

        addUserID(digest, userID);

        ECParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec(GMSSLX509Utils.ECC_SM2_NAME);
        ECCurve curve = ecParameterSpec.getCurve();
        ECPoint G = ecParameterSpec.getG();


        addFieldElement(digest, curve.getA().toBigInteger());
        addFieldElement(digest, curve.getB().toBigInteger());
        addFieldElement(digest, G.getAffineXCoord().toBigInteger());
        addFieldElement(digest, G.getAffineYCoord().toBigInteger());
        addFieldElement(digest, BigIntegers.fromUnsignedByteArray(x));
        addFieldElement(digest, BigIntegers.fromUnsignedByteArray(y));

        byte[] result = new byte[digest.getDigestSize()];

        digest.doFinal(result, 0);

        return result;
    }

    private void addUserID(Digest digest, byte[] userID) {
        int len = userID.length * 8;
        digest.update((byte) (len >> 8 & 0xFF));
        digest.update((byte) (len & 0xFF));
        digest.update(userID, 0, userID.length);
    }

    private void addFieldElement(Digest digest, BigInteger p) {
        byte[] in = GMSSLByteArrayUtils.asUnsignedByteArray32(p);
        digest.update(in, 0, in.length);
    }
}
