package com.xdja.pki.gmssl.sdf.yunhsm;

import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;

public class HsmConnectionPool {

    private static Logger logger = LoggerFactory.getLogger(HsmConnectionPool.class);
    private static boolean DEBUG = false;

    private static int POOL_MAX_LEN = 50; // max len
    private static int POOL_LESS_LEN = POOL_MAX_LEN / 5; // less than
    private static int FREE_LONG_TIME = 1000; // time ms

    static {
        checkPoolSize();
    }

    private LinkedBlockingQueue<HsmConnection> freePool = new LinkedBlockingQueue<>(POOL_MAX_LEN);
    private LinkedBlockingQueue<HsmConnection> usedPool = new LinkedBlockingQueue<>(POOL_MAX_LEN);

    public static void checkPoolSize() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        List<StackTraceElement> list = Arrays.asList(stackTrace);
        for (StackTraceElement element : list) {
            if (element.getClassName().startsWith("org.junit.")) {
                POOL_MAX_LEN = 10;
                FREE_LONG_TIME = 1000;
                DEBUG = false;
            }
            if (element.getClassName().startsWith("com.xdja.pki.gmssl.keystore.main")) {
                POOL_MAX_LEN = 1;
            }
            if (element.getClassName().startsWith("com.xdja.pki.gmssl.main.")) {
                POOL_MAX_LEN = 1;
            }
        }
    }

    private HsmConnectionPool() {
        initConnectionPool();
    }

    public synchronized boolean isConnection() {
        return this.usedPool.size() != 0 || this.freePool.size() != 0;
    }

    public synchronized void initConnectionPool() {
        if (isConnection()) {
            return;
        }
        try {
            HsmConnection connection = new HsmConnection();
            connection.getDeviceInfo();
            connection.release();
        } catch (SdfSDKException e) {
            logger.info("hsm connection pool init connection error, please check config, {}", e);
            return;
        }

        if (freePool.size() >= POOL_MAX_LEN) {
            logger.info("hsm connection pool is full !!!");
            return;
        }
        for (int i = 0; i < POOL_MAX_LEN; i++) {
            try {
                HsmConnection hsmConnection = new HsmConnection();
                hsmConnection.setId(i);
                hsmConnection.setStartTime(new Date().getTime());
                freePool.put(hsmConnection);
                logger.info("hsm connection pool init id={} dev={} session={}", i, hsmConnection.getDev(), hsmConnection.getSes());
            } catch (Exception e) {
                logger.error("init connection error", e);
            }
        }
        logger.info("hsm connection pool init done");
    }

    private static class HsmConnectionPoolHolder {
        private static final HsmConnectionPool instance = new HsmConnectionPool();
    }

    public static HsmConnectionPool getInstance() {
        return HsmConnectionPoolHolder.instance;
    }

    public synchronized HsmConnection getConnection() {
        if (this.freePool.size() == 0 && this.usedPool.size() == 0) {
            logger.info("get connection used free all null, now init connection pool!");
            initConnectionPool();
        }
        if (this.freePool.size() <= POOL_LESS_LEN && this.usedPool.size() > 0) {
            List<Integer> list = new ArrayList<>();
            for (HsmConnection conn : this.usedPool) {
                long now = new Date().getTime();
                if (now - conn.getStartTime() > FREE_LONG_TIME) {
                    try {
                        HsmConnection take = this.usedPool.take();
                        list.add(take.getId());
                        this.freePool.put(take);
                    } catch (InterruptedException e) {
                        logger.error("move connection error", e);
                    }
                }
            }
            if (DEBUG) {
                logger.info("now move connection from used to free pool len {} {}", list.size(), list);
            }
        }
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        String methodName = stackTrace[2].getMethodName();
        if (DEBUG) {
            logger.info("get now time pool size {}", freePool.size());
        }
        try {
            HsmConnection take = freePool.take();
            if (DEBUG) {
                logger.debug(" {} get connection id {}", methodName, take.getId());
            }
            take.setStartTime(new Date().getTime());
            usedPool.put(take);
            return take;
        } catch (InterruptedException e) {
            logger.error("get connection error {}", methodName, e);
            return null;
        }
    }

    public synchronized void releaseConnection(HsmConnection hsmConnection) {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        String methodName = stackTrace[2].getMethodName();
        try {
            for (HsmConnection conn : this.usedPool) {
                if (conn.getId() == hsmConnection.getId()) {
                    this.usedPool.remove(conn);
                    this.freePool.put(hsmConnection);
                }
            }
            if (DEBUG) {
                logger.info("release now time pool size {} producer is {} connection id {}", freePool.size(), methodName, hsmConnection.getId());
            }
            List<Integer> list = new ArrayList<>();
            for (HsmConnection conn : this.freePool) {
                list.add(conn.getId());
            }
            if (DEBUG) {
                logger.info("now time pool is len {} value {}", this.freePool.size(), list);
            }
        } catch (InterruptedException e) {
            logger.error("release connection error{}", methodName, e);
        }
    }

    public synchronized void releaseAllConnection() {
        for (HsmConnection conn : this.usedPool) {
            try {
                conn.release();
            } catch (SdfSDKException ignored) {
            }
            this.usedPool.remove(conn);
        }
        for (HsmConnection conn : this.freePool) {
            try {
                conn.release();
            } catch (SdfSDKException ignored) {
            }
            this.freePool.remove(conn);
        }
    }
}
