package com.xdja.pki.gmssl.sdf.yunhsm;

import com.xdja.hsm.api.SdfApi;
import com.xdja.hsm.api.bean.*;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfApiCode;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.InputStream;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;

public class YunhsmSdfSDKUtils {

    private static Logger logger = LoggerFactory.getLogger(YunhsmSdfSDKUtils.class);

    public static String getConfigPath() {
        //获取配置文件
        //C:\Program Files\yunhsmsdk\conf\yunhsmsdk.conf
        String cfgPath = System.getenv("YUNHSMSDK_CONF");
        if (cfgPath != null) {
            return cfgPath;
        }
        logger.debug("getenv(\"YUNHSMSDK_CONF\") return null, please set this env variable");
        String os = System.getProperty("os.name");
        if ("Linux".equals(os)) {
            return "/usr/local/yunhsmsdk/conf/yunhsmsdk.conf";
        }
        if (os.startsWith("Windows")) {
            return "C:\\Program Files\\yunhsmsdk\\conf\\yunhsmsdk.conf";
        }
        return "";
    }


    public static void testConnection(String configPath) throws SdfSDKException {
        if (configPath == null) {
            configPath = getConfigPath();
        }
        long[] dev = {0}, ses = {0};
        SdfApi sdfApi = new SdfApi();
        openDevice(sdfApi, dev);
        initialize(sdfApi, dev, configPath);
        openSession(sdfApi, dev, ses);
        getDeviceInfo(sdfApi, dev, ses);
    }

    public static void checkRet(String method, int ret, long dev, long ses) throws SdfSDKException {
        if (ret != 0) {
            logger.error(
                    "{} error! dev = {} session = {} ret = {} HEX={}:{}",
                    method,
                    dev,
                    ses,
                    ret,
                    Integer.toHexString(ret),
                    SdfApiCode.apiCodeToString(ret)
            );
            throw new SdfSDKException(method, ret);
        } else {
//            logger.info(method + " success! dev = " + connection.getDev()[0] + " session = " + connection.getSes()[0]);
        }
    }

    //打开设备
    public static void openDevice(SdfApi sdfApi, long[] dev) throws SdfSDKException {
        int ret = sdfApi.openDevice(dev);
        checkRet("openDevice", ret, dev[0], 0);
    }

    //使用配置文件初始化设备
    public static void initialize(SdfApi sdfApi, long[] dev, String configPath) throws SdfSDKException {
        int ret = sdfApi.initialize(dev[0], configPath.getBytes());
        checkRet("initialize", ret, dev[0], 0);
    }

    //打开 session
    public static void openSession(SdfApi sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        int ret = sdfApi.openSession(dev[0], ses);
        checkRet("openSession", ret, dev[0], ses[0]);
    }

    public static void closeSession(SdfApi sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        int ret = sdfApi.closeSession(ses[0]);
        checkRet("closeSession", ret, dev[0], ses[0]);
    }

    public static void closeDevice(SdfApi sdfApi, long[] dev) throws SdfSDKException {
        int ret = sdfApi.closeDevice(dev[0]);
        checkRet("closeDevice", ret, dev[0], 0);
    }

    public static String getDeviceInfo(SdfApi sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        DeviceInfo deviceInfo = new DeviceInfo();
        int ret = sdfApi.getDeviceInfo(ses[0], deviceInfo);
        checkRet("getDeviceInfo", ret, dev[0], ses[0]);
        return deviceInfo.toString();
    }

    /************************************************************************************************
     * ECC SDF to hsm sdk
     *************************************************************************************************/
    public static EccPublicKey generateEccPublicKey(Logger logger, SdfECCPublicKey key) {
        return generateEccPublicKey(logger, key.getX(), key.getY());
    }

    public static EccPublicKey generateEccPublicKey(Logger logger, byte[] x, byte[] y) {
//        logger.info("generateEccPublicKey");

        EccPublicKey pucPublicKey = new EccPublicKey(
                GMSSLX509Utils.ECC_KEY_BITS,
                GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(x),
                GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(y)
        );

//        GMSSLByteArrayUtils.printHexBinary(logger, "EccPublicKey x", pucPublicKey.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "EccPublicKey y", pucPublicKey.getY());
        return pucPublicKey;
    }

    public static EccPrivateKey generateEccPrivateKey(Logger logger, PrivateKey key) {
        ECPrivateKey ecPrivateKey = (ECPrivateKey) key;
        return generateEccPrivateKey(logger, ecPrivateKey.getS().toByteArray());
    }

    public static EccPrivateKey generateEccPrivateKey(Logger logger, byte[] k) {
        logger.info("generateEccPrivateKey");
        EccPrivateKey pucPrivateKey = new EccPrivateKey(
                GMSSLX509Utils.ECC_KEY_BITS,
                GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(k)
        );
//        GMSSLByteArrayUtils.printHexBinary(logger, "pucPrivateKey k", pucPrivateKey.getK());
        return pucPrivateKey;
    }

    /************************************************************************************************
     * RSA SDF to hsm sdk
     *************************************************************************************************/
    public static RsaPublicKey generateRsaPublicKey(Logger logger, SdfRSAPublicKey key) {
        byte[] m = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(key.getM(), 256);
        byte[] e = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(key.getE(), 256);
        return generateRsaPublicKey(logger, key.getBits(), m, e);
    }

    public static RsaPublicKey generateRsaPublicKey(Logger logger, int bits, byte[] m, byte[] e) {
//        logger.info("generateRsaPublicKey");
        return new RsaPublicKey(bits, m, e);
    }

}
