package com.xdja.pki.gmssl.test;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSAKeyUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCRLUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCertUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLExtensionUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;


public class GMSSLKeyStore {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    private char[] password;

    private KeyStore keyStore;
    private KeyStore trustKeyStore;

    private PrivateKey rootKey;
    private X509Certificate rootCert;
    private PrivateKey signKey;
    private X509Certificate signCert;
    private PrivateKey encKey;
    private X509Certificate encCert;
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    public GMSSLKeyStore(
            String keyStoreType, String password,
            PrivateKey rootKey, X509Certificate rootCert,
            PrivateKey signKey, X509Certificate signCert,
            PrivateKey encKey, X509Certificate encCert
    ) throws NoSuchAlgorithmException, CertificateException, NoSuchProviderException, KeyStoreException, IOException {
        this(keyStoreType, keyStoreType, password, rootKey, rootCert, signKey, signCert, encKey, encCert);
    }

    public GMSSLKeyStore(
            String keyStoreType, String trustStoreType, String password,
            PrivateKey rootKey, X509Certificate rootCert,
            PrivateKey signKey, X509Certificate signCert,
            PrivateKey encKey, X509Certificate encCert
    ) throws NoSuchAlgorithmException, CertificateException, NoSuchProviderException, KeyStoreException, IOException {
        this.password = password.toCharArray();
        this.rootKey = rootKey;
        this.rootCert = rootCert;
        this.signKey = signKey;
        this.signCert = signCert;
        this.encKey = encKey;
        this.encCert = encCert;
        this.keyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStoreWithType(password, rootCert, "sign", signKey, signCert, "enc", encKey, encCert, keyStoreType);
        this.trustKeyStore = GMSSLKeyStoreUtils.generateGMSSLTrustStoreWithType(rootCert, trustStoreType);
    }

    public char[] getPassword() {
        return password;
    }

    public PrivateKey getRootKey() {
        return rootKey;
    }

    public X509Certificate getRootCert() {
        return rootCert;
    }

    public PrivateKey getSignKey() {
        return signKey;
    }

    public X509Certificate getSignCert() {
        return signCert;
    }

    public PrivateKey getEncKey() {
        return encKey;
    }

    public X509Certificate getEncCert() {
        return encCert;
    }

    public KeyStore getKeyStore() {
        return keyStore;
    }

    public void setKeyStore(KeyStore keyStore) {
        this.keyStore = keyStore;
    }

    public KeyStore getTrustKeyStore() {
        return trustKeyStore;
    }

    public void setTrustKeyStore(KeyStore trustKeyStore) {
        this.trustKeyStore = trustKeyStore;
    }

    public void save() throws Exception {
        save(null);
    }

    public void save(String path) throws Exception {
        if (path == null) {
            path = "." + File.separator + "target" + File.separator + "tomcat" + File.separator + "cert";
        }
        File pathf = new File(path);
        if (!pathf.exists()) {
            pathf.mkdirs();
        }
        String url = pathf.getAbsolutePath() + File.separator;
        logger.debug("keystore save absolute path is {}", url);

        if (this.keyStore != null) {
            FileOutputStream outputStream = new FileOutputStream(url + "keystore." + keyStore.getType().toLowerCase());
            this.keyStore.store(outputStream, this.getPassword());
            outputStream.close();
        }

        if (this.trustKeyStore != null) {
            FileOutputStream outputStream = new FileOutputStream(url + "trust." + trustKeyStore.getType().toLowerCase());
            this.trustKeyStore.store(outputStream, this.getPassword());
            outputStream.close();
        }
    }

    public static GMSSLKeyStore generateGMSSLKeyStoreByBC(String keyStoreType) throws Exception {
        return generateGMSSLKeyStoreByBC(keyStoreType, keyStoreType);
    }

    public static GMSSLKeyStore generateGMSSLKeyStoreByBC(String keyStoreType, String trustStroeType) throws Exception {
        String path = "target\\tomcat\\";
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rootDN = "cn=sub,o=xdja,c=cn";
        KeyPair rootKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());

        System.out.println(rootKeyPair.getPublic());

        X509Certificate rootCert = GMSSLCertUtils.generateCertByBC(
                rootDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                rootKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                rootExtensions
        );
        GMSSLX509Utils.printPEMObject("sub", rootCert);
        GMSSLX509Utils.writeCertificateToCer(path, "sub", rootCert);

        //生成 server 密钥、证书
        String signDN = "cn=sign,o=xdja,c=cn";
        KeyPair signKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());
        signExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(signKeyPair.getPublic()));
        signExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        signExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        signExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate signCert = GMSSLCertUtils.generateCertByBC(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                signKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                signExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "signCert", signCert);

        String encDN = "cn=enc,o=xdja,c=cn";
        KeyPair encKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());
        encExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(encKeyPair.getPublic()));
        encExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        encExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        encExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate encCert = GMSSLCertUtils.generateCertByBC(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                encKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                encExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "encCert", encCert);
        return new GMSSLKeyStore(keyStoreType, trustStroeType, "password", rootKeyPair.getPrivate(), rootCert, signKeyPair.getPrivate(), signCert, encKeyPair.getPrivate(), encCert);
    }

    public static GMSSLKeyStore generateGMSSLKeyStoreByBCWithRSA() throws Exception {
        return generateGMSSLKeyStoreByBCWithRSA(2048, GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName());
    }

    public static GMSSLKeyStore generateGMSSLKeyStoreByBCWithRSA(int keySize, String signName) throws Exception {
        String path = "target\\rsa\\";
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rootDN = "cn=root,o=xdja,c=cn";
        KeyPair rootKeyPair = GMSSLRSAKeyUtils.generateKeyPairByBC(keySize);
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());

        System.out.println(rootKeyPair.getPublic());

        X509Certificate rootCert = GMSSLCertUtils.generateCertByBC(
                rootDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                rootKeyPair.getPublic(),
                signName,
                rootExtensions
        );
        GMSSLX509Utils.printPEMObject("sub", rootCert);
        GMSSLX509Utils.writeCertificateToCer(path, "sub", rootCert);

        //生成 server 密钥、证书
        String signDN = "cn=client,o=xdja,c=cn";
        KeyPair signKeyPair = GMSSLRSAKeyUtils.generateKeyPairByBC(keySize);
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());
        signExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(signKeyPair.getPublic()));
        signExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        signExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        signExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate signCert = GMSSLCertUtils.generateCertByBC(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                signKeyPair.getPublic(),
                signName,
                signExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "signCert", signCert);
        String encDN = "cn=11.12.86.53,o=xdja,c=cn";
        KeyPair encKeyPair = GMSSLRSAKeyUtils.generateKeyPairByBC(keySize);
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());
        encExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(encKeyPair.getPublic()));
        encExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        encExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        encExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate encCert = GMSSLCertUtils.generateCertByBC(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                encKeyPair.getPublic(),
                signName,
                encExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "encCert", encCert);
        return new GMSSLKeyStore(GMSSLKeyStoreUtils.JKS_KEYSTORE_TYPE, "password", rootKeyPair.getPrivate(), rootCert, signKeyPair.getPrivate(), signCert, encKeyPair.getPrivate(), encCert);
    }


    public static GMSSLKeyStore generateGMSSLKeyStoreByBCWithSubCA(String keyStoreType) throws Exception {
        String path = "target\\";
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rDN = "cn=root,o=xdja,c=cn";
        KeyPair rKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> rExtensions = new ArrayList<>();
        rExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());
        rExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(rKeyPair.getPublic()));
//        rExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        rExtensions.add(GMSSLExtensionUtils.genBasicConstraintsExtension(null));
        X509Certificate rCert = GMSSLCertUtils.generateCertByBC(
                rDN, rDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rKeyPair.getPrivate(),
                rKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                rExtensions
        );
        GMSSLX509Utils.printPEMObject("root", rCert);
        GMSSLX509Utils.writeCertificateToCer(path, "root", rCert);

        X509CRL x509CRL = GMSSLCRLUtils.generateCRLByBC(
                rCert, rKeyPair.getPrivate(), GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                now, noAfter, null,
                null
        );
        GMSSLX509Utils.printPEMObject("crl", x509CRL);

        String rootDN = "cn=sub,o=xdja,c=cn";
        KeyPair rootKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());
        rootExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(rootKeyPair.getPublic()));
        rootExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rCert));
        rootExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://11.12.110.92:389/CN=caroot,OU=ll,C=cn?userCertificate;binary", "http://11.12.110.90:8081/ocsp-web/ocsp/v1"));
        rootExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://11.12.110.92:389/CN=withto,OU=baba,C=cn,OU=ll,C=cn?userCertificate;binary", "http://11.12.110.90:8081/ocsp-web/ocsp/v1"));
        rootExtensions.add(GMSSLExtensionUtils.genBasicConstraintsExtension(2));

        Map<Integer, String> map = new HashMap<>();
        map.put(GeneralName.iPAddress, "1.1.1.1");
        Extension subjectAlternativeName = GMSSLExtensionUtils.genSubjectAlternativeName(false, map);
        rootExtensions.add(subjectAlternativeName);

        Map<Integer, String> permittedValues = new HashMap<>();
        permittedValues.put(GeneralName.iPAddress, "1.1.1.1");
        Map<Integer, String> excludedValues = new HashMap<>();
        excludedValues.put(GeneralName.directoryName, "OU=test");
        Extension nameConstraints = GMSSLExtensionUtils.genNameConstraints(false, permittedValues, excludedValues);
        rootExtensions.add(nameConstraints);

        Extension policyConstraints = GMSSLExtensionUtils.genPolicyConstraints(true, "1", "1");
        rootExtensions.add(policyConstraints);

        Extension crlDistributionPointsExtension = GMSSLExtensionUtils.genCRLDistributionPointsExtension("ldap://11.12.85.57:389/cn=crl0,o=RootCRL,o=xdja,c=cn?certificateRevocationList;binary");
        rootExtensions.add(crlDistributionPointsExtension);

        Extension freshestCRL = GMSSLExtensionUtils.genFreshestCRL("ldap://11.12.85.57:389/cn=crl0,o=RootCRL,o=xdja,c=cn?deltaRevocationList;binary");
        rootExtensions.add(freshestCRL);

        X509Certificate rootCert = GMSSLCertUtils.generateCertByBC(
                rDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rKeyPair.getPrivate(),
                rootKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                rootExtensions
        );
        GMSSLX509Utils.printPEMObject("sub", rootCert);
        GMSSLX509Utils.writeCertificateToCer(path, "sub", rootCert);

        //生成 server 密钥、证书
        String signDN = "cn=sign,o=xdja,c=cn";
        KeyPair signKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());
        signExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(signKeyPair.getPublic()));
        signExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        signExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        signExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate signCert = GMSSLCertUtils.generateCertByBC(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                signKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                signExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "signCert", signCert);

        String encDN = "cn=enc,o=xdja,c=cn";
        KeyPair encKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());
        encExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(encKeyPair.getPublic()));
        encExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
//        encExtensions.add(GMSSLExtensionUtils.genAuthorityInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
//        encExtensions.add(GMSSLExtensionUtils.genSubjectInfoAccessExtension("ldap://ldapurl", "http://ocspurl"));
        X509Certificate encCert = GMSSLCertUtils.generateCertByBC(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                encKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                encExtensions
        );
        GMSSLX509Utils.writeCertificateToCer(path, "encCert", encCert);
        return new GMSSLKeyStore(keyStoreType, "password", rootKeyPair.getPrivate(), rootCert, signKeyPair.getPrivate(), signCert, encKeyPair.getPrivate(), encCert);
    }


    public static GMSSLKeyStore generateGMSSLKeyStoreByYunhsm(int privateKeyIndex, String privateKeyPassword, String keyStoreType) throws Exception {
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rootDN = "CN=ROOTCA,O=XDJA,C=CN";
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());

        PublicKey rootPublicKey = GMSSLSM2KeyUtils.getSignPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey rootPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate rootCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                rootPublicKey,
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                rootExtensions
        );

        //生成 server 密钥、证书
        String signDN = "CN=SIGN,O=SERVER,O=XDJA,C=CN";
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());

        PublicKey signPublicKey = GMSSLSM2KeyUtils.getSignPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey signPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate signCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                signPublicKey,
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                signExtensions
        );

        String encDN = "CN=ENC,O=SERVER,O=XDJA,C=CN";
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());

        PublicKey encPublicKey = GMSSLSM2KeyUtils.getEncryptPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey encPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate encCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                encPublicKey,
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                encExtensions
        );

        return new GMSSLKeyStore(keyStoreType, "password", rootPrivateKey, rootCert, signPrivateKey, signCert, encPrivateKey, encCert);
    }
}
