package com.xdja.pki.oer.gbt.asn1;

import com.xdja.pki.oer.base.BitByte;
import com.xdja.pki.oer.base.OctetString;
import com.xdja.pki.oer.base.Sequence;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

/**
 * 该附录定义了证书请求内包含数据的具体结构。
 * CertRequest ::= SEQUENCE {
 * version  Uint8, 2
 * currentTime Time32,
 * tbsCertData TbsCert,
 * encryptionKey PublicEncryptionKey OPTIONAL
 * * }
 * version ：证书请求版本号。Uint8定义参考《GB/TXXXXX—XXXX：交通运输  数字证书应用接口规范》。
 * Time32：当前请求时间。Time32定义参考《GB/TXXXXX—XXXX：交通运输  数字证书应用接口规范》。
 * tbsCertData：证书请求的具体内容。TbsCert定义参考《GB/TXXXXX—XXXX：交通运输  数字证书应用接口规范》。
 * encryptionKey:证书请求应答加密公钥，
 * PublicEncryptionKey定义参考《GB/TXXXXX—XXXX：交通运输  数字证书应用接口规范》。
 * 当该项存在时，其返回的证书请求应答为加密数据。
 * <p>
 * 1、	EnrollmentCertRequestData -> payload -> signedData -> tbs -> data-CertRequest内，
 * 扩展一个itsId字段：基于公认公钥申请第一张注册证书时，设该值为设备唯一ID；
 * 基于当前有效注册证书申请新的注册证书时，该值设为当前有效注册证书的HashedId8。
 */
public class CertRequest extends Sequence {

    private static Logger logger = LoggerFactory.getLogger(CertRequest.class);
    private Time32 currentTime;
    private TbsCert tbsCertData;
    private PublicEncryptionKey encryptionKey;
    private OctetString itsId;

    public CertRequest() {
        super(false, true);
    }

    public static CertRequest getInstance(byte[] data) throws Exception {
        //ByteArrayUtils.printHexBinary(logger, "CertRequest start data", data);
        BigInteger optional = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        BitByte bitByte = BitByte.setBit(optional.intValue());
        //去version 和 可选
        byte[] goal = new byte[data.length - 1 - 1];
        System.arraycopy(data, 1 + 1, goal, 0, goal.length);
        data = goal;
        List<Integer> integers = bitByte.readIndexes();
        CertRequest certRequest = new CertRequest();
        //Time 32
        BigInteger time = BigIntegers.fromUnsignedByteArray(data, 0, 4);
        Time32 currentTime = new Time32(time.longValue());
        //ByteArrayUtils.printHexBinary(logger, "CertRequest currentTime data", currentTime.getEncode());
        goal = new byte[data.length - 4];
        System.arraycopy(data, 4, goal, 0, goal.length);
        data = goal;
        certRequest.setCurrentTime(currentTime);
        TbsCert tbsCert = TbsCert.getInstance(data);
        data = tbsCert.getGoal();
        certRequest.setTbsCertData(tbsCert);
        //ByteArrayUtils.printHexBinary(logger, "CertRequest tbsCert data", tbsCert.getEncode());
        if (integers.contains(0)) {
            PublicEncryptionKey encryptionKey = PublicEncryptionKey.getInstance(data);
            certRequest.setEncryptionKey(encryptionKey);
            data = encryptionKey.getGoal();
        }
        //ByteArrayUtils.printHexBinary(logger, "CertRequest tbsCert lave data", data);
        if (integers.contains(1)){
            OctetString itsId = OctetString.getInstance(data);
            //ByteArrayUtils.printHexBinary(logger, "CertRequest itsId data", itsId.getEncode());
            data = itsId.getGoal();
            certRequest.setItsId(itsId);
        }
        //ByteArrayUtils.printHexBinary(logger, "CertRequest lave data", data);
        certRequest.setGoal(data);
        return certRequest;
    }

    public OctetString getItsId() {
        return itsId;
    }

    public void setItsId(OctetString itsId) {
        this.itsId = itsId;
    }

    public void setItsId(byte[] itsId) {
        this.itsId = new OctetString();
        this.itsId.setString(itsId);
    }

    public Time32 getCurrentTime() {
        return currentTime;
    }

    public void setCurrentTime(Time32 currentTime) {
        this.currentTime = currentTime;
    }

    public TbsCert getTbsCertData() {
        return tbsCertData;
    }

    public void setTbsCertData(TbsCert tbsCertData) {
        this.tbsCertData = tbsCertData;
    }

    public PublicEncryptionKey getEncryptionKey() {
        return encryptionKey;
    }


    public void setEncryptionKey(PublicEncryptionKey encryptionKey) {
        this.encryptionKey = encryptionKey;
    }

    @Override
    public Vector getSequenceValues() {
        Vector vector = new Vector();
        List<Integer> optionals = new ArrayList<>();
        if (null != this.encryptionKey) {
            optionals.add(7);
        }
        if (null != this.itsId) {
            optionals.add(6);
        }
        if (optionals.size() > 0) {
            this.addOptional(optionals);
        }
        vector.add(2);
        vector.add(currentTime);
        vector.add(tbsCertData);
        vector.add(encryptionKey);
        vector.add(itsId);
        return vector;
    }
}
