package com.xdja.pki.oer.gbt.asn1;

import com.xdja.pki.oer.base.Sequence;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.util.Vector;

/**
 * --  公钥加密的对称密钥密文
 * --  说明 hashAlg 用于计算recipientId的摘要算法
 * --  recipientId 非对称加密的公钥所述内容的hash。当公钥来自证书，则为证书的hash,当公钥来自签名消息中的encKey，则为SignedData的hash
 * --  kek  用接受者公钥加密的对称密钥
 * PKRecipientInfo ::= SEQUENCE{
 * hashAlg     HashAlgorithm,
 * recipientId HashedId8,
 * kek         EciesEncryptedKey
 * };
 */
public class PKRecipientInfo extends Sequence {
    private static Logger logger = LoggerFactory.getLogger(PKRecipientInfo.class);
    private HashAlgorithm hashAlg;
    private HashedId8 recipientId;
    private EciesEncryptedKey kek;

    public PKRecipientInfo() {
        super(false, false);
    }

    public static PKRecipientInfo getInstance(byte[] data) throws Exception {
        PKRecipientInfo pkRecipientInfo = new PKRecipientInfo();
        //ByteArrayUtils.printHexBinary(logger, "PKRecipientInfo  start data ", data);
        BigInteger hashAlgorithm = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        if (hashAlgorithm.intValue() == HashAlgorithm.SGD_SM3.getIndex()) {
            pkRecipientInfo.setHashAlg(new HashAlgorithm(HashAlgorithm.SGD_SM3));
        } else if (hashAlgorithm.intValue() == HashAlgorithm.SHA_256.getIndex()) {
            pkRecipientInfo.setHashAlg(new HashAlgorithm(HashAlgorithm.SHA_256));
        } else {
            throw new Exception("unsupported hashAlgorithm type " + hashAlgorithm.intValue());
        }
        byte[] hashedId8 = new byte[8];
        System.arraycopy(data, 1, hashedId8, 0, hashedId8.length);
        pkRecipientInfo.setRecipientId(new HashedId8(hashedId8));
        //ByteArrayUtils.printHexBinary(logger, "PKRecipientInfo HashedId8 data ", hashedId8);

        byte[] goal = new byte[data.length - 9];
        System.arraycopy(data, 9, goal, 0, goal.length);
        EciesEncryptedKey instance = EciesEncryptedKey.getInstance(goal);
        //ByteArrayUtils.printHexBinary(logger, "PKRecipientInfo EciesEncryptedKey data ", instance.getEncode());
        pkRecipientInfo.setKek(instance);
        pkRecipientInfo.setGoal(instance.getGoal());
        //ByteArrayUtils.printHexBinary(logger, "PKRecipientInfo lave data ", pkRecipientInfo.getGoal());
        return pkRecipientInfo;
    }

    public void setHashAlg(HashAlgorithm hashAlg) {
        this.hashAlg = hashAlg;
    }

    public void setRecipientId(HashedId8 recipientId) {
        this.recipientId = recipientId;
    }

    public void setKek(EciesEncryptedKey kek) {
        this.kek = kek;
    }

    public HashAlgorithm getHashAlg() {
        return hashAlg;
    }

    public HashedId8 getRecipientId() {
        return recipientId;
    }

    public EciesEncryptedKey getKek() {
        return kek;
    }


    @Override
    public Vector getSequenceValues() {
        Vector vector = new Vector();
        vector.add(hashAlg);
        vector.add(recipientId);
        vector.add(kek);
        return vector;
    }
}
