package com.xdja.pki.oer.gbt.asn1;

import com.xdja.pki.oer.base.Choice;
import com.xdja.pki.oer.base.OctetString;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.util.Vector;

/**
 * --  6.3.2消息载荷
 * --  6.3.2.1 结构说明
 * --  unSecuredData:未签名、未加密的内容
 * --  signedData  签名负载
 * --  encData     加密负载
 * Payload ::= CHOICE{
 * unSecuredData   Opaque,
 * signedData      SignedData,
 * encData         EncryptedData
 * }
 */
public class Payload extends Choice {
    private static Logger logger = LoggerFactory.getLogger(Payload.class);
    private Opaque unSecuredData;

    private SignedData signedData;

    private EncryptedData encData;

    public Payload(Opaque opaque) {
        this.unSecuredData = opaque;
        this.signedData = null;
        this.encData = null;
    }

    public Payload(SignedData signedData) {
        this.addIndex(1);
        this.unSecuredData = null;
        this.signedData = signedData;
        this.encData = null;
    }

    public Payload(EncryptedData encData) {
        this.addIndex(2);
        this.unSecuredData = null;
        this.signedData = null;
        this.encData = encData;
    }


    public Opaque getUnSecuredData() {
        return unSecuredData;
    }

    public SignedData getSignedData() {
        return signedData;
    }

    public EncryptedData getEncData() {
        return encData;
    }

    public static Payload getInstance(byte[] data) throws Exception {
        //ByteArrayUtils.printHexBinary(logger, "Payload start data ", data);
        BigInteger choice = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        byte[] goal = new byte[data.length - 1];
        System.arraycopy(data, 1, goal, 0, goal.length);
        Payload payload;
        if (choice.intValue() - 0x80 == 0) {
            //logger.debug("Payload choice unSecuredData");
            OctetString instance = Opaque.getInstance(data);
            Opaque opaque = new Opaque();
            opaque.setString(instance.getString());
            opaque.setGoal(instance.getGoal());
            //ByteArrayUtils.printHexBinary(logger, "Payload opaque data ", opaque.getEncode());
            payload = new Payload(opaque);
            payload.setGoal(opaque.getGoal());
        } else if (choice.intValue() - 0x80 == 1) {
            //logger.debug("Payload choice signedData");
            SignedData signedData = SignedData.getInstance(goal);
            payload = new Payload(signedData);
            payload.setGoal(signedData.getGoal());
            //ByteArrayUtils.printHexBinary(logger, "Payload signedData data ", signedData.getEncode());
        } else if (choice.intValue() - 0x80 == 2) {
            //logger.debug("Payload choice encryptedData");
            EncryptedData encryptedData = EncryptedData.getInstance(goal);
            payload = new Payload(encryptedData);
            payload.setGoal(encryptedData.getGoal());
            //ByteArrayUtils.printHexBinary(logger, "Payload encryptedData data ", encryptedData.getEncode());
        } else {
            //logger.debug("Payload choice is error " + choice.intValue());
            throw new Exception("unsupported Payload type " + choice.intValue());
        }
        //ByteArrayUtils.printHexBinary(logger, "Payload lave data ", payload.getGoal());
        return payload;
    }

    @Override
    public Vector getChoiceValues() {
        Vector vector = new Vector();
        if (null != this.unSecuredData) {
            // 0*80 不加 第一个
            vector.add(unSecuredData);
        }
        if (null != this.signedData) {

            vector.add(signedData);
        }
        if (null != this.encData) {

            vector.add(encData);
        }
        return vector;
    }
}
