package com.xdja.pki.oer.gbt.asn1;


import com.xdja.pki.oer.base.Choice;
import com.xdja.pki.oer.gbt.asn1.bean.PKRecipientInfoType;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.util.Vector;

/**
 * -- 6.2.11 接受者消息
 * RecipientInfo ::=  CHOICE{
 * pskRecipientInfo    PreSharedKeyRecipientInfo,
 * symmRecipInfo       SymmRecipInfo,
 * certRecipInfo       PKRecipientInfo,
 * signedDataRecipInfo PKRecipientInfo
 * };
 * <p>
 * -- 说明 PreSharedKeyRecipientInfo 预共享的对称密钥；
 * --      hashAlg 计算smyKeyHash 的摘要算法
 * PreSharedKeyRecipientInfo ::= SEQUENCE{
 * hashAlg     HashAlgorithm,
 * smyKeyHash  HashedId8,
 * };
 * <p>
 * -- 说明 hashAlg 用于计算recipientId的摘要算法
 * -- recipientId 用于解密encKey的对称密钥的摘要
 * -- encKey       用于数据加密的对称密钥，被其他对称密钥加密过的密文
 * SymmRecipInfo ::= SEQUENCE{
 * hashAlg     HashAlgorithm,
 * recipientId HashedId8,
 * encKey      SymmetricCipherText
 * };
 * <p>
 * --  公钥加密的对称密钥密文
 * PKRecipientInfo ::= SEQUENCE{
 * hashAlg     HashAlgorithm,
 * recipientId HashedId8,
 * kek         EciesEncryptedKey
 * };
 */
public class RecipientInfo extends Choice {
    private static Logger logger = LoggerFactory.getLogger(RecipientInfo.class);
    private PreSharedKeyRecipientInfo pskRecipientInfo;
    private SymmRecipInfo symmRecipInfo;
    private PKRecipientInfo certRecipInfo;
    private PKRecipientInfo signedDataRecipInfo;

    public RecipientInfo(PreSharedKeyRecipientInfo pskRecipientInfo) {
        this.pskRecipientInfo = pskRecipientInfo;
        this.symmRecipInfo = null;
        this.certRecipInfo = null;
        this.signedDataRecipInfo = null;
    }


    public RecipientInfo(SymmRecipInfo symmRecipInfo) {
        this.addIndex(1);
        this.pskRecipientInfo = null;
        this.symmRecipInfo = symmRecipInfo;
        this.certRecipInfo = null;
        this.signedDataRecipInfo = null;
    }


    public RecipientInfo(PKRecipientInfo info, PKRecipientInfoType pkRecipientInfoType) {
        this.pskRecipientInfo = null;
        this.symmRecipInfo = null;
        this.certRecipInfo = null;
        this.signedDataRecipInfo = null;
        if (pkRecipientInfoType.id == PKRecipientInfoType.CERT_RECIPINFO.id) {
            this.addIndex(2);
            this.certRecipInfo = info;
        }
        if (pkRecipientInfoType.id == PKRecipientInfoType.SIGNEDDATA_RECIPINFO.id) {
            this.addIndex(3);
            this.signedDataRecipInfo = info;
        }
    }

    public static RecipientInfo getInstance(byte[] data) throws Exception {
        //ByteArrayUtils.printHexBinary(logger, "RecipientInfo start data ", data);
        RecipientInfo recipientInfo;
        BigInteger choice = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        byte[] goal = new byte[data.length - 1];
        System.arraycopy(data, 1, goal, 0, goal.length);
        data = goal;
        if (choice.intValue() - 0x80 == 0) {
            //logger.debug("RecipientInfo choice pskRecipientInfo");
            PreSharedKeyRecipientInfo info = PreSharedKeyRecipientInfo.getInstance(data);
            //ByteArrayUtils.printHexBinary(logger, "RecipientInfo pskRecipientInfo data ", info.getEncode());
            recipientInfo = new RecipientInfo(info);
            recipientInfo.setGoal(info.getGoal());
        } else if (choice.intValue() - 0x80 == 1) {
            //logger.debug("RecipientInfo choice symmRecipInfo");
            SymmRecipInfo symmRecipInfo = SymmRecipInfo.getInstance(data);
            //ByteArrayUtils.printHexBinary(logger, "RecipientInfo symmRecipInfo data ", symmRecipInfo.getEncode());
            recipientInfo = new RecipientInfo(symmRecipInfo);
            recipientInfo.setGoal(symmRecipInfo.getGoal());
        } else if (choice.intValue() - 0x80 == 2) {
            //logger.debug("RecipientInfo choice certRecipInfo");
            PKRecipientInfo instance = PKRecipientInfo.getInstance(data);
            //ByteArrayUtils.printHexBinary(logger, "RecipientInfo certRecipInfo data ", instance.getEncode());
            recipientInfo = new RecipientInfo(instance, PKRecipientInfoType.CERT_RECIPINFO);
            recipientInfo.setGoal(instance.getGoal());
        } else if (choice.intValue() - 0x80 == 3) {
            //logger.debug("RecipientInfo choice signedDataRecipInfo");
            PKRecipientInfo instance = PKRecipientInfo.getInstance(data);
            //ByteArrayUtils.printHexBinary(logger, "RecipientInfo signedDataRecipInfo data ", instance.getEncode());
            recipientInfo = new RecipientInfo(instance, PKRecipientInfoType.SIGNEDDATA_RECIPINFO);
            recipientInfo.setGoal(instance.getGoal());
        } else {
            logger.error("RecipientInfo choice is error " + choice.intValue());
            throw new Exception("unsupported RecipientInfo type " + choice.intValue());
        }
        return recipientInfo;
    }

    public PreSharedKeyRecipientInfo getPskRecipientInfo() {
        return pskRecipientInfo;
    }

    public SymmRecipInfo getSymmRecipInfo() {
        return symmRecipInfo;
    }

    public PKRecipientInfo getCertRecipInfo() {
        return certRecipInfo;
    }

    public PKRecipientInfo getSignedDataRecipInfo() {
        return signedDataRecipInfo;
    }

    @Override
    public Vector getChoiceValues() {
        Vector vector = new Vector();
        if (null != this.pskRecipientInfo) {
            vector.add(pskRecipientInfo);
        }
        if (null != this.symmRecipInfo) {
            vector.add(symmRecipInfo);
        }
        if (null != this.certRecipInfo) {
            vector.add(certRecipInfo);
        }
        if (null != this.signedDataRecipInfo) {
            vector.add(signedDataRecipInfo);
        }
        return vector;
    }
}
