package com.xdja.pki.oer.gbt.asn1.data;

import com.xdja.pki.oer.base.Int;
import com.xdja.pki.oer.core.TimeUtils;
import com.xdja.pki.oer.core.calculate.CalculateFactory;
import com.xdja.pki.oer.core.calculate.CalculateService;
import com.xdja.pki.oer.gbt.asn1.*;
import com.xdja.pki.oer.gbt.asn1.utils.SignatureBuilder;

import java.security.PrivateKey;

/***
 *--- 4. SignedExternalPayload
 * SignedExternalPayload ::= SecuredMessage
 * SecuredMessage ::= SEQUENCE{
 *     version  Uint8,
 *     payload  Payload
 * }
 * Payload ::= CHOICE{
 *     unSecuredData   Opaque,
 *     signedData      SignedData,  --- 签名消息体
 *     encData         EncryptedData
 * }
 * SignedData ::= SEQUENCE{
 *     signer  SignerInfo,  ---签名者为 “OBU 设备注册证书”
 *     tbs     TBSData,    ---授权证书申请 hash
 *     sign    Signature   ---使用 “OBU 设备注册证书” 对应私钥进行签名 原文为 “注册证书 hash + tbsData hash”
 * }
 * TBSData ::= SEQUENCE{
 *     headerInfo  HeaderInfo,
 *     data        OCTET STRING(SIZE(0..MAX))  Optional,
 *     extHash     OCTET STRING(SIZE(32))  Optional,  ---为 SharedATRequest 的hash值
 * }
 */
public class SignedExternalPayloadBuilder {
    private static CalculateService calculateService = CalculateFactory.getInstance();

    /**
     * @param sharedATRequest
     * @param ecCert          注册证书
     * @param ecPrivateKey    注册证书签名私钥
     */
    public static SecuredMessage builder(SharedATRequest sharedATRequest, Certificate ecCert, PrivateKey ecPrivateKey) throws Exception {
        SignedData signedData = new SignedData();
        //拼凑SignInfo
        SequenceOfCertificate sequenceOfCertificate = new SequenceOfCertificate();
        sequenceOfCertificate.addCertificate(ecCert);
        SignerInfo signerInfo = new SignerInfo(sequenceOfCertificate);
        signedData.setSignerInfo(signerInfo);
        //TBSData
        TBSData tbsData = new TBSData();
        HeaderInfo headerInfo = new HeaderInfo();
        headerInfo.setItsAid(new Int(40965));
        headerInfo.setGenTime(new Time64(TimeUtils.getNowTime() * 1000L));
        tbsData.setHeaderInfo(headerInfo);
        //byte[] bytes = BCUtils.sm3DigestByBC(sharedATRequest.getEncode());
        byte[] bytes = calculateService.sm3Hash(sharedATRequest.getEncode());
        tbsData.setExtHash(bytes);
        signedData.setTbs(tbsData);
        //Signature
        Signature signature = SignatureBuilder.build(ecPrivateKey, tbsData);
        signedData.setSign(signature);
        SecuredMessage securedMessage = new SecuredMessage();
        Payload payload = new Payload(signedData);
        securedMessage.setPayload(payload);
        return securedMessage;
    }
}
