package com.xdja.pki.oer.gbt.asn1.utils;

import com.xdja.pki.oer.base.Null;
import com.xdja.pki.oer.core.ByteArrayUtils;
import com.xdja.pki.oer.core.calculate.CalculateFactory;
import com.xdja.pki.oer.core.calculate.CalculateService;
import com.xdja.pki.oer.gbt.asn1.*;
import com.xdja.pki.oer.gbt.asn1.utils.enums.CertHashTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.EccCurveTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.EccPointTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.SubjectTypeEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.PublicKey;

public class CertificateBuilder {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private CalculateService calculateService = CalculateFactory.getInstance();
    private SequenceOfItsAidList itsAidList;
    private SequenceOfItsAidSspList itsAidSspList;

    private GeographicRegion geographicRegion;
    private Time32 startTime;
    private Time32 endTime;
    private EccPointTypeEnum signKeyEccPointType = EccPointTypeEnum.UNCOMPRESSED;
    private EccPointTypeEnum encKeyEccPointType = EccPointTypeEnum.UNCOMPRESSED;
    private EccCurveTypeEnum eccCurveTypeEnum;

    public void setSignKeyEccPointType(EccPointTypeEnum signKeyEccPointType) {
        this.signKeyEccPointType = signKeyEccPointType;
    }

    public void setEncKeyEccPointType(EccPointTypeEnum encKeyEccPointType) {
        this.encKeyEccPointType = encKeyEccPointType;
    }

    public void setEccCurveTypeEnum(EccCurveTypeEnum eccCurveTypeEnum) {
        this.eccCurveTypeEnum = eccCurveTypeEnum;
    }

    public void setStartTime(Time32 startTime) {
        this.startTime = startTime;
    }

    public void setEndTime(Time32 endTime) {
        this.endTime = endTime;
    }

    public void setItsAidList(SequenceOfItsAidList itsAidList) {
        this.itsAidList = itsAidList;
    }

    public void setItsAidSspList(SequenceOfItsAidSspList itsAidSspList) {
        this.itsAidSspList = itsAidSspList;
    }

    public void setGeographicRegion(GeographicRegion geographicRegion) {
        this.geographicRegion = geographicRegion;
    }

    public SequenceOfItsAidList getItsAidList() {
        return itsAidList;
    }

    public SequenceOfItsAidSspList getItsAidSspList() {
        return itsAidSspList;
    }

    public GeographicRegion getGeographicRegion() {
        return geographicRegion;
    }

    public Time32 getStartTime() {
        return startTime;
    }

    public Time32 getEndTime() {
        return endTime;
    }

    public EccPointTypeEnum getSignKeyEccPointType() {
        return signKeyEccPointType;
    }

    public EccPointTypeEnum getEncKeyEccPointType() {
        return encKeyEccPointType;
    }

    public EccCurveTypeEnum getEccCurveTypeEnum() {
        return eccCurveTypeEnum;
    }

    /**
     * 构造 四边形区域
     *
     * @param northWestLatitude  北 西 维度
     * @param northWestLongitude 北 西 经度
     * @param southEastLatitude  南 东 纬度
     * @param southEastLongitude 南 东 经度
     * @return 四边形区域信息
     */
    public void setGeographicRegion(
            int northWestLatitude,
            int northWestLongitude,
            int southEastLatitude,
            int southEastLongitude
    ) {
        this.geographicRegion = new GeographicRegion();
        SequenceOfRectangularRegion sequenceOfRectangularRegion = new SequenceOfRectangularRegion();
        RectangularRegionBuilder rectangularRegionBuilder = new RectangularRegionBuilder();
        RectangularRegion rectangularRegion = rectangularRegionBuilder.build(northWestLatitude, northWestLongitude, southEastLatitude, southEastLongitude);
        sequenceOfRectangularRegion.addRectangularRegion(rectangularRegion);
        this.geographicRegion.setRectangularRegion(sequenceOfRectangularRegion);
    }

    public Certificate build(PrivateKey privateKey, PublicKey verifyPublicKey, PublicKey encPublicKey, String subjectName, byte[] issuedHashId, SubjectTypeEnum subjectType) throws Exception {
        return build(privateKey, verifyPublicKey, encPublicKey, subjectName.getBytes(), CertHashTypeEnum.SGD_SM3, issuedHashId, subjectType);
    }

    public Certificate build(PrivateKey privateKey, PublicKey verifyPublicKey, PublicKey encPublicKey, byte[] subjectName, byte[] issuedHashId, SubjectTypeEnum subjectType) throws Exception {
        return build(privateKey, verifyPublicKey, encPublicKey, subjectName, CertHashTypeEnum.SGD_SM3, issuedHashId, subjectType);
    }

    public Certificate build(PrivateKey privateKey, PublicKey verifyPublicKey, PublicKey encPublicKey, String subjectName, CertHashTypeEnum hashType, byte[] issuedHashId, SubjectTypeEnum subjectType) throws Exception {
        return build(privateKey, verifyPublicKey, encPublicKey, subjectName.getBytes(), hashType, issuedHashId, subjectType);

    }

    public Certificate build(PrivateKey privateKey, PublicKey verifyPublicKey, PublicKey encPublicKey, byte[] subjectName, CertHashTypeEnum hashType, byte[] issuedHashId, SubjectTypeEnum subjectType) throws Exception {

        TbsCertBuilder tbsCertBuilder = new TbsCertBuilder();
        if (null != endTime) {
            tbsCertBuilder.setEndTime(endTime);
        }
        if (null != startTime) {
            tbsCertBuilder.setStartTime(startTime);
        }
        if (itsAidList != null) {
            tbsCertBuilder.setItsAidList(itsAidList);
        }
        if (itsAidSspList != null) {
            tbsCertBuilder.setItsAidSspList(itsAidSspList);
        }
        if (geographicRegion != null) {
            tbsCertBuilder.setGeographicRegion(geographicRegion);
        }
        tbsCertBuilder.setSignKeyEccPointType(signKeyEccPointType);
        tbsCertBuilder.setEncKeyEccPointType(encKeyEccPointType);
        tbsCertBuilder.setEccCurveTypeEnum(eccCurveTypeEnum);
        TbsCert tbsCert = tbsCertBuilder.build(verifyPublicKey, encPublicKey, subjectName, subjectType);

        return build(privateKey, hashType, issuedHashId, tbsCert);
    }


    public Certificate build(PrivateKey privateKey, CertHashTypeEnum hashType, byte[] issueHash, TbsCert tbsCert) throws Exception {
        Certificate certificate = new Certificate();
        //ByteArrayUtils.printHexBinary(0, "Certificate SEQUENCE", null);

        //ByteArrayUtils.printHexBinary(1, "Version Uint8 INTEGER", null);
        //ByteArrayUtils.printHexBinary(1, "Version", ByteBuffer.allocate(2).putShort((short) 2).array());

        IssuerId issuerId = new IssuerId();
        if (issueHash != null) {
            //ByteArrayUtils.printHexBinary(1, "IssuerId CHOICE [index = 1]", null);

            //ByteArrayUtils.printHexBinary(2, "CertificateDigest SEQUENCE", null);
            CertificateDigest certificateDigest = new CertificateDigest();

            //ByteArrayUtils.printHexBinary(3, "HashAlgorithm ENUMERATED", null);
            HashAlgorithm hashAlgorithm;
            if (hashType.id == CertHashTypeEnum.SGD_SM3.id) {
                hashAlgorithm = new HashAlgorithm(HashAlgorithm.SGD_SM3);
            } else {
                hashAlgorithm = new HashAlgorithm(HashAlgorithm.SHA_256);
            }

            certificateDigest.setHashAlgorithm(hashAlgorithm);
            //ByteArrayUtils.printHexBinary(3, "HashAlgorithm", hashAlgorithm.getEncode());

            //ByteArrayUtils.printHexBinary(3, "HashedId8 OCTER STRING", null);
            byte[] hashId;
            try {
                hashId = ByteArrayUtils.subByteHashId(issueHash, 8);
            } catch (Exception e) {
                e.printStackTrace();
                return null;
            }
            HashedId8 hashedId8 = new HashedId8(hashId);
            certificateDigest.setHashedId8(hashedId8);
            //ByteArrayUtils.printHexBinary(3, "HashedId8", hashedId8.getEncode());

            //ByteArrayUtils.printHexBinary(2, "CertificateDigest", certificateDigest.getEncode());
            issuerId.setCertificateDigest(certificateDigest);
        } else {
            //ByteArrayUtils.printHexBinary(1, "IssuerId CHOICE [index = 0]", null);
            Null aNull = new Null();
            issuerId.setSelf(aNull);
        }
        certificate.setIssuerId(issuerId);
        //ByteArrayUtils.printHexBinary(1, "IssuerId", issuerId.getEncode());

        //ByteArrayUtils.printHexBinary(1, "TbsCert", null);
        certificate.setTbsCert(tbsCert);
        //ByteArrayUtils.printHexBinary(1, "TbsCert", tbsCert.getEncode());

        byte[] tbsEncode = tbsCert.getEncode();
        byte[] data;
        if (issueHash != null) {
            if (hashType == CertHashTypeEnum.SGD_SM3) {
                data = ByteArrayUtils.buildUpByte(calculateService.sm3Hash(tbsEncode), issueHash);
            } else {
                data = ByteArrayUtils.buildUpByte(calculateService.sha256Hash(tbsEncode), issueHash);
            }

        } else {
            if (hashType == CertHashTypeEnum.SGD_SM3) {
                data = ByteArrayUtils.buildUpByte(calculateService.sm3Hash(tbsEncode),
                        calculateService.sm3Hash("".getBytes()));
            } else {
                data = ByteArrayUtils.buildUpByte(calculateService.sha256Hash(tbsEncode),
                        calculateService.sha256Hash("".getBytes()));
            }
        }
        Signature signature = SignatureBuilder.build(privateKey, data, eccCurveTypeEnum);
        certificate.setSignature(signature);
        //ByteArrayUtils.printHexBinary(1, "Signature", signature.getEncode());
        //ByteArrayUtils.printHexBinary(0, "Certificate", certificate.getEncode());
        return certificate;
    }

}
