package com.xdja.pki.oer.gbt.asn1.utils;

import com.xdja.pki.gmssl.crypto.utils.GMSSLSHA256DigestUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM3DigestUtils;
import com.xdja.pki.oer.base.Enumerated;
import com.xdja.pki.oer.core.ButterFlyKeyUtils;
import com.xdja.pki.oer.core.ByteArrayUtils;
import com.xdja.pki.oer.gbt.asn1.Certificate;
import com.xdja.pki.oer.gbt.asn1.HashAlgorithm;
import org.bouncycastle.jce.interfaces.ECPrivateKey;
import org.bouncycastle.jce.interfaces.ECPublicKey;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.BigIntegers;

import java.math.BigInteger;

/**
 * @ClassName ImplicitCertPublicUtils
 * @Description TODO
 * @Date 2019/11/4 10:12
 * @Author FengZhen
 */
public class ImplicitCertPublicUtils {

    /**
     * 获取返回重构值
     *
     * @param hashAlgorithm hash算法
     * @param caCert        签发该EE证书的CA证书
     * @param eeCert        签发出的EE证书
     * @param s             CA证书对应的私钥
     * @return BigInteger 私钥重构值
     */
    public static BigInteger buildReconstructionValue(Enumerated.Value hashAlgorithm, Certificate caCert, Certificate eeCert, ECPrivateKey s) throws Exception {
        BigInteger N = s.getParameters().getN();
        BigInteger hashBigInter = getHashBigInteger(hashAlgorithm, caCert, eeCert, N);
        //使用CA私钥对mod值进行加计算并对N取余
        return s.getD().add(hashBigInter).mod(N);
    }

    /**
     * 获取隐式证书的公钥
     *
     * @param hashAlgorithm hash算法
     * @param caCert        签发该EE证书的CA证书
     * @param eeCert        签发出的EE证书
     * @param S             CA系统随机生成的密钥对对应的公钥值
     * @param B             传输过来的公钥值
     * @return ECPublicKey 显式证书的公钥
     */
    public static ECPublicKey buildReconstructionPublicKey(Enumerated.Value hashAlgorithm, Certificate caCert, Certificate eeCert, ECPublicKey S, ECPublicKey B) throws Exception {
        BigInteger N = B.getParameters().getN();
        ECPoint G = B.getParameters().getG();
        BigInteger hashBigInteger = getHashBigInteger(hashAlgorithm, caCert, eeCert, N);
        ECPoint hashEcPoint = G.multiply(hashBigInteger);
        ECPoint reconstructionPoint = hashEcPoint.add(S.getQ());
        ECPoint lastPoint = ButterFlyKeyUtils.addECPoint(reconstructionPoint, B.getQ());
        return ButterFlyKeyUtils.generateECPublicWithECPoint(S.getParameters().getCurve(), lastPoint);
    }

    /**
     * 获取隐式证书的私钥
     *
     * @param reconstructionValue CA系统返回的私钥重构值
     * @param bPrivateKey         传输前的私钥
     * @return ECPrivateKey 隐式证书的私钥
     */
    public static ECPrivateKey buildEcPrivateKey(BigInteger reconstructionValue, ECPrivateKey bPrivateKey) throws Exception {
        return ButterFlyKeyUtils.addPrivateKey(reconstructionValue, bPrivateKey);
    }

    private static BigInteger getHashBigInteger(Enumerated.Value hashAlgorithm, Certificate caCert, Certificate eeCert, BigInteger N) throws Exception {
        byte[] hash;
        if (hashAlgorithm == HashAlgorithm.SGD_SM3) {
            byte[] issueCertHash = GMSSLSM3DigestUtils.digestByYunhsm(caCert.getEncode());
            byte[] eeCertHash = GMSSLSM3DigestUtils.digestByYunhsm(eeCert.getEncode());
            hash = ByteArrayUtils.buildUpByte(issueCertHash, eeCertHash);
        } else if (hashAlgorithm == HashAlgorithm.SHA_256) {
            byte[] issueCertHash = GMSSLSHA256DigestUtils.digestByYunHsm(caCert.getEncode());
            byte[] eeCertHash = GMSSLSHA256DigestUtils.digestByYunHsm(eeCert.getEncode());
            hash = ByteArrayUtils.buildUpByte(issueCertHash, eeCertHash);
        } else {
            throw new Exception("unknown hashAlgorithm value : " + hashAlgorithm.toString());
        }
        return BigIntegers.fromUnsignedByteArray(ButterFlyKeyUtils.mod(hash, N));
    }

}
