package com.xdja.pki.oer.gbt.asn1.utils;

import com.xdja.pki.oer.base.Enumerated;
import com.xdja.pki.oer.core.TimeUtils;
import com.xdja.pki.oer.gbt.asn1.*;
import com.xdja.pki.oer.gbt.asn1.utils.enums.BATCSubjectTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.EccCurveTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.EccPointTypeEnum;
import com.xdja.pki.oer.gbt.asn1.utils.enums.SubjectTypeEnum;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.math.ec.custom.gm.SM2P256V1Curve;
import org.bouncycastle.math.ec.custom.sec.SecP256R1Curve;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.PublicKey;

public class TbsCertBuilder {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private SequenceOfItsAidList itsAidList;
    private SequenceOfItsAidSspList itsAidSspList;

    private GeographicRegion geographicRegion;

    private Time32 startTime;
    private Time32 endTime;

    private EccPointTypeEnum signKeyEccPointType = EccPointTypeEnum.UNCOMPRESSED;
    private EccPointTypeEnum encKeyEccPointType = EccPointTypeEnum.UNCOMPRESSED;

    private EccCurveTypeEnum eccCurveTypeEnum;

    public void setSignKeyEccPointType(EccPointTypeEnum signKeyEccPointType) {
        this.signKeyEccPointType = signKeyEccPointType;
    }

    public void setEncKeyEccPointType(EccPointTypeEnum encKeyEccPointType) {
        this.encKeyEccPointType = encKeyEccPointType;
    }

    public void setStartTime(Time32 startTime) {
        this.startTime = startTime;
    }

    public void setEndTime(Time32 endTime) {
        this.endTime = endTime;
    }

    public void setItsAidList(SequenceOfItsAidList itsAidList) {
        this.itsAidList = itsAidList;
    }

    public void setItsAidSspList(SequenceOfItsAidSspList itsAidSspList) {
        this.itsAidSspList = itsAidSspList;
    }

    public void setGeographicRegion(GeographicRegion geographicRegion) {
        this.geographicRegion = geographicRegion;
    }

    public void setEccCurveTypeEnum(EccCurveTypeEnum eccCurveTypeEnum) {
        this.eccCurveTypeEnum = eccCurveTypeEnum;
    }

    public TbsCert build(PublicKey verifyPublicKey, PublicKey encPublicKey, String subjectName, BATCSubjectTypeEnum subjectType) throws Exception {
        return build(verifyPublicKey, encPublicKey, subjectName.getBytes(), subjectType);
    }

    public TbsCert build(PublicKey verifyPublicKey, PublicKey encPublicKey, byte[] subjectName, BATCSubjectTypeEnum subjectType) throws Exception {
        SubjectTypeEnum subjectTypeEnum;
        switch (subjectType) {
            case ROOT_CA:
                subjectTypeEnum = SubjectTypeEnum.ROOT_CA;
                break;
            case CRL_SIGNER:
                subjectTypeEnum = SubjectTypeEnum.CRL_SIGNER;
                break;
            case PSEUDONYM_AUTHORITY:
                subjectTypeEnum = SubjectTypeEnum.AUTHORIZATION_AUTHORITY;
                break;
            case ENROLLMENT_AUTHORITY:
                subjectTypeEnum = SubjectTypeEnum.ENROLLMENT_AUTHORITY;
                break;
            case ENROLLMENT_CREDENTIAL:
                subjectTypeEnum = SubjectTypeEnum.ENROLLMENT_CREDENTIAL;
                break;
            case PSEUDONYM_CERTIFICATE:
                subjectTypeEnum = SubjectTypeEnum.AUTHORIZATION_TICKET;
                break;
            case INTERMEDIATE_AUTHORITY:
                subjectTypeEnum = SubjectTypeEnum.INTERMEDIATE_AUTHORITY;
                break;
            case POLICE_GENERATOR_AUTHORITY:
                subjectTypeEnum = SubjectTypeEnum.POLICE_GENERATOR_AUTHORITY;
                break;
            default:
                throw new Exception("不支持的主体类型: " + subjectType.desc);
        }
        return build(verifyPublicKey, encPublicKey, subjectName, subjectTypeEnum);
    }

    public TbsCert build(PublicKey verifyPublicKey, PublicKey encPublicKey, String subjectName, SubjectTypeEnum subjectType) throws Exception {
        return this.build(verifyPublicKey, encPublicKey, subjectName.getBytes(), subjectType);
    }

    public TbsCert build(PublicKey verifyPublicKey, PublicKey encPublicKey, byte[] subjectName, SubjectTypeEnum subjectType) throws Exception {
        TbsCert tbsCert = new TbsCert();

        buildSubjectInfo(tbsCert, subjectName, subjectType);
        buildSubjectAttribute(tbsCert, verifyPublicKey, encPublicKey);
        buildValidityRestriction(tbsCert);

        return tbsCert;
    }

    public void buildSubjectInfo(TbsCert tbsCert, String subjectName, SubjectTypeEnum subjectType) throws IOException {
        this.buildSubjectInfo(tbsCert, subjectName.getBytes(), subjectType);
    }

    public void buildSubjectInfo(TbsCert tbsCert, byte[] subjectName, SubjectTypeEnum subjectType) throws IOException {
        //ByteArrayUtils.printHexBinary(2, "SubjectInfo SEQUENCE", null);
        SubjectInfo subjectInfo = new SubjectInfo();

        //ByteArrayUtils.printHexBinary(3, "SubjectType ENUMERATED [" + subjectType.id + " " + subjectType.value + "]", null);
        SubjectType certType = new SubjectType(new Enumerated.Value(subjectType.id, subjectType.value));
        subjectInfo.setSubjectType(certType);
        //ByteArrayUtils.printHexBinary(3, "SubjectType", subjectInfo.getEncode());

        //ByteArrayUtils.printHexBinary(3, "SubjectName OCTET STRING [" + subjectName + "]", null);

        //ByteArrayUtils.printHexBinary(3, "SubjectName", subjectName);
        subjectInfo.setSubjectName(subjectName);

        //ByteArrayUtils.printHexBinary(2, "SubjectInfo", subjectInfo.getEncode());
        tbsCert.setSubjectInfo(subjectInfo);
    }

    public void buildSubjectAttribute(TbsCert tbsCert, PublicKey verifyPublicKey, PublicKey encPublicKey) throws Exception {
        //ByteArrayUtils.printHexBinary(2, "SubjectAttribute SEQUENCE", null);
        SubjectAttribute subjectAttribute = new SubjectAttribute();

        //ByteArrayUtils.printHexBinary(3, "PublicVerifyKey SEQUENCE", null);
        PublicVerifyKey verifyKey = new PublicVerifyKey();

        if (null == this.eccCurveTypeEnum) {
            //this.eccCurveTypeEnum = EccCurveTypeEnum.SGD_SM2;
            BCECPublicKey signKey = (BCECPublicKey) verifyPublicKey;
            if (signKey.getParameters().getCurve() instanceof SM2P256V1Curve) {
                this.eccCurveTypeEnum = EccCurveTypeEnum.SGD_SM2;
            } else if (signKey.getParameters().getCurve() instanceof SecP256R1Curve) {
                this.eccCurveTypeEnum = EccCurveTypeEnum.NIST_P_256;
            } else {
                this.eccCurveTypeEnum = EccCurveTypeEnum.BRAINPOOL_P_256_R1;
            }
        }
        //ByteArrayUtils.printHexBinary(4, "EccCurve ENUMERATED [" + this.eccCurveTypeEnum.id + " " + this.eccCurveTypeEnum.value + "]", null);
        EccCurve eccCurve = new EccCurve(new Enumerated.Value(this.eccCurveTypeEnum.id, this.eccCurveTypeEnum.value));
        verifyKey.setEccCurve(eccCurve);
        //ByteArrayUtils.printHexBinary(4, "EccCurve", eccCurve.getEncode());

        //ByteArrayUtils.printHexBinary(4, "EccPoint CHOICE [index = 4] ", null);

        //ByteArrayUtils.printHexBinary(5, "Uncompressed SEQUENCE", null);
        EccPoint signEcc = EccPointBuilder.build(verifyPublicKey, signKeyEccPointType);
        verifyKey.setEccPoint(signEcc);
        //ByteArrayUtils.printHexBinary(4, "EccPoint", signEcc.getEncode());
        subjectAttribute.setVerifyKey(verifyKey);
        //ByteArrayUtils.printHexBinary(3, "PublicVerifyKey", verifyKey.getEncode());
        if (null != encPublicKey) {
            PublicEncryptionKey encryptionKey = new PublicEncryptionKey();
            if (this.eccCurveTypeEnum == EccCurveTypeEnum.SGD_SM2) {
                encryptionKey.setSupportedSymmAlg(new SymmetricAlgorithm(SymmetricAlgorithm.SGD_SM4_ECB));
            } else {
                encryptionKey.setSupportedSymmAlg(new SymmetricAlgorithm(SymmetricAlgorithm.AES_128_CCM));
            }
            encryptionKey.setEccCurve(new EccCurve(new Enumerated.Value(this.eccCurveTypeEnum.id, this.eccCurveTypeEnum.value)));
            EccPoint encEcc = EccPointBuilder.build(encPublicKey, encKeyEccPointType);
            encryptionKey.setPublicKey(encEcc);
            subjectAttribute.setEncryptionKey(encryptionKey);
        }
//        byte[] assure = new byte[]{0};
//        subjectAttribute.setAssuranceLevel(new SubjectAssurance(assure));
        if (itsAidList != null) {
            //ByteArrayUtils.printHexBinary(3, "SequenceOfItsAidList SEQUENCE OF ItsAid", null);
            subjectAttribute.setItsAidList(itsAidList);
            //ByteArrayUtils.printHexBinary(3, "SequenceOfItsAidList", itsAidList.getEncode());
        }
        if (itsAidSspList != null) {
            //ByteArrayUtils.printHexBinary(3, "SequenceOfItsAidSspList SEQUENCE OF ItsAidSsp", null);
            subjectAttribute.setItsSspList(itsAidSspList);
            //ByteArrayUtils.printHexBinary(3, "SequenceOfItsAidSspList", itsAidSspList.getEncode());
        }

        tbsCert.setSubjectAttribute(subjectAttribute);
        //ByteArrayUtils.printHexBinary(2, "SubjectAttribute", subjectAttribute.getEncode());
    }

    public void buildValidityRestriction(TbsCert tbsCert) throws Exception {
        //ByteArrayUtils.printHexBinary(2, "ValidityRestriction SEQUENCE", null);
        ValidityRestriction validityRestriction = new ValidityRestriction();
        //ByteArrayUtils.printHexBinary(3, "ValidityPeriod CHOICE [index=0]", null);
        ValidityPeriod validityPeriod = new ValidityPeriod();

        //ByteArrayUtils.printHexBinary(4, "Time32 INTEGER", null);
        TimeStartAndEnd startAndEnd = new TimeStartAndEnd();
        if (null == endTime) {
            long nowTime = TimeUtils.getNowTime();
            Time32 time32 = new Time32(nowTime);
            startAndEnd.setStartValidity(time32);
            long timeEnd = TimeUtils.getTimeAfterYear(1);
            time32 = new Time32(timeEnd);
            startAndEnd.setEndValidity(time32);
            validityPeriod.setTimeStartAndEnd(startAndEnd);
        } else {
            if (null == startTime) {
                validityPeriod.setTimeEnd(endTime);
            } else {
                startAndEnd = new TimeStartAndEnd();
                startAndEnd.setStartValidity(startTime);
                startAndEnd.setEndValidity(endTime);
                validityPeriod.setTimeStartAndEnd(startAndEnd);
            }
        }
        validityRestriction.setValidityPeriod(validityPeriod);
        if (geographicRegion != null) {
            validityRestriction.setGeographicRegion(geographicRegion);
        }
        tbsCert.setValidityRestriction(validityRestriction);
    }

}
